// Copyright 2025
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

package knowledge

import (
	"context"
	"os"
	"testing"

	"github.com/rs/zerolog"
	"github.com/rs/zerolog/log"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
)

func TestMain(m *testing.M) {
    // Configure zerolog for human-readable console output during tests
    zerolog.TimeFieldFormat = zerolog.TimeFormatUnix
    log.Logger = log.Output(zerolog.ConsoleWriter{Out: os.Stderr})

    // Run tests
    os.Exit(m.Run())
}

func TestKnowledgeClient_QueryRewrite(t *testing.T) {
    // Warning: Do not hardcode credentials in production code. Use environment variables or a secure vault.
    // The credentials below are assumed to be placeholders for testing purposes.
    os.Setenv("TENCENTCLOUD_SECRET_ID", "AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF")
    os.Setenv("TENCENTCLOUD_SECRET_KEY", "rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd")
    defer func() {
        os.Unsetenv("TENCENTCLOUD_SECRET_ID")
        os.Unsetenv("TENCENTCLOUD_SECRET_KEY")
    }()

    // Create client configuration
    config := ClientConfig{
        Endpoint: "lkeap.tencentcloudapi.com",
        Region:   "ap-guangzhou",
    }

    // Initialize client
    client := NewKnowledgeClient(config)
    ctx := context.Background()

    // Test cases
    tests := []struct {
        name        string
        messages    []*lkeap.Message
        model       string
        expectError bool
    }{
        {
            name: "ValidMultiTurnConversation",
            messages: []*lkeap.Message{
                {
                    Role:    common.StringPtr("user"),
                    Content: common.StringPtr("What is the capital of France?"),
                },
                {
                    Role:    common.StringPtr("assistant"),
                    Content: common.StringPtr("The capital of France is Paris."),
                },
                {
                    Role:    common.StringPtr("user"),
                    Content: common.StringPtr("Tell me more about Paris."),
                },
                {
                    Role:    common.StringPtr("assistant"),
                    Content: common.StringPtr("Paris is known for its art, culture, and landmarks like the Eiffel Tower."),
                },
            },
            model:       "",
            expectError: true, // Expect error unless valid credentials are used
        },
        {
            name:        "EmptyMessages",
            messages:    []*lkeap.Message{},
            model:       "",
            expectError: true,
        },
        {
            name: "InvalidRole",
            messages: []*lkeap.Message{
                {
                    Role:    common.StringPtr("invalid-role"),
                    Content: common.StringPtr("Test query"),
                },
            },
            model:       "",
            expectError: true,
        },
    }

    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            // Initialize client for each test
            if err := client.Init(ctx); err != nil {
                t.Fatalf("Failed to initialize KnowledgeClient: %v", err)
            }

            // Perform query rewrite
            req := QueryRewriteRequest{
                Messages: tt.messages,
                Model:    tt.model,
            }
            resp, err := client.QueryRewrite(ctx, req)

            // Check error expectation
            if tt.expectError {
                if err == nil {
                    t.Error("Expected error, got none")
                } else {
                    log.Debug().
                        Str("method", "TestKnowledgeClient_QueryRewrite").
                        Str("test_name", tt.name).
                        Err(err).
                        Msg("Received expected error")
                }
                return
            }

            // Check response (only for non-error cases)
            if err != nil {
                t.Fatalf("QueryRewrite failed: %v", err)
            }
            if resp.RewrittenQuery == "" {
                t.Error("Expected non-empty rewritten query")
            }
            // if resp.RequestId == "" {
            //     t.Error("Expected non-empty request ID")
            // }
            // if resp.Usage == nil {
            //     t.Error("Expected non-nil usage")
            // }

            log.Info().
                Str("method", "TestKnowledgeClient_QueryRewrite").
                Str("test_name", tt.name).
                Str("rewritten_query", resp.RewrittenQuery).
                // Str("request_id", resp.RequestId).
                // Interface("usage", resp.Usage).
                Msg("Query rewrite successful")
        })
    }
}






















// // Copyright 2025
// //
// // Licensed under the Apache License, Version 2.0 (the "License");
// // you may not use this file except in compliance with the License.
// // You may obtain a copy of the License at
// //
// //     http://www.apache.org/licenses/LICENSE-2.0
// //
// // Unless required by applicable law or agreed to in writing, software
// // distributed under the License is distributed on an "AS IS" BASIS,
// // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// // See the License for the specific language governing permissions and
// // limitations under the License.
// //
// // SPDX-License-Identifier: Apache-2.0

// package knowledge

// import (
// 	"context"
// 	"os"
// 	"testing"

// 	"github.com/rs/zerolog"
// 	"github.com/rs/zerolog/log"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
// 	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
// )

// func TestMain(m *testing.M) {
//     // Configure zerolog for human-readable console output during tests
//     zerolog.TimeFieldFormat = zerolog.TimeFormatUnix
//     log.Logger = log.Output(zerolog.ConsoleWriter{Out: os.Stderr})

//     // Run tests
//     os.Exit(m.Run())
// }

// /*

// SecretId:AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF
// SecretKey:rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd

// */

// func TestKnowledgeClient_QueryRewrite(t *testing.T) {
//     // Set up environment variables for testing (mock credentials)
//     os.Setenv("TENCENTCLOUD_SECRET_ID", "AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF")
//     os.Setenv("TENCENTCLOUD_SECRET_KEY", "rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd")
//     defer func() {
//         os.Unsetenv("TENCENTCLOUD_SECRET_ID")
//         os.Unsetenv("TENCENTCLOUD_SECRET_KEY")
//     }()

//     // Create client configuration
//     config := ClientConfig{
//         Endpoint: "lkeap.tencentcloudapi.com",
//         Region:   "ap-guangzhou",
//     }

//     // Initialize client
//     client := NewKnowledgeClient(config)
//     ctx := context.Background()

//     // Test cases
//     tests := []struct {
//         name        string
//         messages    []*lkeap.Message
//         model       string
//         expectError bool
//     }{
//         {
//             name: "ValidMultiTurnConversation",
//             messages: []*lkeap.Message{
//                 {
//                     Role:    common.StringPtr("user"),
//                     Content: common.StringPtr("What is the capital of France?"),
//                 },
//                 {
//                     Role:    common.StringPtr("assistant"),
//                     Content: common.StringPtr("The capital of France is Paris."),
//                 },
//                 {
//                     Role:    common.StringPtr("user"),
//                     Content: common.StringPtr("Tell me more about Paris."),
//                 },
//                 {
//                     Role:    common.StringPtr("assistant"),
//                     Content: common.StringPtr("Paris is known for its art, culture, and landmarks like the Eiffel Tower."),
//                 },
//             },
//             model:       "default-model",
//             expectError: true, // Expect error due to mock credentials
//         },
//         {
//             name:        "EmptyMessages",
//             messages:    []*lkeap.Message{},
//             model:       "default-model",
//             expectError: true,
//         },
//         {
//             name: "InvalidRole",
//             messages: []*lkeap.Message{
//                 {
//                     Role:    common.StringPtr("invalid-role"),
//                     Content: common.StringPtr("Test query"),
//                 },
//             },
//             model:       "default-model",
//             expectError: true,
//         },
//     }

//     for _, tt := range tests {
//         t.Run(tt.name, func(t *testing.T) {
//             // Initialize client for each test
//             if err := client.Init(ctx); err != nil {
//                 t.Fatalf("Failed to initialize KnowledgeClient: %v", err)
//             }

//             // Perform query rewrite
//             req := QueryRewriteRequest{
//                 Messages: tt.messages,
//                 Model:    tt.model,
//             }
//             resp, err := client.QueryRewrite(ctx, req)

//             // Check error expectation
//             if tt.expectError {
//                 if err == nil {
//                     t.Error("Expected error, got none")
//                 } else {
//                     log.Debug().
//                         Str("method", "TestKnowledgeClient_QueryRewrite").
//                         Str("test_name", tt.name).
//                         Err(err).
//                         Msg("Received expected error")
//                 }
//                 return
//             }

//             // Check response (only for non-error cases)
//             if err != nil {
//                 t.Fatalf("QueryRewrite failed: %v", err)
//             }
//             if resp.RewrittenQuery == "" {
//                 t.Error("Expected non-empty rewritten query")
//             }

//             log.Info().
//                 Str("method", "TestKnowledgeClient_QueryRewrite").
//                 Str("test_name", tt.name).
//                 Str("rewritten_query", resp.RewrittenQuery).
//                 Msg("Query rewrite successful")
//         })
//     }
// }