package question

import (
	"context"
	"database/sql"
	"testing"
	"time"

	_ "github.com/lib/pq"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

// TestQAStore 测试 QAStore 的读写功能
func TestQAStore(t *testing.T) {
	// 初始化测试数据库连接
	store, err := qa.InitQAStore()
	require.NoError(t, err, "failed to initialize QAStore")
	defer store.(*qaStore).db.Close()

	ctx := context.Background()

	// 清理测试数据（可选，确保测试环境干净）
	// cleanup := func() {
	//     _, err := store.(*qaStore).db.ExecContext(ctx, "TRUNCATE TABLE qa RESTART IDENTITY")
	//     require.NoError(t, err, "failed to truncate qa table")
	// }
	// cleanup()

	t.Run("WriteQA_Insert", func(t *testing.T) {
		// 测试插入新记录
		qa := QA{
			Question: stringPtr("What is Go?"),
			Answer:   stringPtr("A programming language"),
			Summary:  stringPtr("Go introduction"),
			From:     stringPtr("Alice"),
			FromID:   stringPtr("user123"),
			To:       stringPtr("Bob"),
			ToID:     stringPtr("user456"),
		}

		id, err := store.WriteQA(ctx, qa)
		require.NoError(t, err, "failed to insert QA")
		assert.NotZero(t, id, "inserted ID should not be zero")

		// 验证数据库中的记录
		var stored QA
		row := store.(*qaStore).db.QueryRowContext(ctx, `
            SELECT id, created_at, question, answer, summary, "from", "to", from_id, to_id
            FROM qa
            WHERE id = $1`, id)
		var question, answer, summary, from, to, fromID, toID sql.NullString
		err = row.Scan(&stored.ID, &stored.CreatedAt, &question, &answer, &summary, &from, &to, &fromID, &toID)
		require.NoError(t, err, "failed to query inserted QA")

		// 设置结构体字段
		if question.Valid {
			stored.Question = &question.String
		}
		if answer.Valid {
			stored.Answer = &answer.String
		}
		if summary.Valid {
			stored.Summary = &summary.String
		}
		if from.Valid {
			stored.From = &from.String
		}
		if to.Valid {
			stored.To = &to.String
		}
		if fromID.Valid {
			stored.FromID = &fromID.String
		}
		if toID.Valid {
			stored.ToID = &toID.String
		}

		// 验证插入的数据
		assert.Equal(t, qa.Question, stored.Question)
		assert.Equal(t, qa.Answer, stored.Answer)
		assert.Equal(t, qa.Summary, stored.Summary)
		assert.Equal(t, qa.From, stored.From)
		assert.Equal(t, qa.To, stored.To)
		assert.Equal(t, qa.FromID, stored.FromID)
		assert.Equal(t, qa.ToID, stored.ToID)
	})

	t.Run("WriteQA_Update", func(t *testing.T) {
		// 先插入一条记录
		qa := QA{
			Question: stringPtr("What is Python?"),
			Answer:   stringPtr("Another language"),
			FromID:   stringPtr("user789"),
		}
		id, err := store.WriteQA(ctx, qa)
		require.NoError(t, err, "failed to insert QA for update test")

		// 更新记录
		updatedQA := QA{
			ID:       id,
			Question: stringPtr("Updated: What is Python?"),
			Answer:   stringPtr("A versatile language"),
			Summary:  stringPtr("Python summary"),
			FromID:   stringPtr("user789"),
		}
		updatedID, err := store.WriteQA(ctx, updatedQA)
		require.NoError(t, err, "failed to update QA")
		assert.Equal(t, id, updatedID, "updated ID should match inserted ID")

		// 验证更新后的记录
		var stored QA
		row := store.(*qaStore).db.QueryRowContext(ctx, `
            SELECT id, created_at, question, answer, summary, "from", "to", from_id, to_id
            FROM qa
            WHERE id = $1`, id)
		var question, answer, summary, from, to, fromID, toID sql.NullString
		err = row.Scan(&stored.ID, &stored.CreatedAt, &question, &answer, &summary, &from, &to, &fromID, &toID)
		require.NoError(t, err, "failed to query updated QA")

		if question.Valid {
			stored.Question = &question.String
		}
		if answer.Valid {
			stored.Answer = &answer.String
		}
		if summary.Valid {
			stored.Summary = &summary.String
		}
		if fromID.Valid {
			stored.FromID = &fromID.String
		}

		assert.Equal(t, updatedQA.Question, stored.Question)
		assert.Equal(t, updatedQA.Answer, stored.Answer)
		assert.Equal(t, updatedQA.Summary, stored.Summary)
		assert.Equal(t, updatedQA.FromID, stored.FromID)
	})

	t.Run("GetLatestQA", func(t *testing.T) {
		// 插入多条记录以测试视图
		qa1 := QA{
			Question:  stringPtr("First question"),
			Answer:    stringPtr("First answer"),
			FromID:    stringPtr("user123"),
			CreatedAt: time.Now().Add(-2 * time.Hour),
		}
		_, err := store.WriteQA(ctx, qa1)
		require.NoError(t, err, "failed to insert first QA")

		qa2 := QA{
			Question:  stringPtr("Second question"),
			Answer:    stringPtr("Second answer"),
			FromID:    stringPtr("user123"),
			CreatedAt: time.Now(),
		}
		_, err = store.WriteQA(ctx, qa2)
		require.NoError(t, err, "failed to insert second QA")

		// 查询最新记录
		fromID := stringPtr("user123")
		results, err := store.GetLatestQA(ctx, fromID)
		require.NoError(t, err, "failed to get latest QA")
		require.Len(t, results, 1, "should return exactly one record for from_id")

		// 验证返回的记录是最新的一条
		assert.Equal(t, qa2.Question, results[0].Question)
		assert.Equal(t, qa2.Answer, results[0].Answer)
		assert.Equal(t, qa2.FromID, results[0].FromID)
	})

	t.Run("GetLatestQA_NullFromID", func(t *testing.T) {
		// 插入一条 from_id 为 NULL 的记录
		qa := QA{
			Question: stringPtr("Null from_id question"),
			Answer:   stringPtr("Null from_id answer"),
			FromID:   nil,
		}
		_, err := store.WriteQA(ctx, qa)
		require.NoError(t, err, "failed to insert QA with null from_id")

		// 查询 from_id 为 NULL 的记录
		results, err := store.GetLatestQA(ctx, nil)
		require.NoError(t, err, "failed to get latest QA for null from_id")
		assert.NotEmpty(t, results, "should return records for null from_id")
		assert.Nil(t, results[0].FromID, "from_id should be nil")
		assert.Equal(t, qa.Question, results[0].Question)
	})

	// 清理测试数据
	// cleanup()
}
