package deepseek

import (
	"context"
	"fmt"
	"strings"
	"sync"

	"github.com/firebase/genkit/go/ai"
	"github.com/firebase/genkit/go/genkit"
	"github.com/rs/zerolog/log"

	deepseek "github.com/cohesion-org/deepseek-go"
)

const provider = "deepseek"
const Provider = "deepseek"

// 常量定义支持的模型
const (
	// DeepSeekChat     = "deepseek-chat"
	// DeepSeekCoder    = "deepseek-coder"
	// DeepSeekReasoner = "deepseek-reasoner"
	DeepSeekChat     = deepseek.DeepSeekChat
	DeepSeekCoder    = deepseek.DeepSeekCoder
	DeepSeekReasoner = deepseek.DeepSeekReasoner
)

var (
	mediaSupportedModels = []string{DeepSeekChat, DeepSeekCoder, DeepSeekReasoner}
	roleMapping          = map[ai.Role]string{
		ai.RoleUser:   deepseek.ChatMessageRoleUser,
		ai.RoleModel:  deepseek.ChatMessageRoleAssistant,
		ai.RoleSystem: deepseek.ChatMessageRoleSystem,
		ai.RoleTool:   deepseek.ChatMessageRoleTool,
	}
)

// modelInfoMap 定义支持的模型及其信息
var modelInfoMap = map[string]ai.ModelInfo{
	DeepSeekChat: {
		Label: "DeepSeek Chat",
		Supports: &ai.ModelSupports{
			Multiturn:  true,
			SystemRole: true,
			Media:      false,
			Tools:      true,
		},
		Versions: []string{},
	},
	DeepSeekCoder: {
		Label: "DeepSeek Coder",
		Supports: &ai.ModelSupports{
			Multiturn:  true,
			SystemRole: true,
			Media:      false,
			Tools:      true,
		},
		Versions: []string{},
	},
	DeepSeekReasoner: {
		Label: "DeepSeek Reasoner",
		Supports: &ai.ModelSupports{
			Multiturn:  true,
			SystemRole: true,
			Media:      false,
			Tools:      true,
		},
		Versions: []string{},
	},
}

func ListModels() map[string]ai.ModelInfo {
	return modelInfoMap
}

// DeepSeek holds configuration for the plugin.
type DeepSeek struct {
	APIKey string // DeepSeek API key

	mu      sync.Mutex // Mutex to control access.
	initted bool       // Whether the plugin has been initialized.
}

// Name returns the provider name.
func (d *DeepSeek) Name() string {
	return provider
}

// ModelDefinition represents a model with its name and type.
type ModelDefinition struct {
	Name string
	Type string
}

// Init initializes the DeepSeek plugin and registers all known models.
func (d *DeepSeek) Init(ctx context.Context, g *genkit.Genkit) error {
	log.Info().Str("method", "DeepSeek.Init").Msg("Initializing DeepSeek plugin")
	d.mu.Lock()
	defer d.mu.Unlock()
	if d.initted {
		log.Error().Str("method", "DeepSeek.Init").Msg("Plugin already initialized")
		return fmt.Errorf("deepseek.Init already called")
	}

	if d == nil || d.APIKey == "" {
		log.Error().Str("method", "DeepSeek.Init").Msg("APIKey is required")
		return fmt.Errorf("deepseek: need APIKey")
	}

	d.initted = true

	//d.mu.Unlock()
	// 自动注册所有支持的模型
	for _, modelName := range mediaSupportedModels {
		modelDef := ModelDefinition{
			Name: modelName,
			Type: "chat", // 假设所有模型都支持 chat 类型
		}
		d.DefineModel(g, modelDef, nil)
	}

	log.Info().Str("method", "DeepSeek.Init").Msg("Initialization successful")
	return nil
}

// DefineModel defines a DeepSeek model in Genkit.
func (d *DeepSeek) DefineModel(g *genkit.Genkit, model ModelDefinition, info *ai.ModelInfo) ai.Model {
	log.Info().
		Str("method", "DeepSeek.DefineModel").
		Str("model_name", model.Name).
		Msg("Defining DeepSeek model")
	// d.mu.Lock()
	// defer d.mu.Unlock()
	if !d.initted {
		log.Error().Str("method", "DeepSeek.DefineModel").Msg("DeepSeek not initialized")
		return nil
	}

	// 获取模型信息
	var mi ai.ModelInfo
	if info != nil {
		mi = *info
	} else {
		var ok bool
		mi, ok = modelInfoMap[model.Name]
		if !ok {
			log.Error().
				Str("method", "DeepSeek.DefineModel").
				Str("model_name", model.Name).
				Msg("Unknown model and no ModelInfo provided")
			return nil
		}
	}

	// 定义生成器
	gen := &generator{model: model, apiKey: d.APIKey}
	modelDef := genkit.DefineModel(g, provider, model.Name, &mi, gen.generate)
	log.Info().
		Str("method", "Deep lantern").
		Str("model_name", model.Name).
		Msg("Model defined successfully")
	return modelDef
}

// generator handles model generation.
type generator struct {
	model  ModelDefinition
	apiKey string
}

// generate implements the Genkit model generation interface.
func (g *generator) generate(ctx context.Context, input *ai.ModelRequest, cb func(context.Context, *ai.ModelResponseChunk) error) (*ai.ModelResponse, error) {
	log.Info().
		Str("method", "generator.generate").
		Str("model_name", g.model.Name).
		Int("messages", len(input.Messages)).
		Msg("Starting model generation")

	if len(input.Messages) == 0 {
		log.Error().Str("method", "generator.generate").Msg("Prompt or messages required")
		return nil, fmt.Errorf("prompt or messages required")
	}

	// Initialize DeepSeek client.
	client := deepseek.NewClient(g.apiKey)
	log.Debug().Str("method", "generator.generate").Msg("DeepSeek client initialized")

	// Create a chat completion request
	request := &deepseek.ChatCompletionRequest{
		Model: g.model.Name,
	}

	for _, msg := range input.Messages {
		role, ok := roleMapping[msg.Role]
		if !ok {
			log.Error().
				Str("method", "generator.generate").
				Str("role", string(msg.Role)).
				Msg("Unsupported role")
			return nil, fmt.Errorf("unsupported role: %s", msg.Role)
		}
		content := concatMessageParts(msg.Content)
		request.Messages = append(request.Messages, deepseek.ChatCompletionMessage{
			Role:    role,
			Content: content,
		})
		log.Debug().
			Str("method", "generator.generate").
			Str("role", role).
			Str("content", content).
			Msg("Added message to request")
	}

	// Send the request and handle the response
	response, err := client.CreateChatCompletion(ctx, request)
	if err != nil {
		log.Error().
			Err(err).
			Str("method", "generator.generate").
			Msg("Failed to create chat completion")
		return nil, fmt.Errorf("create chat completion: %w", err)
	}

	log.Debug().
		Str("method", "generator.generate").
		Int("choices", len(response.Choices)).
		Msg("Received chat completion response")

	// Create a final response with the merged chunks
	finalResponse := &ai.ModelResponse{
		Request:      input,
		FinishReason: ai.FinishReason("stop"),
		Message: &ai.Message{
			Role: ai.RoleModel,
		},
	}

	for _, chunk := range response.Choices {
		log.Debug().
			Str("method", "generator.generate").
			Int("index", chunk.Index).
			Str("content", chunk.Message.Content).
			Msg("Processing response chunk")
		p := ai.Part{
			Text: chunk.Message.Content,
			Kind: ai.PartKind(chunk.Index),
		}
		finalResponse.Message.Content = append(finalResponse.Message.Content, &p)
	}

	log.Info().
		Str("method", "generator.generate").
		Str("model_name", g.model.Name).
		Int("content_parts", len(finalResponse.Message.Content)).
		Msg("Model generation completed successfully")
	return finalResponse, nil
}

// concatMessageParts concatenates message parts into a single string.
func concatMessageParts(parts []*ai.Part) string {
	log.Debug().
		Str("method", "concatMessageParts").
		Int("parts", len(parts)).
		Msg("Concatenating message parts")
	var sb strings.Builder
	for _, part := range parts {
		if part.IsText() {
			sb.WriteString(part.Text)
		}
		// Ignore non-text parts (e.g., media, tools) as DeepSeek API doesn't support them.
	}
	result := sb.String()
	log.Debug().
		Str("method", "concatMessageParts").
		Str("result", result).
		Msg("Concatenation complete")
	return result
}

// package deepseek

// import (
// 	"context"
// 	"fmt"
// 	"strings"
// 	"sync"

// 	"github.com/firebase/genkit/go/ai"
// 	"github.com/firebase/genkit/go/genkit"
// 	"github.com/rs/zerolog/log"

// 	deepseek "github.com/cohesion-org/deepseek-go"
// )

// const provider = "deepseek"

// var (
// 	mediaSupportedModels = []string{deepseek.DeepSeekChat, deepseek.DeepSeekCoder, deepseek.DeepSeekReasoner}
// 	roleMapping          = map[ai.Role]string{
// 		ai.RoleUser:   deepseek.ChatMessageRoleUser,
// 		ai.RoleModel:  deepseek.ChatMessageRoleAssistant,
// 		ai.RoleSystem: deepseek.ChatMessageRoleSystem,
// 		ai.RoleTool:   deepseek.ChatMessageRoleTool,
// 	}
// )

// // DeepSeek holds configuration for the plugin.
// type DeepSeek struct {
// 	APIKey string // DeepSeek API key

// 	mu      sync.Mutex // Mutex to control access.
// 	initted bool       // Whether the plugin has been initialized.
// }

// // Name returns the provider name.
// func (d DeepSeek) Name() string {
// 	return provider
// }

// // ModelDefinition represents a model with its name and type.
// type ModelDefinition struct {
// 	Name string
// 	Type string
// }

// // DefineModel defines a DeepSeek model in Genkit.
// func (d *DeepSeek) DefineModel(g *genkit.Genkit, model ModelDefinition, info *ai.ModelInfo) ai.Model {
// 	log.Info().
// 		Str("method", "DeepSeek.DefineModel").
// 		Str("model_name", model.Name).
// 		Msg("Defining DeepSeek model")
// 	d.mu.Lock()
// 	defer d.mu.Unlock()
// 	if !d.initted {
// 		log.Error().Str("method", "DeepSeek.DefineModel").Msg("DeepSeek not initialized")
// 		panic("deepseek.Init not called")
// 	}

// 	// Define model info, supporting multiturn and system role.
// 	mi := ai.ModelInfo{
// 		Label: model.Name,
// 		Supports: &ai.ModelSupports{
// 			Multiturn:  true,
// 			SystemRole: true,
// 			Media:      false, // DeepSeek API primarily supports text.
// 			Tools:      false, // Tools not yet supported in this implementation.
// 		},
// 		Versions: []string{},
// 	}
// 	if info != nil {
// 		mi = *info
// 	}

// 	meta := &ai.ModelInfo{
// 		Label:    model.Name,
// 		Supports: mi.Supports,
// 		Versions: []string{},
// 	}
// 	gen := &generator{model: model, apiKey: d.APIKey}
// 	modelDef := genkit.DefineModel(g, provider, model.Name, meta, gen.generate)
// 	log.Info().
// 		Str("method", "DeepSeek.DefineModel").
// 		Str("model_name", model.Name).
// 		Msg("Model defined successfully")
// 	return modelDef
// }

// // Init initializes the DeepSeek plugin.
// func (d *DeepSeek) Init(ctx context.Context, g *genkit.Genkit) error {
// 	log.Info().Str("method", "DeepSeek.Init").Msg("Initializing DeepSeek plugin")
// 	d.mu.Lock()
// 	defer d.mu.Unlock()
// 	if d.initted {
// 		log.Error().Str("method", "DeepSeek.Init").Msg("Plugin already initialized")
// 		return fmt.Errorf("deepseek.Init already called")
// 	}

// 	if d == nil || d.APIKey == "" {
// 		log.Error().Str("method", "DeepSeek.Init").Msg("APIKey is required")
// 		return fmt.Errorf("deepseek: need APIKey")
// 	}
// 	d.initted = true
// 	log.Info().Str("method", "DeepSeek.Init").Msg("Initialization successful")
// 	return nil
// }

// // generator handles model generation.
// type generator struct {
// 	model  ModelDefinition
// 	apiKey string
// }

// // generate implements the Genkit model generation interface.
// func (g *generator) generate(ctx context.Context, input *ai.ModelRequest, cb func(context.Context, *ai.ModelResponseChunk) error) (*ai.ModelResponse, error) {
// 	log.Info().
// 		Str("method", "generator.generate").
// 		Str("model_name", g.model.Name).
// 		Int("messages", len(input.Messages)).
// 		Msg("Starting model generation")

// 	if len(input.Messages) == 0 {
// 		log.Error().Str("method", "generator.generate").Msg("Prompt or messages required")
// 		return nil, fmt.Errorf("prompt or messages required")
// 	}

// 	// Initialize DeepSeek client.
// 	client := deepseek.NewClient(g.apiKey)
// 	log.Debug().Str("method", "generator.generate").Msg("DeepSeek client initialized")

// 	// Create a chat completion request
// 	request := &deepseek.ChatCompletionRequest{
// 		Model: g.model.Name,
// 	}

// 	for _, msg := range input.Messages {
// 		role, ok := roleMapping[msg.Role]
// 		if !ok {
// 			log.Error().
// 				Str("method", "generator.generate").
// 				Str("role", string(msg.Role)).
// 				Msg("Unsupported role")
// 			return nil, fmt.Errorf("unsupported role: %s", msg.Role)
// 		}
// 		content := concatMessageParts(msg.Content)
// 		request.Messages = append(request.Messages, deepseek.ChatCompletionMessage{
// 			Role:    role,
// 			Content: content,
// 		})
// 		log.Debug().
// 			Str("method", "generator.generate").
// 			Str("role", role).
// 			Str("content", content).
// 			Msg("Added message to request")
// 	}

// 	// Send the request and handle the response
// 	response, err := client.CreateChatCompletion(ctx, request)
// 	if err != nil {
// 		log.Error().
// 			Err(err).
// 			Str("method", "generator.generate").
// 			Msg("Failed to create chat completion")
// 		return nil, fmt.Errorf("create chat completion: %w", err)
// 	}

// 	log.Debug().
// 		Str("method", "generator.generate").
// 		Int("choices", len(response.Choices)).
// 		Msg("Received chat completion response")

// 	// Create a final response with the merged chunks
// 	finalResponse := &ai.ModelResponse{
// 		Request:      input,
// 		FinishReason: ai.FinishReason("stop"),
// 		Message: &ai.Message{
// 			Role: ai.RoleModel,
// 		},
// 	}

// 	for _, chunk := range response.Choices {
// 		log.Debug().
// 			Str("method", "generator.generate").
// 			Int("index", chunk.Index).
// 			Str("content", chunk.Message.Content).
// 			Msg("Processing response chunk")
// 		p := ai.Part{
// 			Text: chunk.Message.Content,
// 			Kind: ai.PartKind(chunk.Index),
// 		}
// 		finalResponse.Message.Content = append(finalResponse.Message.Content, &p)
// 	}

// 	log.Info().
// 		Str("method", "generator.generate").
// 		Str("model_name", g.model.Name).
// 		Int("content_parts", len(finalResponse.Message.Content)).
// 		Msg("Model generation completed successfully")
// 	return finalResponse, nil
// }

// // concatMessageParts concatenates message parts into a single string.
// func concatMessageParts(parts []*ai.Part) string {
// 	log.Debug().
// 		Str("method", "concatMessageParts").
// 		Int("parts", len(parts)).
// 		Msg("Concatenating message parts")
// 	var sb strings.Builder
// 	for _, part := range parts {
// 		if part.IsText() {
// 			sb.WriteString(part.Text)
// 		}
// 		// Ignore non-text parts (e.g., media, tools) as DeepSeek API doesn't support them.
// 	}
// 	result := sb.String()
// 	log.Debug().
// 		Str("method", "concatMessageParts").
// 		Str("result", result).
// 		Msg("Concatenation complete")
// 	return result
// }

// package deepseek

// import (
// 	"context"
// 	"fmt"
// 	"log"
// 	"strings"
// 	"sync"

// 	"github.com/firebase/genkit/go/ai"
// 	"github.com/firebase/genkit/go/genkit"

// 	deepseek "github.com/cohesion-org/deepseek-go"
// )

// const provider = "deepseek"

// var (
// 	mediaSupportedModels = []string{deepseek.DeepSeekChat, deepseek.DeepSeekCoder, deepseek.DeepSeekReasoner}
// 	// toolSupportedModels  = []string{
// 	// 	"qwq", "mistral-small3.1", "llama3.3", "llama3.2", "llama3.1", "mistral",
// 	// 	"qwen2.5", "qwen2.5-coder", "qwen2", "mistral-nemo", "mixtral", "smollm2",
// 	// 	"mistral-small", "command-r", "hermes3", "mistral-large", "command-r-plus",
// 	// 	"phi4-mini", "granite3.1-dense", "granite3-dense", "granite3.2", "athene-v2",
// 	// 	"nemotron-mini", "nemotron", "llama3-groq-tool-use", "aya-expanse", "granite3-moe",
// 	// 	"granite3.2-vision", "granite3.1-moe", "cogito", "command-r7b", "firefunction-v2",
// 	// 	"granite3.3", "command-a", "command-r7b-arabic",
// 	// }
// 	roleMapping = map[ai.Role]string{
// 		ai.RoleUser:   deepseek.ChatMessageRoleUser,
// 		ai.RoleModel:  deepseek.ChatMessageRoleAssistant,
// 		ai.RoleSystem: deepseek.ChatMessageRoleSystem,
// 		ai.RoleTool:   deepseek.ChatMessageRoleTool,
// 	}
// )

// // DeepSeek holds configuration for the plugin.
// type DeepSeek struct {
// 	APIKey string // DeepSeek API key
// 	//ServerAddress string

// 	mu      sync.Mutex // Mutex to control access.
// 	initted bool       // Whether the plugin has been initialized.
// }

// // Name returns the provider name.
// func (d DeepSeek) Name() string {
// 	return provider
// }

// // ModelDefinition represents a model with its name and type.
// type ModelDefinition struct {
// 	Name string
// 	Type string
// }

// // // DefineModel defines a DeepSeek model in Genkit.
// func (d *DeepSeek) DefineModel(g *genkit.Genkit, model ModelDefinition, info *ai.ModelInfo) ai.Model {
// 	d.mu.Lock()
// 	defer d.mu.Unlock()
// 	if !d.initted {
// 		panic("deepseek.Init not called")
// 	}

// 	// Define model info, supporting multiturn and system role.
// 	mi := ai.ModelInfo{
// 		Label: model.Name,
// 		Supports: &ai.ModelSupports{
// 			Multiturn:  true,
// 			SystemRole: true,
// 			Media:      false, // DeepSeek API primarily supports text.
// 			Tools:      false, // Tools not yet supported in this implementation.
// 		},
// 		Versions: []string{},
// 	}
// 	if info != nil {
// 		mi = *info
// 	}

// 	meta := &ai.ModelInfo{
// 		// Label:    "DeepSeek - " + model.Name,
// 		Label:    model.Name,
// 		Supports: mi.Supports,
// 		Versions: []string{},
// 	}
// 	gen := &generator{model: model, apiKey: d.APIKey}
// 	return genkit.DefineModel(g, provider, model.Name, meta, gen.generate)
// }

// // Init initializes the DeepSeek plugin.
// func (d *DeepSeek) Init(ctx context.Context, g *genkit.Genkit) error {
// 	d.mu.Lock()
// 	defer d.mu.Unlock()
// 	if d.initted {
// 		panic("deepseek.Init already called")
// 	}

// 	if d == nil || d.APIKey == "" {
// 		return fmt.Errorf("deepseek: need APIKey")
// 	}
// 	d.initted = true
// 	return nil
// }

// // generator handles model generation.
// type generator struct {
// 	model  ModelDefinition
// 	apiKey string
// }

// // generate implements the Genkit model generation interface.
// func (g *generator) generate(ctx context.Context, input *ai.ModelRequest, cb func(context.Context, *ai.ModelResponseChunk) error) (*ai.ModelResponse, error) {

// 	// stream := cb != nil
// 	if len(input.Messages) == 0 {
// 		return nil, fmt.Errorf("prompt or messages required")
// 	}
// 	// Set up the Deepseek client
// 	// Initialize DeepSeek client.
// 	client := deepseek.NewClient(g.apiKey)
// 	// Create a chat completion request
// 	request := &deepseek.ChatCompletionRequest{
// 		Model: g.model.Name,
// 	}

// 	for _, msg := range input.Messages {
// 		role, ok := roleMapping[msg.Role]
// 		if !ok {
// 			return nil, fmt.Errorf("unsupported role: %s", msg.Role)
// 		}
// 		content := concatMessageParts(msg.Content)
// 		request.Messages = append(request.Messages, deepseek.ChatCompletionMessage{
// 			Role:    role,
// 			Content: content,
// 		})
// 	}

// 	// Send the request and handle the response
// 	response, err := client.CreateChatCompletion(ctx, request)
// 	if err != nil {
// 		log.Fatalf("error: %v", err)
// 	}

// 	// Print the response
// 	fmt.Println("Response:", response.Choices[0].Message.Content)

// 	// Create a final response with the merged chunks
// 	finalResponse := &ai.ModelResponse{
// 		Request:      input,
// 		FinishReason: ai.FinishReason("stop"),
// 		Message: &ai.Message{
// 			Role: ai.RoleModel,
// 		},
// 	}

// 	for _, chunk := range response.Choices {
// 		p := ai.Part{
// 			Text: chunk.Message.Content,
// 			Kind: ai.PartKind(chunk.Index),
// 		}

// 		finalResponse.Message.Content = append(finalResponse.Message.Content, &p)
// 	}
// 	return finalResponse, nil // Return the final merged response
// }

// // concatMessageParts concatenates message parts into a single string.
// func concatMessageParts(parts []*ai.Part) string {
// 	var sb strings.Builder
// 	for _, part := range parts {
// 		if part.IsText() {
// 			sb.WriteString(part.Text)
// 		}
// 		// Ignore non-text parts (e.g., media, tools) as DeepSeek API doesn't support them.
// 	}
// 	return sb.String()
// }

/*

// Choice represents a completion choice generated by the model.
type Choice struct {
	Index        int     `json:"index"`              // Index of the choice in the list of choices.
	Message      Message `json:"message"`            // The message generated by the model.
	Logprobs     any     `json:"logprobs,omitempty"` // Log probabilities of the tokens, if available. // Changed to any as of April 21 2025 because the logprobs field is sometimes a flot64 and sometimes a Logprobs struct.
	FinishReason string  `json:"finish_reason"`      // Reason why the completion finished.
}

// A Part is one part of a [Document]. This may be plain text or it
// may be a URL (possibly a "data:" URL with embedded data).
type Part struct {
	Kind         PartKind       `json:"kind,omitempty"`
	ContentType  string         `json:"contentType,omitempty"`  // valid for kind==blob
	Text         string         `json:"text,omitempty"`         // valid for kind∈{text,blob}
	ToolRequest  *ToolRequest   `json:"toolRequest,omitempty"`  // valid for kind==partToolRequest
	ToolResponse *ToolResponse  `json:"toolResponse,omitempty"` // valid for kind==partToolResponse
	Custom       map[string]any `json:"custom,omitempty"`       // valid for plugin-specific custom parts
	Metadata     map[string]any `json:"metadata,omitempty"`     // valid for all kinds
}

*/
