package knowledge

import (
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"sync"

	"github.com/rs/zerolog/log"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/errors"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
)

// ClientConfig holds configuration options for the Tencent Cloud LKEAP client.
type ClientConfig struct {
	SecretID  string // Tencent Cloud Secret ID
	SecretKey string // Tencent Cloud Secret Key
	Token     string // Optional: Temporary token for authentication
	Endpoint  string // API endpoint (default: lkeap.tencentcloudapi.com)
	Region    string // Tencent Cloud region (optional)
}

// KnowledgeClient manages interactions with the Tencent Cloud LKEAP API.
type KnowledgeClient struct {
	client  *lkeap.Client
	config  ClientConfig
	mu      sync.Mutex
	initted bool
}

// NewKnowledgeClient creates a new KnowledgeClient with the given configuration.
func NewKnowledgeClient(config ClientConfig) *KnowledgeClient {
	log.Info().
		Str("method", "NewKnowledgeClient").
		Str("endpoint", config.Endpoint).
		Str("region", config.Region).
		Str("secret_id", maskCredential(config.SecretID)).
		Str("token", maskCredential(config.Token)).
		Msg("Creating new KnowledgeClient")
	return &KnowledgeClient{
		config: config,
	}
}

// Init initializes the KnowledgeClient.
func (kc *KnowledgeClient) Init(ctx context.Context) error {
	log.Info().Str("method", "KnowledgeClient.Init").Msg("Initializing KnowledgeClient")
	kc.mu.Lock()
	defer kc.mu.Unlock()

	if kc.initted {
		log.Error().Str("method", "KnowledgeClient.Init").Msg("Client already initialized")
		return fmt.Errorf("knowledge client already initialized")
	}

	// Load configuration from environment variables if not set
	if kc.config.SecretID == "" {
		kc.config.SecretID = os.Getenv("TENCENTCLOUD_SECRET_ID")
	}
	if kc.config.SecretKey == "" {
		kc.config.SecretKey = os.Getenv("TENCENTCLOUD_SECRET_KEY")
	}
	if kc.config.Token == "" {
		kc.config.Token = os.Getenv("TENCENTCLOUD_TOKEN")
	}
	if kc.config.Endpoint == "" {
		kc.config.Endpoint = "lkeap.tencentcloudapi.com"
	}
	if kc.config.Region == "" {
		kc.config.Region = "ap-guangzhou" // Default to ap-guangzhou as per curl
	}

	// Validate configuration
	if kc.config.SecretID == "" || kc.config.SecretKey == "" {
		log.Error().Str("method", "KnowledgeClient.Init").Msg("SecretID and SecretKey are required")
		return fmt.Errorf("knowledge: SecretID and SecretKey are required")
	}

	// Create credential
	var credential *common.Credential
	if kc.config.Token != "" {
		credential = common.NewTokenCredential(kc.config.SecretID, kc.config.SecretKey, kc.config.Token)
		log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using temporary token credential")
	} else {
		credential = common.NewCredential(kc.config.SecretID, kc.config.SecretKey)
		log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using standard credential")
	}

	// Create client profile
	cpf := profile.NewClientProfile()
	cpf.HttpProfile.Endpoint = kc.config.Endpoint

	// Initialize client
	client, err := lkeap.NewClient(credential, kc.config.Region, cpf)
	if err != nil {
		log.Error().
			Err(err).
			Str("method", "KnowledgeClient.Init").
			Msg("Failed to create LKEAP client")
		return err
	}

	kc.client = client
	kc.initted = true
	log.Info().Str("method", "KnowledgeClient.Init").Msg("Initialization successful")
	return nil
}

// QueryRewriteRequest defines the input for a query rewrite operation.
type QueryRewriteRequest struct {
	Messages []*lkeap.Message // Multi-turn conversation history (up to 4 turns)
	Model    string           // Model name for query rewriting
}

// QueryRewriteResponse defines the output of a query rewrite operation.
type QueryRewriteResponse struct {
	RewrittenQuery string // The rewritten query
	RawResponse    *lkeap.QueryRewriteResponse
}

// QueryRewrite performs a query rewrite using the Tencent Cloud LKEAP API.
func (kc *KnowledgeClient) QueryRewrite(ctx context.Context, req QueryRewriteRequest) (*QueryRewriteResponse, error) {
	log.Info().
		Str("method", "KnowledgeClient.QueryRewrite").
		Int("message_count", len(req.Messages)).
		Str("model", req.Model).
		Msg("Starting query rewrite operation")

	if !kc.initted {
		log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("Client not initialized")
		return nil, fmt.Errorf("knowledge client not initialized; call Init first")
	}

	// Validate input
	if len(req.Messages) == 0 {
		log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("At least one message is required")
		return nil, fmt.Errorf("at least one message is required")
	}
	if len(req.Messages) > 4 {
		log.Warn().
			Str("method", "KnowledgeClient.QueryRewrite").
			Int("message_count", len(req.Messages)).
			Msg("Message count exceeds 4, truncating to 4")
		req.Messages = req.Messages[:4]
	}
	for i, msg := range req.Messages {
		if msg.Role == nil || *msg.Role == "" {
			log.Error().
				Str("method", "KnowledgeClient.QueryRewrite").
				Int("index", i).
				Msg("Role is required in each message")
			return nil, fmt.Errorf("message at index %d missing role", i)
		}
		if *msg.Role != "user" && *msg.Role != "assistant" {
			log.Error().
				Str("method", "KnowledgeClient.QueryRewrite").
				Int("index", i).
				Str("role", *msg.Role).
				Msg("Invalid role; must be 'user' or 'assistant'")
			return nil, fmt.Errorf("invalid role '%s' at index %d", *msg.Role, i)
		}
		if msg.Content == nil || *msg.Content == "" {
			log.Error().
				Str("method", "KnowledgeClient.QueryRewrite").
				Int("index", i).
				Msg("Content is required in each message")
			return nil, fmt.Errorf("message at index %d missing content", i)
		}
		log.Debug().
			Str("method", "KnowledgeClient.QueryRewrite").
			Int("index", i).
			Str("role", *msg.Role).
			Str("content", *msg.Content).
			Msg("Validated message")
	}
	if req.Model == "" {
		log.Warn().Str("method", "KnowledgeClient.QueryRewrite").Msg("Model not specified, using default")
		req.Model = "lke-query-rewrite-base" // Default as per curl
	}

	// Create Tencent Cloud request
	tencentReq := lkeap.NewQueryRewriteRequest()
	tencentReq.Messages = req.Messages
	if req.Model != "" {
		tencentReq.Model = common.StringPtr(req.Model)
	}

	// Debug request
	tencentReqAsJson, _ := json.Marshal(tencentReq)
	log.Debug().
		Str("method", "KnowledgeClient.QueryRewrite").
		Str("request_json", string(tencentReqAsJson)).
		Msg("Prepared Tencent Cloud request")

	// Perform request
	response, err := kc.client.QueryRewriteWithContext(ctx, tencentReq)
	if err != nil {
		if _, ok := err.(*errors.TencentCloudSDKError); ok {
			log.Error().
				Err(err).
				Str("method", "KnowledgeClient.QueryRewrite").
				Msg("Tencent Cloud API error")
			return nil, fmt.Errorf("tencent cloud api error: %w", err)
		}
		log.Error().
			Err(err).
			Str("method", "KnowledgeClient.QueryRewrite").
			Msg("Failed to perform query rewrite")
		return nil, fmt.Errorf("query rewrite failed: %w", err)
	}

	// Extract response fields
	var rewrittenQuery string
	var requestId string
	if response.Response.Content != nil {
		rewrittenQuery = *response.Response.Content
	}
	if response.Response.RequestId != nil {
		requestId = *response.Response.RequestId
	}

	result := &QueryRewriteResponse{
		RewrittenQuery: rewrittenQuery,
		RawResponse:    response,
	}

	log.Info().
		Str("method", "KnowledgeClient.QueryRewrite").
		Str("rewritten_query", rewrittenQuery).
		Str("request_id", requestId).
		Interface("usage", response.Response.Usage).
		Str("raw_response", response.ToJsonString()).
		Msg("Query rewrite operation completed successfully")
	return result, nil
}

// maskCredential masks sensitive credentials for logging
func maskCredential(cred string) string {
	if len(cred) <= 8 {
		return strings.Repeat("*", len(cred))
	}
	return cred[:4] + strings.Repeat("*", len(cred)-8) + cred[len(cred)-4:]
}

// QueryRewriteWithSummary wraps QueryRewrite to handle a user question, assistant answer, and history summary.
func (kc *KnowledgeClient) QueryRewriteWithSummary(ctx context.Context, userQuestion, assistantAnswer, historySummary string) (*QueryRewriteResponse, error) {
	log.Info().
		Str("method", "KnowledgeClient.QueryRewriteWithSummary").
		Str("user_question", userQuestion).
		Str("assistant_answer", assistantAnswer).
		Str("history_summary", historySummary).
		Msg("Starting query rewrite with summary operation")

	if userQuestion == "" || assistantAnswer == "" {
		log.Error().Str("method", "KnowledgeClient.QueryRewriteWithSummary").Msg("User question and assistant answer are required")
		return nil, fmt.Errorf("user question and assistant answer are required")
	}

	// Construct messages
	messages := []*lkeap.Message{
		{
			Role:    common.StringPtr("user"),
			Content: common.StringPtr(userQuestion),
		},
		{
			Role:    common.StringPtr("assistant"),
			Content: common.StringPtr(assistantAnswer),
		},
	}

	// Append history summary as an assistant message if provided
	if historySummary != "" {
		messages = append(messages, &lkeap.Message{
			Role:    common.StringPtr("user"),
			Content: common.StringPtr(fmt.Sprintf("Conversation summary: %s", historySummary)),
		})
	}

	// Create request
	req := QueryRewriteRequest{
		Messages: messages,
		Model:    "lke-query-rewrite-base",
	}

	// Call QueryRewrite
	return kc.QueryRewrite(ctx, req)
}
















// ReconstructDocumentSSE performs document reconstruction using the Tencent Cloud LKEAP API.
// It takes a text string, encodes it as Base64, and returns the response as a JSON string.
func (kc *KnowledgeClient) ReconstructDocumentSSE(ctx context.Context, text string) (string, error) {
    log.Info().
        Str("method", "KnowledgeClient.ReconstructDocumentSSE").
        Str("text_prefix", text[:min(32, len(text))]).
        Msg("Starting document reconstruction operation")

    if !kc.initted {
        log.Error().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Client not initialized")
        return "", fmt.Errorf("knowledge client not initialized; call Init first")
    }

    // Validate input
    if text == "" {
        log.Error().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Text is required")
        return "", fmt.Errorf("text is required")
    }

    // Encode text to Base64
    base64Data := base64.StdEncoding.EncodeToString([]byte(text))
    // Check Base64 size (8MB limit)
    if len(base64Data)/4*3 > 8*1024*1024 {
        log.Error().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Encoded text exceeds 8MB limit")
        return "", fmt.Errorf("encoded text exceeds 8MB limit")
    }

    // Remove Base64 prefix if present
    if strings.HasPrefix(base64Data, "data:application/octet-stream;base64,") {
        base64Data = strings.TrimPrefix(base64Data, "data:application/octet-stream;base64,")
        log.Debug().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Removed Base64 data URI prefix")
    }

    // Create SDK request
    request := lkeap.NewReconstructDocumentSSERequest()
    request.FileType = common.StringPtr("TXT")
    request.FileBase64 = common.StringPtr(base64Data)

    // Debug request
    requestAsJson, _ := json.Marshal(request)
    log.Debug().
        Str("method", "KnowledgeClient.ReconstructDocumentSSE").
        Str("request_json", string(requestAsJson)).
        Msg("Prepared Tencent Cloud request")

    // Perform request
    response, err := kc.client.ReconstructDocumentSSEWithContext(ctx, request)
    if _, ok := err.(*errors.TencentCloudSDKError); ok {
        log.Error().
            Err(err).
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Msg("Tencent Cloud API error")
        return "", fmt.Errorf("tencent cloud api error: %s", err)
    }
    if err != nil {
        log.Error().
            Err(err).
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Msg("Failed to perform document reconstruction")
        return "", fmt.Errorf("document reconstruction failed: %w", err)
    }

    // Handle response
    var result string
    if response.Response != nil {
        // Non-streaming response
        result = response.ToJsonString()
        log.Info().
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Str("response_json", result).
            Msg("Received non-streaming response")
    } else {
        // Streaming response
        var events []string
        for event := range response.Events {
            eventData := string(event.Data)
            events = append(events, eventData)
            log.Debug().
                Str("method", "KnowledgeClient.ReconstructDocumentSSE").
                Str("event_data", eventData).
                Msg("Received streaming event")
        }
        // Combine events into a single JSON array
        eventsJson, _ := json.Marshal(events)
        result = string(eventsJson)
        log.Info().
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Str("response_json", result).
            Msg("Received streaming response")
    }

    log.Info().
        Str("method", "KnowledgeClient.ReconstructDocumentSSE").
        Msg("Document reconstruction operation completed successfully")

    return result, nil
}

// min returns the minimum of two integers
func min(a, b int) int {
    if a < b {
        return a
    }
    return b
}












// // DocumentFragment represents a split document fragment (custom definition due to SDK missing DocumentFragment).
// type DocumentFragment struct {
//     Content *string `json:"Content"`
//     Index   *int64  `json:"Index"`
// }

// // SplitDocument performs a synchronous document split using the Tencent Cloud LKEAP API.
// // It takes a text string, encodes it as Base64, and returns the split content as a slice of strings.
// func (kc *KnowledgeClient) SplitDocument(ctx context.Context, text string) ([]string, error) {
//     log.Info().
//         Str("method", "KnowledgeClient.SplitDocument").
//         Str("text_prefix", text[:min(32, len(text))]).
//         Msg("Starting document split operation")

//     if !kc.initted {
//         log.Error().Str("method", "KnowledgeClient.SplitDocument").Msg("Client not initialized")
//         return nil, fmt.Errorf("knowledge client not initialized; call Init first")
//     }

//     // Validate input
//     if text == "" {
//         log.Error().Str("method", "KnowledgeClient.SplitDocument").Msg("Text is required")
//         return nil, fmt.Errorf("text is required")
//     }

//     // Encode text to Base64
//     base64Data := base64.StdEncoding.EncodeToString([]byte(text))
//     // Check Base64 size (8MB limit)
//     if len(base64Data)/4*3 > 8*1024*1024 {
//         log.Error().Str("method", "KnowledgeClient.SplitDocument").Msg("Encoded text exceeds 8MB limit")
//         return nil, fmt.Errorf("encoded text exceeds 8MB limit")
//     }

//     // Create SDK request
//     tencentReq := lkeap.NewCreateSplitDocumentFlowRequest()
//     tencentReq.FileType = common.StringPtr("TXT")
//     tencentReq.FileBase64 = common.StringPtr(base64Data)
//     tencentReq.FileName = common.StringPtr("input.txt")
//     // tencentReq.SplitType = common.StringPtr("paragraph")

//     // Debug request
//     tencentReqAsJson, _ := json.Marshal(tencentReq)
//     log.Debug().
//         Str("method", "KnowledgeClient.SplitDocument").
//         Str("request_json", string(tencentReqAsJson)).
//         Msg("Prepared Tencent Cloud request")

//     // Perform request
//     response, err := kc.client.CreateSplitDocumentFlowWithContext(ctx, tencentReq)
//     if _, ok := err.(*errors.TencentCloudSDKError); ok {
//         log.Error().
//             Err(err).
//             Str("method", "KnowledgeClient.SplitDocument").
//             Msg("Tencent Cloud API error")
//         return nil, fmt.Errorf("tencent cloud api error: %s", err)
//     }
//     if err != nil {
//         log.Error().
//             Err(err).
//             Str("method", "KnowledgeClient.SplitDocument").
//             Msg("Failed to perform document split")
//         return nil, fmt.Errorf("document split failed: %w", err)
//     }


//     // Convert SplitResults to DocumentFragment and extract Content
//     var splitContents []string
//     for i, result := range *response.Response.TaskId {
//         resultJson, _ := json.Marshal(result)
//         var fragment DocumentFragment
//         if err := json.Unmarshal(resultJson, &fragment); err != nil {
//             log.Error().
//                 Err(err).
//                 Str("method", "KnowledgeClient.SplitDocument").
//                 Int("index", i).
//                 Msg("Failed to parse split result")
//             return nil, fmt.Errorf("failed to parse split result at index %d: %w", i, err)
//         }
//         if fragment.Content == nil {
//             log.Warn().
//                 Str("method", "KnowledgeClient.SplitDocument").
//                 Int("index", i).
//                 Msg("Split result has nil Content")
//             continue
//         }
//         splitContents = append(splitContents, *fragment.Content)
//     }

//     log.Info().
//         Str("method", "KnowledgeClient.SplitDocument").
//         Str("request_id", *response.Response.RequestId).
//         Int("results_count", len(splitContents)).
//         Str("raw_response", response.ToJsonString()).
//         Msg("Document split operation completed successfully")

//     return splitContents, nil
// }

// // min returns the minimum of two integers
// func min(a, b int) int {
//     if a < b {
//         return a
//     }
//     return b
// }






// func (kc *KnowledgeClient) ReconstructDocumentSSE(text string) {
// 	// 实例化一个认证对象，入参需要传入腾讯云账户 SecretId 和 SecretKey，此处还需注意密钥对的保密
// 	// 代码泄露可能会导致 SecretId 和 SecretKey 泄露，并威胁账号下所有资源的安全性
// 	// 以下代码示例仅供参考，建议采用更安全的方式来使用密钥
// 	// 请参见：https://cloud.tencent.com/document/product/1278/85305
// 	// 密钥可前往官网控制台 https://console.cloud.tencent.com/cam/capi 进行获取
// 	// credential := common.NewCredential(
// 	// 		"AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF",
// 	// 		"rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd",
// 	// )
// 	// // 使用临时密钥示例
// 	// // credential := common.NewTokenCredential("SecretId", "SecretKey", "Token")
// 	// // 实例化一个client选项，可选的，没有特殊需求可以跳过
// 	// cpf := profile.NewClientProfile()
// 	// cpf.HttpProfile.Endpoint = "lkeap.tencentcloudapi.com"
// 	// // 实例化要请求产品的client对象,clientProfile是可选的
// 	// client, _ := lkeap.NewClient(credential, "ap-guangzhou", cpf)

// 	// 实例化一个请求对象,每个接口都会对应一个request对象
// 	request := lkeap.NewReconstructDocumentSSERequest()


// 	    // Encode text to Base64
// 		base64Data := base64.StdEncoding.EncodeToString([]byte(text))
// 		// Check Base64 size (8MB limit)
// 		if len(base64Data)/4*3 > 8*1024*1024 {
// 			log.Error().Str("method", "KnowledgeClient.SplitDocument").Msg("Encoded text exceeds 8MB limit")
// 			return nil, fmt.Errorf("encoded text exceeds 8MB limit")
// 		}

	
// 	request.FileType = common.StringPtr("TXT")
// 	request.FileBase64 = common.StringPtr("data:application/octet-stream;base64,4oCc6L+Z5bCx6LGh6Iqx5LiA5qC344CC5aaC5p6c5L2g54ix5LiK5LqG5LiA5py155Sf6ZW/IArlnKjkuIDpopfmmJ/mmJ/kuIrnmoToirHvvIzpgqPkuYjlpJzpl7TvvIwgCuS9oOeci+edgOWkqeepuuWwseaEn+WIsOeUnOicnOaEiSAK5b+r44CC5omA5pyJ55qE5pif5pif5LiK6YO9CuWlveixoeW8gOedgOiKseOAguKAnQ==")
// 	// 返回的resp是一个ReconstructDocumentSSEResponse的实例，与请求对象对应
// 	response, err := kc.client.ReconstructDocumentSSE(request)
// 	if _, ok := err.(*errors.TencentCloudSDKError); ok {
// 			fmt.Printf("An API error has returned: %s", err)
// 			return
// 	}
// 	if err != nil {
// 			panic(err)
// 	}
// 	// 输出json格式的字符串回包
// 	if response.Response != nil {
// 		// 非流式响应
// 		fmt.Println(response.ToJsonString())
// 	} else {
// 		// 流式响应
// 		for event := range response.Events {
// 			fmt.Println(string(event.Data))
// 		}
// 	}
// } 








