// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

package graphrag

import (
	"bytes"
	"context"
	"crypto/rand"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"math/big"
	"net/http"
	"strconv"
	"strings"
	"sync"

	"github.com/firebase/genkit/go/ai"
	"github.com/firebase/genkit/go/genkit"
	"github.com/rs/zerolog/log"
	"github.com/wade-liwei/agentchat/util"
)

// Client 知识库客户端
type Client struct {
    BaseURL string // 基础URL，例如 "http://54.92.111.204:5670"
}

// SpaceRequest 创建空间的请求结构体
type SpaceRequest struct {
    ID         int    `json:"id"`
    Name       string `json:"name"`
    VectorType string `json:"vector_type"`
    DomainType string `json:"domain_type"`
    Desc       string `json:"desc"`
    Owner      string `json:"owner"`
    SpaceID    int    `json:"space_id"`
}

// DocumentRequest 添加文档的请求结构体
type DocumentRequest struct {
    DocName   string                 `json:"doc_name"`
    DocID     int                    `json:"doc_id"`
    DocType   string                 `json:"doc_type"`
    DocToken  string                 `json:"doc_token"`
    Content   string                 `json:"content"`
    Source    string                 `json:"source"`
    Labels    string                 `json:"labels"`
    Questions []string               `json:"questions"`
    Metadata  map[string]interface{} `json:"metadata"`
}

// ChunkParameters 分片参数
type ChunkParameters struct {
    ChunkStrategy string `json:"chunk_strategy"`
    TextSplitter  string `json:"text_splitter"`
    SplitterType  string `json:"splitter_type"`
    ChunkSize     int    `json:"chunk_size"`
    ChunkOverlap  int    `json:"chunk_overlap"`
    Separator     string `json:"separator"`
    EnableMerge   bool   `json:"enable_merge"`
}

// SyncBatchRequest 同步批量处理的请求结构体
type SyncBatchRequest struct {
    DocID           int             `json:"doc_id"`
    SpaceID         string          `json:"space_id"`
    ModelName       string          `json:"model_name"`
    ChunkParameters ChunkParameters `json:"chunk_parameters"`
}

// NewClient 创建新的客户端实例
func NewClient(ip string, port int) *Client {
    log.Info().
        Str("method", "NewClient").
        Str("ip", ip).
        Int("port", port).
        Msg("Creating new GraphRAG client")
    return &Client{
        BaseURL: fmt.Sprintf("http://%s:%d", ip, port),
    }
}

// AddSpace 创建知识空间
func (c *Client) AddSpace(req SpaceRequest) (*http.Response, error) {
    log.Info().
        Str("method", "Client.AddSpace").
        Str("name", req.Name).
        Str("owner", req.Owner).
        Msg("Adding knowledge space")
    url := fmt.Sprintf("%s/knowledge/space/add", c.BaseURL)
    body, err := json.Marshal(req)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.AddSpace").Msg("Failed to marshal request")
        return nil, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        log.Error().Err(err).Str("method", "Client.AddSpace").Msg("Failed to create request")
        return nil, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.AddSpace").Msg("Failed to send request")
        return nil, fmt.Errorf("failed to send request: %w", err)
    }

    log.Info().
        Str("method", "Client.AddSpace").
        Int("status_code", resp.StatusCode).
        Msg("Space addition request completed")
    return resp, nil
}

// AddDocument 添加文档
func (c *Client) AddDocument(spaceID string, req DocumentRequest) (*http.Response, error) {
    log.Info().
        Str("method", "Client.AddDocument").
        Str("space_id", spaceID).
        Str("doc_name", req.DocName).
        Msg("Adding document")
    url := fmt.Sprintf("%s/knowledge/%s/document/add", c.BaseURL, spaceID)
    body, err := json.Marshal(req)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.AddDocument").Msg("Failed to marshal request")
        return nil, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        log.Error().Err(err).Str("method", "Client.AddDocument").Msg("Failed to create request")
        return nil, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.AddDocument").Msg("Failed to send request")
        return nil, fmt.Errorf("failed to send request: %w", err)
    }

    log.Info().
        Str("method", "Client.AddDocument").
        Str("space_id", spaceID).
        Int("status_code", resp.StatusCode).
        Msg("Document addition request completed")
    return resp, nil
}

// SyncDocumentsRequest defines the request body for the sync documents endpoint.
type SyncDocumentsRequest struct {
    DocIDs []string `json:"doc_ids"`
}

// SyncDocuments sends a POST request to sync documents for the given spaceID.
func (c *Client) SyncDocuments(spaceID string, docIDs []string) (success bool, err error) {
    log.Info().
        Str("method", "Client.SyncDocuments").
        Str("space_id", spaceID).
        Strs("doc_ids", docIDs).
        Msg("Syncing documents")
    url := fmt.Sprintf("%s/knowledge/%s/document/sync", c.BaseURL, spaceID)
    reqBody := SyncDocumentsRequest{
        DocIDs: docIDs,
    }
    body, err := json.Marshal(reqBody)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncDocuments").Msg("Failed to marshal request")
        return false, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncDocuments").Msg("Failed to create request")
        return false, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncDocuments").Msg("Failed to send request")
        return false, fmt.Errorf("failed to send request: %w", err)
    }
    defer resp.Body.Close()

    respBody, err := io.ReadAll(resp.Body)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncDocuments").Msg("Failed to read response body")
        return false, fmt.Errorf("failed to read response body: %w", err)
    }

    if resp.StatusCode != http.StatusOK {
        log.Error().
            Str("method", "Client.SyncDocuments").
            Int("status_code", resp.StatusCode).
            Str("response_body", string(respBody)).
            Msg("Sync request failed")
        return false, fmt.Errorf("request failed with status %d: %s", resp.StatusCode, string(respBody))
    }

    log.Info().
        Str("method", "Client.SyncDocuments").
        Str("space_id", spaceID).
        Msg("Documents synced successfully")
    return true, nil
}

// SyncBatchDocument 同步批量处理文档
func (c *Client) SyncBatchDocument(spaceID string, req []SyncBatchRequest) (*http.Response, error) {
    log.Info().
        Str("method", "Client.SyncBatchDocument").
        Str("space_id", spaceID).
        Int("requests", len(req)).
        Msg("Syncing batch documents")
    url := fmt.Sprintf("%s/knowledge/%s/document/sync_batch", c.BaseURL, spaceID)
    body, err := json.Marshal(req)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncBatchDocument").Msg("Failed to marshal request")
        return nil, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncBatchDocument").Msg("Failed to create request")
        return nil, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        log.Error().Err(err).Str("method", "Client.SyncBatchDocument").Msg("Failed to send request")
        return nil, fmt.Errorf("failed to send request: %w", err)
    }

    log.Info().
        Str("method", "Client.SyncBatchDocument").
        Str("space_id", spaceID).
        Int("status_code", resp.StatusCode).
        Msg("Batch document sync request completed")
    return resp, nil
}

// The provider used in the registry.
const provider = "graphrag"

// Field names for schema.
const (
    idField       = "id"
    textField     = "text"
    metadataField = "metadata"
)

// GraphKnowledge holds configuration for the plugin.
type GraphKnowledge struct {
    Addr string // Knowledge server address (host:port, e.g., "54.92.111.204:5670").

    client  *Client    // Knowledge client.
    mu      sync.Mutex // Mutex to control access.
    initted bool       // Whether the plugin has been initialized.
}

// Name returns the plugin name.
func (k *GraphKnowledge) Name() string {
    return provider
}

// Init initializes the GraphKnowledge plugin.
func (k *GraphKnowledge) Init(ctx context.Context, g *genkit.Genkit) (err error) {
    log.Info().Str("method", "GraphKnowledge.Init").Msg("Initializing GraphKnowledge plugin")
    if k == nil {
        k = &GraphKnowledge{}
    }

    k.mu.Lock()
    defer k.mu.Unlock()
    defer func() {
        if err != nil {
            log.Error().Err(err).Str("method", "GraphKnowledge.Init").Msg("Initialization failed")
            err = fmt.Errorf("graphrag.Init: %w", err)
        } else {
            log.Info().Str("method", "GraphKnowledge.Init").Msg("Initialization successful")
        }
    }()

    if k.initted {
        return errors.New("plugin already initialized")
    }

    // Load configuration.
    addr := k.Addr
    if addr == "" {
        addr = "54.92.111.204:5670" // Default address.
    }

    // Initialize Knowledge client.
    host, port := parseAddr(addr)
    client := NewClient(host, port)
    k.client = client
    k.initted = true
    return nil
}

// parseAddr splits host:port into host and port.
func parseAddr(addr string) (string, int) {
    parts := strings.Split(addr, ":")
    if len(parts) != 2 {
        log.Warn().
            Str("method", "parseAddr").
            Str("addr", addr).
            Msg("Invalid address format, using default")
        return "54.92.111.204", 5670
    }
    port, err := strconv.Atoi(parts[1])
    if err != nil {
        log.Error().
            Err(err).
            Str("method", "parseAddr").
            Str("port", parts[1]).
            Msg("Failed to parse port, using default")
        return "54.92.111.204", 5670
    }
    return parts[0], port
}

// DefineIndexerAndRetriever defines an Indexer and Retriever for a Knowledge space.
func DefineIndexerAndRetriever(ctx context.Context, g *genkit.Genkit) (ai.Indexer, ai.Retriever, error) {
    log.Info().Str("method", "DefineIndexerAndRetriever").Msg("Defining indexer and retriever")
    spaceID := ""
    modelName := "Qwen/Qwen2.5-Coder-32B-Instruct"

    k := genkit.LookupPlugin(g, provider)
    if k == nil {
        log.Error().Str("method", "DefineIndexerAndRetriever").Msg("GraphRAG plugin not found")
        return nil, nil, errors.New("graphrag plugin not found; did you call genkit.Init with the graphrag plugin?")
    }
    knowledge := k.(*GraphKnowledge)

    ds, err := knowledge.newDocStore(ctx, spaceID, modelName)
    if err != nil {
        log.Error().Err(err).Str("method", "DefineIndexerAndRetriever").Msg("Failed to create doc store")
        return nil, nil, err
    }

    indexer := genkit.DefineIndexer(g, provider, spaceID, ds.Index)
    retriever := genkit.DefineRetriever(g, provider, spaceID, ds.Retrieve)
    log.Info().Str("method", "DefineIndexerAndRetriever").Msg("Indexer and retriever defined successfully")
    return indexer, retriever, nil
}

// docStore defines an Indexer and a Retriever.
type docStore struct {
    client    *Client
    spaceID   string
    modelName string
}

// newDocStore creates a docStore.
func (k *GraphKnowledge) newDocStore(ctx context.Context, spaceID, modelName string) (*docStore, error) {
    log.Info().
        Str("method", "GraphKnowledge.newDocStore").
        Str("space_id", spaceID).
        Str("model_name", modelName).
        Msg("Creating new doc store")
    if k.client == nil {
        log.Error().Str("method", "GraphKnowledge.newDocStore").Msg("GraphRAG client not initialized")
        return nil, errors.New("graphrag.Init not called")
    }

    log.Info().Str("method", "GraphKnowledge.newDocStore").Msg("Doc store created successfully")
    return &docStore{
        client:    k.client,
        spaceID:   spaceID,
        modelName: modelName,
    }, nil
}

// Indexer returns the indexer for a space.
func Indexer(g *genkit.Genkit, spaceID string) ai.Indexer {
    log.Info().
        Str("method", "Indexer").
        Str("space_id", spaceID).
        Msg("Looking up indexer")
    indexer := genkit.LookupIndexer(g, provider, spaceID)
    if indexer == nil {
        log.Warn().
            Str("method", "Indexer").
            Str("space_id", spaceID).
            Msg("Indexer not found")
    }
    return indexer
}

// Retriever returns the retriever for a space.
func Retriever(g *genkit.Genkit, spaceID string) ai.Retriever {
    log.Info().
        Str("method", "Retriever").
        Str("space_id", spaceID).
        Msg("Looking up retriever")
    retriever := genkit.LookupRetriever(g, provider, spaceID)
    if retriever == nil {
        log.Warn().
            Str("method", "Retriever").
            Str("space_id", spaceID).
            Msg("Retriever not found")
    }
    return retriever
}

// generateRandomDocName generates a random alphanumeric string of the specified length.
func GenerateRandomDocName(length int) (string, error) {
    log.Debug().
        Str("method", "GenerateRandomDocName").
        Int("length", length).
        Msg("Generating random document name")
    const charset = "abcdefghijklmnopqrstuvwxyz0123456789"
    var result strings.Builder
    result.Grow(length)

    for i := 0; i < length; i++ {
        idx, err := rand.Int(rand.Reader, big.NewInt(int64(len(charset))))
        if err != nil {
            log.Error().
                Err(err).
                Str("method", "GenerateRandomDocName").
                Msg("Failed to generate random index")
            return "", fmt.Errorf("failed to generate random index: %w", err)
        }
        result.WriteByte(charset[idx.Int64()])
    }

    docName := result.String()
    log.Debug().
        Str("method", "GenerateRandomDocName").
        Str("doc_name", docName).
        Msg("Generated document name")
    return docName, nil
}

// ParseJSONResponse parses a JSON byte slice and extracts the success boolean and data fields as a string.
func ParseJSONResponse(jsonBytes []byte) (success bool, data string, err error) {
    log.Debug().
        Str("method", "ParseJSONResponse").
        Str("json", string(jsonBytes)).
        Msg("Parsing JSON response")
    // Define struct to capture only the needed fields
    type jsonResponse struct {
        Success bool `json:"success"`
        Data    int  `json:"data"`
    }

    var resp jsonResponse
    if err := json.Unmarshal(jsonBytes, &resp); err != nil {
        log.Error().
            Err(err).
            Str("method", "ParseJSONResponse").
            Msg("Failed to unmarshal JSON")
        return false, "", fmt.Errorf("failed to unmarshal JSON: %w", err)
    }

    dataStr := fmt.Sprintf("%d", resp.Data)
    log.Debug().
        Str("method", "ParseJSONResponse").
        Bool("success", resp.Success).
        Str("data", dataStr).
        Msg("Parsed JSON response")
    return resp.Success, dataStr, nil
}

type IndexReqOption struct {
    UserId   string
    UserName string
}

const DocNameKey = "doc_name"

// Index implements the Indexer.Index method.
func (ds *docStore) Index(ctx context.Context, req *ai.IndexerRequest) error {
    log.Info().
        Str("method", "docStore.Index").
        Str("space_id", ds.spaceID).
        Int("documents", len(req.Documents)).
        Msg("Starting index operation")

    if len(req.Documents) == 0 {
        log.Debug().
            Str("method", "docStore.Index").
            Str("space_id", ds.spaceID).
            Msg("No documents to index")
        return nil
    }

    // Type-assert req.Options to IndexReqOption
    opt, ok := req.Options.(*IndexReqOption)
    if !ok {
        log.Error().
            Str("method", "docStore.Index").
            Str("options_type", fmt.Sprintf("%T", req.Options)).
            Msg("Invalid options type")
        return fmt.Errorf("invalid options type: got %T, want *IndexReqOption", req.Options)
    }

    // Validate required fields
    if opt.UserId == "" {
        log.Error().Str("method", "docStore.Index").Msg("UserId is required")
        return fmt.Errorf("UserId is required in IndexReqOption")
    }
    if opt.UserName == "" {
        log.Error().Str("method", "docStore.Index").Msg("UserName is required")
        return fmt.Errorf("UserName is required in IndexReqOption")
    }

    // Create knowledge space
    spaceReq := SpaceRequest{
        Name:       opt.UserId,
        VectorType: "KnowledgeGraph",
        DomainType: "Normal",
        Desc:       opt.UserName,
        Owner:      opt.UserId,
    }
    resp, err := ds.client.AddSpace(spaceReq)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Index").Msg("Failed to add space")
        return fmt.Errorf("add space: %w", err)
    }
    defer resp.Body.Close()
    if resp.StatusCode != http.StatusOK {
        body, _ := io.ReadAll(resp.Body)
        log.Error().
            Str("method", "docStore.Index").
            Int("status_code", resp.StatusCode).
            Str("response_body", string(body)).
            Msg("Add space failed")
        return fmt.Errorf("add space failed with status %d: %s", resp.StatusCode, string(body))
    }

    log.Info().Str("method", "docStore.Index").Str("space_id", opt.UserId).Msg("Space created successfully")
    spaceId := opt.UserId

    // Index each document
    for i, doc := range req.Documents {
        // Use DocName from metadata
        docName, ok := doc.Metadata[DocNameKey].(string)
        if !ok {
            log.Error().
                Str("method", "docStore.Index").
                Int("index", i).
                Msg("Missing doc_name in metadata")
            return fmt.Errorf("must provide doc_name key in metadata")
        }
        if docName == "" {
            log.Error().
                Str("method", "docStore.Index").
                Int("index", i).
                Msg("doc_name is empty")
            return fmt.Errorf("must provide non-empty doc_name str value in metadata")
        }

        log.Debug().
            Str("method", "docStore.Index").
            Int("index", i).
            Str("doc_name", docName).
            Msg("Processing document")

        // Add document
        var sb strings.Builder
        for _, p := range doc.Content {
            if p.IsText() {
                sb.WriteString(p.Text)
            }
        }
        text := sb.String()
        log.Debug().
            Str("method", "docStore.Index").
            Int("index", i).
            Str("text", text).
            Msg("Extracted document text")

        docReq := DocumentRequest{
            DocName:  docName,
            Source:   "api",
            DocType:  "TEXT",
            Content:  text,
            Labels:   "",
            Metadata: doc.Metadata,
        }
        resp, err := ds.client.AddDocument(spaceId, docReq)
        if err != nil {
            log.Error().
                Err(err).
                Str("method", "docStore.Index").
                Int("index", i+1).
                Msg("Failed to add document")
            return fmt.Errorf("add document %d: %w", i+1, err)
        }
        body, err := io.ReadAll(resp.Body)
        if err != nil {
            resp.Body.Close()
            log.Error().
                Err(err).
                Str("method", "docStore.Index").
                Int("index", i+1).
                Msg("Failed to read add document response")
            return fmt.Errorf("read add document response %d: %w", i+1, err)
        }
        defer resp.Body.Close()

        if resp.StatusCode != http.StatusOK {
            log.Error().
                Str("method", "docStore.Index").
                Int("index", i+1).
                Int("status_code", resp.StatusCode).
                Str("response_body", string(body)).
                Msg("Add document failed")
            return fmt.Errorf("add document %d failed with status %d: %s", i+1, resp.StatusCode, string(body))
        }

        // Parse AddDocument response
        ok, idx, err := ParseJSONResponse(body)
        if err != nil {
            log.Error().
                Err(err).
                Str("method", "docStore.Index").
                Int("index", i+1).
                Msg("Failed to parse add document")
            return fmt.Errorf("parse add document response %d: %w", i+1, err)
        }
        if !ok {
            log.Error().
                Str("method", "docStore.Index").
                Int("index", i+1).
				Str("data", idx).
                Msg("Add document response indicated failure")
            return fmt.Errorf("add document %d failed: response success=false, data=%s", i+1, idx)
        }

        log.Info().
            Str("method", "docStore.Index").
            Int("index", i+1).
            Str("doc_id", idx).
            Msg("Document added successfully")

        // Sync document
        _, err = ds.client.SyncDocuments(spaceId, []string{idx})
        if err != nil {
            log.Error().
                Err(err).
                Str("method", "docStore.Index").
                Int("index", i+1).
                Msg("Failed to sync document")
            return fmt.Errorf("sync document %d: %w", i+1, err)
        }
    }

    log.Info().
        Str("method", "docStore.Index").
        Str("space_id", ds.spaceID).
        Int("documents", len(req.Documents)).
        Msg("Index operation completed successfully")
    return nil
}

// ChatRequest defines the request structure for chat completions.
type ChatRequest struct {
    Model             string  `json:"model"`
    Messages          string  `json:"messages"`
    Temperature       float64 `json:"temperature"`
    TopP              float64 `json:"top_p"`
    TopK              int     `json:"top_k"`
    N                 int     `json:"n"`
    MaxTokens         int64   `json:"max_tokens"`
    Stream            bool    `json:"stream"`
    RepetitionPenalty float64 `json:"repetition_penalty"`
    FrequencyPenalty  float64 `json:"frequency_penalty"`
    PresencePenalty   float64 `json:"presence_penalty"`
    ChatMode          string  `json:"chat_mode"`
    ChatParam         string  `json:"chat_param"`
    EnableVis         bool    `json:"enable_vis"`
}

// ChatResponse defines the response structure from the API.
type ChatResponse struct {
    ID      string `json:"id"`
    Object  string `json:"object"`
    Created int64  `json:"created"`
    Model   string `json:"model"`
    Choices []struct {
        Index        int `json:"index"`
        Message      struct {
            Role             string      `json:"role"`
            Content          string      `json:"content"`
            ReasoningContent interface{} `json:"reasoning_content"`
        } `json:"message"`
        FinishReason interface{} `json:"finish_reason"`
    } `json:"choices"`
    Usage struct {
        PromptTokens     int `json:"prompt_tokens"`
        TotalTokens      int `json:"total_tokens"`
        CompletionTokens int `json:"completion_tokens"`
    } `json:"usage"`
}

// Assuming ai.Part has a Text() method or Text field to get string content.
func partsToString(parts []*ai.Part) string {
    log.Debug().
        Str("method", "partsToString").
        Int("parts", len(parts)).
        Msg("Converting parts to string")
    var texts []string
    for _, part := range parts {
        texts = append(texts, part.Text)
    }
    result := strings.Join(texts, " ")
    log.Debug().
        Str("method", "partsToString").
        Str("result", result).
        Msg("Conversion complete")
    return result
}

// Retrieve implements the Retriever.Retrieve method.
func (ds *docStore) Retrieve(ctx context.Context, req *ai.RetrieverRequest) (*ai.RetrieverResponse, error) {
    log.Info().
        Str("method", "docStore.Retrieve").
        Str("space_id", ds.spaceID).
        Msg("Starting retrieve operation")

    // Format query for retrieval.
    queryContent := partsToString(req.Query.Content)
    queryText := fmt.Sprintf("Search for: %s", queryContent)
    log.Debug().
        Str("method", "docStore.Retrieve").
        Str("query", queryText).
        Msg("Formatted query")

    if req.Query.Metadata == nil {
        log.Error().
            Str("method", "docStore.Retrieve").
            Str("metadata_type", fmt.Sprintf("%T", req.Query.Metadata)).
            Msg("Query metadata is nil")
        return nil, fmt.Errorf("req.Query.Metadata must be not nil, got type %T", req.Query.Metadata)
    }

    for k, v := range req.Query.Metadata {
        log.Debug().
            Str("method", "docStore.Retrieve").
            Str("key", k).
            Interface("value", v).
            Msg("Metadata entry")
    }

    // Extract username and user_id from req.Query.Metadata
    userName, ok := req.Query.Metadata[util.UserNameKey].(string)
    if !ok {
        log.Error().Str("method", "docStore.Retrieve").Msg("Missing username in metadata")
        return nil, fmt.Errorf("req.Query.Metadata must provide username key")
    }
    userId, ok := req.Query.Metadata[util.UserIdKey].(string)
    if !ok {
        log.Error().Str("method", "docStore.Retrieve").Msg("Missing user_id in metadata")
        return nil, fmt.Errorf("req.Query.Metadata must provide user_id key")
    }

    // Prepare request for chat completions endpoint.
    url := fmt.Sprintf("%s/api/v2/chat/completions", ds.client.BaseURL)
    chatReq := ChatRequest{
        Model:             ds.modelName,
        Messages:          queryText,
        Temperature:       0.7,
        TopP:              1,
        TopK:              -1,
        N:                 1,
        MaxTokens:         0,
        Stream:            false,
        RepetitionPenalty: 1,
        FrequencyPenalty:  0,
        PresencePenalty:   0,
        ChatMode:          "chat_knowledge",
        ChatParam:         userId,
        EnableVis:         true,
    }

    log.Debug().
        Str("method", "docStore.Retrieve").
        Str("url", url).
        Interface("chat_request", chatReq).
        Msg("Preparing chat completion request")

    body, err := json.Marshal(chatReq)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Msg("Failed to marshal chat request")
        return nil, fmt.Errorf("marshal chat request: %w", err)
    }

    httpReq, err := http.NewRequestWithContext(ctx, "POST", url, bytes.NewBuffer(body))
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Msg("Failed to create chat request")
        return nil, fmt.Errorf("create chat request: %w", err)
    }
    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Msg("Failed to send chat request")
        return nil, fmt.Errorf("send chat request: %w", err)
    }
    defer resp.Body.Close()

    if resp.StatusCode != http.StatusOK {
        body, _ := io.ReadAll(resp.Body)
        log.Error().
            Str("method", "docStore.Retrieve").
            Int("status_code", resp.StatusCode).
            Str("response_body", string(body)).
            Msg("Chat completion failed")
        return nil, fmt.Errorf("chat completion failed with status %d: %s", resp.StatusCode, string(body))
    }

    // Parse response
    var chatResp ChatResponse
    respBody, err := io.ReadAll(resp.Body)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Msg("Failed to read response body")
        return nil, fmt.Errorf("read chat response body: %w", err)
    }
    if err := json.Unmarshal(respBody, &chatResp); err != nil {
        log.Error().
            Err(err).
            Str("method", "docStore.Retrieve").
            Str("raw_response", string(respBody)).
            Msg("Failed to decode chat response")
        return nil, fmt.Errorf("decode chat response: %w, raw response: %s", err, string(respBody))
    }

    log.Debug().
        Str("method", "docStore.Retrieve").
        Interface("chat_response", chatResp).
        Msg("Parsed chat response")

    // Convert response to ai.Document
    var docs []*ai.Document
    if len(chatResp.Choices) > 0 {
        content := chatResp.Choices[0].Message.Content
        metadata := map[string]interface{}{
            util.UserIdKey:   userId,
            util.UserNameKey: userName,
        }
        aiDoc := ai.DocumentFromText(content, metadata)
        docs = append(docs, aiDoc)
        log.Debug().
            Str("method", "docStore.Retrieve").
            Str("content", content).
            Interface("metadata", metadata).
            Msg("Created document from response")
    }

    log.Info().
        Str("method", "docStore.Retrieve").
        Str("space_id", ds.spaceID).
        Int("documents", len(docs)).
        Msg("Retrieve operation completed successfully")
    return &ai.RetrieverResponse{
        Documents: docs,
    }, nil
}



















// // Copyright 2025 Google LLC
// //
// // Licensed under the Apache License, Version 2.0 (the "License");
// // you may not use this file except in compliance with the License.
// // You may obtain a copy of the License at
// //
// //     http://www.apache.org/licenses/LICENSE-2.0
// //
// // Unless required by applicable law or agreed to in writing, software
// // distributed under the License is distributed on an "AS IS" BASIS,
// // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// // See the License for the specific language governing permissions and
// // limitations under the License.
// //
// // SPDX-License-Identifier: Apache-2.0

// package graphrag

// import (
// 	"bytes"
// 	"context"
// 	"crypto/rand"
// 	"encoding/json"
// 	"errors"
// 	"fmt"
// 	"io"
// 	"math/big"
// 	"net/http"
// 	"strconv"
// 	"strings"
// 	"sync"

// 	"github.com/firebase/genkit/go/ai"
// 	"github.com/firebase/genkit/go/genkit"

// 	"github.com/wade-liwei/agentchat/util"
// )

// // Client 知识库客户端
// type Client struct {
// 	BaseURL string // 基础URL，例如 "http://54.92.111.204:5670"
// }

// // SpaceRequest 创建空间的请求结构体
// type SpaceRequest struct {
// 	ID         int    `json:"id"`
// 	Name       string `json:"name"`
// 	VectorType string `json:"vector_type"`
// 	DomainType string `json:"domain_type"`
// 	Desc       string `json:"desc"`
// 	Owner      string `json:"owner"`
// 	SpaceID    int    `json:"space_id"`
// }

// // DocumentRequest 添加文档的请求结构体
// type DocumentRequest struct {
// 	DocName   string                 `json:"doc_name"`
// 	DocID     int                    `json:"doc_id"`
// 	DocType   string                 `json:"doc_type"`
// 	DocToken  string                 `json:"doc_token"`
// 	Content   string                 `json:"content"`
// 	Source    string                 `json:"source"`
// 	Labels    string                 `json:"labels"`
// 	Questions []string               `json:"questions"`
// 	Metadata  map[string]interface{} `json:"metadata"`
// }

// // ChunkParameters 分片参数
// type ChunkParameters struct {
// 	ChunkStrategy string `json:"chunk_strategy"`
// 	TextSplitter  string `json:"text_splitter"`
// 	SplitterType  string `json:"splitter_type"`
// 	ChunkSize     int    `json:"chunk_size"`
// 	ChunkOverlap  int    `json:"chunk_overlap"`
// 	Separator     string `json:"separator"`
// 	EnableMerge   bool   `json:"enable_merge"`
// }

// // SyncBatchRequest 同步批量处理的请求结构体
// type SyncBatchRequest struct {
// 	DocID           int             `json:"doc_id"`
// 	SpaceID         string          `json:"space_id"`
// 	ModelName       string          `json:"model_name"`
// 	ChunkParameters ChunkParameters `json:"chunk_parameters"`
// }

// // NewClient 创建新的客户端实例
// func NewClient(ip string, port int) *Client {
// 	return &Client{
// 		BaseURL: fmt.Sprintf("http://%s:%d", ip, port),
// 	}
// }

// // AddSpace 创建知识空间
// func (c *Client) AddSpace(req SpaceRequest) (*http.Response, error) {
// 	url := fmt.Sprintf("%s/knowledge/space/add", c.BaseURL)
// 	body, err := json.Marshal(req)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to marshal request: %w", err)
// 	}

// 	httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to create request: %w", err)
// 	}

// 	httpReq.Header.Set("Accept", "application/json")
// 	httpReq.Header.Set("Content-Type", "application/json")

// 	client := &http.Client{}
// 	resp, err := client.Do(httpReq)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to send request: %w", err)
// 	}

// 	return resp, nil
// }

// // AddDocument 添加文档
// func (c *Client) AddDocument(spaceID string, req DocumentRequest) (*http.Response, error) {
// 	url := fmt.Sprintf("%s/knowledge/%s/document/add", c.BaseURL, spaceID)
// 	body, err := json.Marshal(req)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to marshal request: %w", err)
// 	}

// 	httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to create request: %w", err)
// 	}

// 	httpReq.Header.Set("Accept", "application/json")
// 	httpReq.Header.Set("Content-Type", "application/json")

// 	client := &http.Client{}
// 	resp, err := client.Do(httpReq)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to send request: %w", err)
// 	}

// 	return resp, nil
// }

// // SyncDocumentsRequest defines the request body for the sync documents endpoint.
// type SyncDocumentsRequest struct {
// 	DocIDs []string `json:"doc_ids"`
// }

// // SyncDocuments sends a POST request to sync documents for the given spaceID.
// func (c *Client) SyncDocuments(spaceID string, docIDs []string) (success bool, err error) {
// 	url := fmt.Sprintf("%s/knowledge/%s/document/sync", c.BaseURL, spaceID)
// 	reqBody := SyncDocumentsRequest{
// 		DocIDs: docIDs,
// 	}
// 	body, err := json.Marshal(reqBody)
// 	if err != nil {
// 		return false, fmt.Errorf("failed to marshal request: %w", err)
// 	}

// 	httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
// 	if err != nil {
// 		return false, fmt.Errorf("failed to create request: %w", err)
// 	}

// 	httpReq.Header.Set("Accept", "application/json")
// 	httpReq.Header.Set("Content-Type", "application/json")

// 	client := &http.Client{}
// 	resp, err := client.Do(httpReq)
// 	if err != nil {
// 		return false, fmt.Errorf("failed to send request: %w", err)
// 	}
// 	defer resp.Body.Close()

// 	respBody, err := io.ReadAll(resp.Body)
// 	if err != nil {
// 		return false, fmt.Errorf("failed to read response body: %w", err)
// 	}

// 	if resp.StatusCode != http.StatusOK {
// 		return false, fmt.Errorf("request failed with status %d: %s", resp.StatusCode, string(respBody))
// 	}

// 	return success, nil
// }

// // SyncBatchDocument 同步批量处理文档
// func (c *Client) SyncBatchDocument(spaceID string, req []SyncBatchRequest) (*http.Response, error) {
// 	url := fmt.Sprintf("%s/knowledge/%s/document/sync_batch", c.BaseURL, spaceID)
// 	body, err := json.Marshal(req)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to marshal request: %w", err)
// 	}

// 	httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to create request: %w", err)
// 	}

// 	httpReq.Header.Set("Accept", "application/json")
// 	httpReq.Header.Set("Content-Type", "application/json")

// 	client := &http.Client{}
// 	resp, err := client.Do(httpReq)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to send request: %w", err)
// 	}

// 	return resp, nil
// }

// // The provider used in the registry.
// const provider = "graphrag"

// // Field names for schema.
// const (
// 	idField       = "id"
// 	textField     = "text"
// 	metadataField = "metadata"
// )

// // GraphKnowledge holds configuration for the plugin.
// type GraphKnowledge struct {
// 	Addr string // Knowledge server address (host:port, e.g., "54.92.111.204:5670").

// 	client  *Client    // Knowledge client.
// 	mu      sync.Mutex // Mutex to control access.
// 	initted bool       // Whether the plugin has been initialized.
// }

// // Name returns the plugin name.
// func (k *GraphKnowledge) Name() string {
// 	return provider
// }

// // Init initializes the GraphKnowledge plugin.
// func (k *GraphKnowledge) Init(ctx context.Context, g *genkit.Genkit) (err error) {
// 	if k == nil {
// 		k = &GraphKnowledge{}
// 	}

// 	k.mu.Lock()
// 	defer k.mu.Unlock()
// 	defer func() {
// 		if err != nil {
// 			err = fmt.Errorf("graphrag.Init: %w", err)
// 		}
// 	}()

// 	if k.initted {
// 		return errors.New("plugin already initialized")
// 	}

// 	// Load configuration.
// 	addr := k.Addr
// 	if addr == "" {
// 		addr = "54.92.111.204:5670" // Default address.
// 	}

// 	// Initialize Knowledge client.
// 	host, port := parseAddr(addr)
// 	client := NewClient(host, port)
// 	k.client = client
// 	k.initted = true
// 	return nil
// }

// // parseAddr splits host:port into host and port.
// func parseAddr(addr string) (string, int) {
// 	parts := strings.Split(addr, ":")
// 	if len(parts) != 2 {
// 		return "54.92.111.204", 5670
// 	}
// 	port, _ := strconv.Atoi(parts[1])
// 	return parts[0], port
// }

// // DefineIndexerAndRetriever defines an Indexer and Retriever for a Knowledge space.
// func DefineIndexerAndRetriever(ctx context.Context, g *genkit.Genkit) (ai.Indexer, ai.Retriever, error) {

// 	spaceID := ""
// 	modelName := "Qwen/Qwen2.5-Coder-32B-Instruct"

// 	k := genkit.LookupPlugin(g, provider)
// 	if k == nil {
// 		return nil, nil, errors.New("graphrag plugin not found; did you call genkit.Init with the graphrag plugin?")
// 	}
// 	knowledge := k.(*GraphKnowledge)

// 	ds, err := knowledge.newDocStore(ctx, spaceID, modelName)
// 	if err != nil {
// 		return nil, nil, err
// 	}

// 	indexer := genkit.DefineIndexer(g, provider, spaceID, ds.Index)
// 	retriever := genkit.DefineRetriever(g, provider, spaceID, ds.Retrieve)
// 	return indexer, retriever, nil
// }

// // docStore defines an Indexer and a Retriever.
// type docStore struct {
// 	client    *Client
// 	spaceID   string
// 	modelName string
// }

// // newDocStore creates a docStore.
// func (k *GraphKnowledge) newDocStore(ctx context.Context, spaceID, modelName string) (*docStore, error) {
// 	if k.client == nil {
// 		return nil, errors.New("graphrag.Init not called")
// 	}

// 	return &docStore{
// 		client:    k.client,
// 		spaceID:   spaceID,
// 		modelName: modelName,
// 	}, nil
// }

// // Indexer returns the indexer for a space.
// func Indexer(g *genkit.Genkit, spaceID string) ai.Indexer {
// 	return genkit.LookupIndexer(g, provider, spaceID)
// }

// // Retriever returns the retriever for a space.
// func Retriever(g *genkit.Genkit, spaceID string) ai.Retriever {
// 	return genkit.LookupRetriever(g, provider, spaceID)
// }

// // generateRandomDocName generates a random alphanumeric string of the specified length.
// func GenerateRandomDocName(length int) (string, error) {
// 	const charset = "abcdefghijklmnopqrstuvwxyz0123456789"
// 	var result strings.Builder
// 	result.Grow(length)

// 	for i := 0; i < length; i++ {
// 		idx, err := rand.Int(rand.Reader, big.NewInt(int64(len(charset))))
// 		if err != nil {
// 			return "", fmt.Errorf("failed to generate random index: %w", err)
// 		}
// 		result.WriteByte(charset[idx.Int64()])
// 	}

// 	return result.String(), nil
// }

// // ParseJSONResponse parses a JSON byte slice and extracts the success boolean and data fields as a string.
// func ParseJSONResponse(jsonBytes []byte) (success bool, data string, err error) {
// 	// Define struct to capture only the needed fields
// 	type jsonResponse struct {
// 		Success bool `json:"success"`
// 		Data    int  `json:"data"` // Use string to capture JSON string data
// 	}

// 	var resp jsonResponse
// 	if err := json.Unmarshal(jsonBytes, &resp); err != nil {
// 		return false, "", fmt.Errorf("failed to unmarshal JSON: %w", err)
// 	}

// 	return resp.Success, fmt.Sprintf("%d", resp.Data), nil
// }

// type IndexReqOption struct {
// 	UserId   string
// 	UserName string
// }

// const DocNameKey = "doc_name"

// // Index implements the Indexer.Index method.
// func (ds *docStore) Index(ctx context.Context, req *ai.IndexerRequest) error {
// 	if len(req.Documents) == 0 {
// 		return nil
// 	}

// 	// Type-assert req.Options to IndexReqOption
// 	opt, ok := req.Options.(*IndexReqOption)
// 	if !ok {
// 		return fmt.Errorf("invalid options type: got %T, want *IndexReqOption", req.Options)
// 	}

// 	// Validate required fields
// 	if opt.UserId == "" {
// 		return fmt.Errorf("UserId is required in IndexReqOption")
// 	}
// 	if opt.UserName == "" {
// 		return fmt.Errorf("UserName is required in IndexReqOption")
// 	}

// 	// Create knowledge space
// 	spaceReq := SpaceRequest{
// 		Name:       opt.UserId,
// 		VectorType: "KnowledgeGraph",
// 		DomainType: "Normal",
// 		Desc:       opt.UserName,
// 		Owner:      opt.UserId,
// 	}
// 	resp, err := ds.client.AddSpace(spaceReq)
// 	if err != nil {
// 		return fmt.Errorf("add space: %w", err)
// 	}
// 	defer resp.Body.Close()
// 	if resp.StatusCode != http.StatusOK {
// 		body, _ := io.ReadAll(resp.Body)
// 		return fmt.Errorf("add space failed with status %d: %s", resp.StatusCode, string(body))
// 	}

// 	fmt.Println("space ok")

// 	spaceId := opt.UserId

// 	// Index each document
// 	for i, doc := range req.Documents {
// 		// Use DocName from options, fall back to random name if empty
// 		docName := ""
// 		if v, ok := doc.Metadata[DocNameKey]; ok {
// 			if str, isString := v.(string); isString {
// 				docName = str
// 			} else {
// 				return fmt.Errorf("must provide doc_name str value in metadata")
// 			}
// 		} else {
// 			return fmt.Errorf("must provide doc_name key  in metadata")
// 		}

// 		fmt.Println("docName:   ", docName)

// 		// Add document
// 		var sb strings.Builder
// 		for _, p := range doc.Content {
// 			if p.IsText() {
// 				sb.WriteString(p.Text)
// 			}
// 		}
// 		text := sb.String()
// 		fmt.Println("text:   ", text)
// 		docReq := DocumentRequest{
// 			DocName:  docName,
// 			Source:   "api",
// 			DocType:  "TEXT",
// 			Content:  text,
// 			Labels:   "",
// 			Metadata: doc.Metadata,
// 		}
// 		resp, err := ds.client.AddDocument(spaceId, docReq)
// 		if err != nil {
// 			return fmt.Errorf("add document %d: %w", i+1, err)
// 		}
// 		body, err := io.ReadAll(resp.Body)
// 		if err != nil {
// 			resp.Body.Close()
// 			return fmt.Errorf("read add document response %d: %w", i+1, err)
// 		}
// 		defer resp.Body.Close()

// 		if resp.StatusCode != http.StatusOK {
// 			return fmt.Errorf("add document %d failed with status %d: %s", i+1, resp.StatusCode, string(body))
// 		}

// 		// Parse AddDocument response
// 		ok, idx, err := ParseJSONResponse(body)
// 		if err != nil {
// 			return fmt.Errorf("parse add document response %d: %w", i+1, err)
// 		}
// 		if !ok {
// 			return fmt.Errorf("add document %d failed: response success=false, data=%s", i+1, idx)
// 		}
// 		fmt.Println("document ok", string(body), idx)

// 		// Sync document
// 		_, err = ds.client.SyncDocuments(spaceId, []string{idx})
// 		if err != nil {
// 			return fmt.Errorf("sync document %d: %w", i+1, err)
// 		}
// 	}

// 	return nil
// }

// // type IndexReqOption struct{
// // 	UserId string
// // 	UserName string
// // 	DocName string
// // }

// // // Index implements the Indexer.Index method.
// // func (ds *docStore) Index(ctx context.Context, req *ai.IndexerRequest) error {
// // 	if len(req.Documents) == 0 {
// // 		return nil
// // 	}

// // 	req.Options

// // 	userid := ""
// // 	usernmae := ""
// // 	for _, doc := range req.Documents {
// // 		if v, ok := doc.Metadata["user_id"]; ok {
// // 			if str, isString := v.(string); isString {
// // 				userid = str
// // 			}
// // 		}
// // 		if v, ok := doc.Metadata["username"]; ok {
// // 			if str, isString := v.(string); isString {
// // 				usernmae = str
// // 			}
// // 		}
// // 	}

// // 	// Create knowledge space.
// // 	spaceReq := SpaceRequest{
// // 		Name:       userid,
// // 		VectorType: "KnowledgeGraph",
// // 		DomainType: "Normal",
// // 		Desc:       usernmae,
// // 		Owner:      userid,
// // 	}
// // 	resp, err := ds.client.AddSpace(spaceReq)
// // 	if err != nil {
// // 		return fmt.Errorf("add space: %w", err)
// // 	}
// // 	defer resp.Body.Close()
// // 	if resp.StatusCode != http.StatusOK {
// // 		body, _ := io.ReadAll(resp.Body)
// // 		return fmt.Errorf("add space failed with status %d: %s", resp.StatusCode, string(body))
// // 	}

// // 	fmt.Println("space ok")

// // 	spaceId := userid

// // 	// Index each document.
// // 	for i, doc := range req.Documents {

// // 		docName := ""
// // 		if v, ok := doc.Metadata["doc_name"]; ok {
// // 			if str, isString := v.(string); isString {
// // 				docName = str
// // 			} else {

// // 				// Generate random docName.
// // 				var err error
// // 				docName, err = generateRandomDocName(8)
// // 				if err != nil {
// // 					return fmt.Errorf("generate random docName for document %d: %w", i+1, err)
// // 				}
// // 			}
// // 		} else {
// // 			// Generate random docName.
// // 			var err error
// // 			docName, err = generateRandomDocName(8)
// // 			if err != nil {
// // 				return fmt.Errorf("generate random docName for document %d: %w", i+1, err)
// // 			}

// // 		}

// // 		fmt.Println("docName:   ", docName)

// // 		// Add document.
// // 		var sb strings.Builder
// // 		for _, p := range doc.Content {
// // 			if p.IsText() {
// // 				sb.WriteString(p.Text)
// // 			}
// // 		}
// // 		text := sb.String()
// // 		fmt.Println("text:   ",text)
// // 		docReq := DocumentRequest{
// // 			DocName: docName,
// // 			Source:  "api",
// // 			DocType: "TEXT",
// // 			Content: text,
// // 			Labels:  "",
// // 			// Questions: []string{},
// // 			Metadata: doc.Metadata,
// // 		}
// // 		resp, err := ds.client.AddDocument(spaceId, docReq)
// // 		if err != nil {
// // 			return fmt.Errorf("add document %d: %w", i+1, err)
// // 		}
// // 		body, _ := io.ReadAll(resp.Body)

// // 		defer resp.Body.Close()
// // 		if resp.StatusCode != http.StatusOK {

// // 			return fmt.Errorf("add document %d failed with status %d: %s", i+1, resp.StatusCode, string(body))
// // 		}

// // 		ok, idx, err := ParseJSONResponse(body)
// // 		if err != nil {
// // 			return fmt.Errorf("ParseJSONResponse %d: %w", i+1, err)
// // 		}

// // 		if !ok{
// // 			return fmt.Errorf("ParseJSONResponse body  %d: %w", i+1, err)
// // 		}
// // 		fmt.Println("document ok",string(body),idx)

// // 		ok ,err =ds.client.SyncDocuments(spaceId,[]string{idx})

// // 		if err != nil{
// // 			return err
// // 		}
// // 	}

// // 	return nil
// // }

// // // RetrieverOptions for Knowledge retrieval.
// // type RetrieverOptions struct {
// // 	Count      int    `json:"count,omitempty"`       // Max documents to retrieve.
// // 	MetricType string `json:"metric_type,omitempty"` // Similarity metric (e.g., "L2", "IP").
// // }

// // // Assuming ai.Part has a Text() method or Text field to get the string content
// // func partsToString(parts []*ai.Part) string {
// //     var texts []string
// //     for _, part := range parts {
// //         // Adjust this based on the actual ai.Part structure
// //         // If ai.Part has a Text() method:
// //         texts = append(texts, part.Text)
// //         // OR if ai.Part has a Text field:
// //         // texts = append(texts, part.Text)
// //     }
// //     return strings.Join(texts, " ")
// // }

// // // Retrieve implements the Retriever.Retrieve method.
// // func (ds *docStore) Retrieve(ctx context.Context, req *ai.RetrieverRequest) (*ai.RetrieverResponse, error) {
// // 	// count := 3
// // 	// metricTypeStr := "L2"
// // 	// if req.Options != nil {
// // 	//     ropt, ok := req.Options.(*RetrieverOptions)
// // 	//     if !ok {
// // 	//         return nil, fmt.Errorf("graphrag.Retrieve options have type %T, want %T", req.Options, &RetrieverOptions{})
// // 	//     }
// // 	//     if ropt.Count > 0 {
// // 	//         count = ropt.Count
// // 	//     }
// // 	//     if ropt.MetricType != "" {
// // 	//         metricTypeStr = ropt.MetricType
// // 	//     }
// // 	// }

// // 	queryContent := partsToString(req.Query.Content)
// // 	// Format query for retrieval.
// // 	queryText := fmt.Sprintf("Search for: %s", queryContent)

// // 	username := "Alice" // Default, override if metadata available.
// // 	if req.Query.Metadata != nil {
// // 		if uname, ok := req.Query.Metadata["username"].(string); ok {
// // 			username = uname
// // 		}
// // 	}

// // 	// Prepare request for chat completions endpoint.
// // 	url := fmt.Sprintf("%s/api/v1/chat/completions", ds.client.BaseURL)
// // 	chatReq := struct {
// // 		ConvUID      string                 `json:"conv_uid"`
// // 		UserInput    string                 `json:"user_input"`
// // 		UserName     string                 `json:"user_name"`
// // 		ChatMode     string                 `json:"chat_mode"`
// // 		AppCode      string                 `json:"app_code"`
// // 		Temperature  float32                `json:"temperature"`
// // 		MaxNewTokens int                    `json:"max_new_tokens"`
// // 		SelectParam  string                 `json:"select_param"`
// // 		ModelName    string                 `json:"model_name"`
// // 		Incremental  bool                   `json:"incremental"`
// // 		SysCode      string                 `json:"sys_code"`
// // 		PromptCode   string                 `json:"prompt_code"`
// // 		ExtInfo      map[string]interface{} `json:"ext_info"`
// // 	}{
// // 		ConvUID:      "",
// // 		UserInput:    queryText,
// // 		UserName:     username,
// // 		ChatMode:     "",
// // 		AppCode:      "",
// // 		Temperature:  0.5,
// // 		MaxNewTokens: 4000,
// // 		SelectParam:  "",
// // 		ModelName:    ds.modelName,
// // 		Incremental:  false,
// // 		SysCode:      "",
// // 		PromptCode:   "",
// // 		ExtInfo: map[string]interface{}{
// // 			"space_id": ds.spaceID,
// // 			//"k":        count,
// // 		},
// // 	}

// // 	body, err := json.Marshal(chatReq)
// // 	if err != nil {
// // 		return nil, fmt.Errorf("marshal chat request: %w", err)
// // 	}

// // 	httpReq, err := http.NewRequestWithContext(ctx, "POST", url, bytes.NewBuffer(body))
// // 	if err != nil {
// // 		return nil, fmt.Errorf("create chat request: %w", err)
// // 	}
// // 	httpReq.Header.Set("Accept", "application/json")
// // 	httpReq.Header.Set("Content-Type", "application/json")

// // 	client := &http.Client{}
// // 	resp, err := client.Do(httpReq)
// // 	if err != nil {
// // 		return nil, fmt.Errorf("send chat request: %w", err)
// // 	}
// // 	defer resp.Body.Close()

// // 	if resp.StatusCode != http.StatusOK {
// // 		body, _ := io.ReadAll(resp.Body)
// // 		return nil, fmt.Errorf("chat completion failed with status %d: %s", resp.StatusCode, string(body))
// // 	}

// // 	// Parse response
// // 	var chatResp struct {
// // 		Success bool `json:"success"`
// // 		Data    struct {
// // 			Answer []struct {
// // 				Content  string                 `json:"content"`
// // 				DocID    int                    `json:"doc_id"`
// // 				Score    float64                `json:"score"`
// // 				Metadata map[string]interface{} `json:"metadata_map"`
// // 			} `json:"answer"`
// // 		} `json:"data"`
// // 	}
// // 	if err := json.NewDecoder(resp.Body).Decode(&chatResp); err != nil {
// // 		return nil, fmt.Errorf("decode chat response: %w", err)
// // 	}

// // 	var docs []*ai.Document
// // 	for _, doc := range chatResp.Data.Answer {
// // 		metadata := doc.Metadata
// // 		if metadata == nil {
// // 			metadata = make(map[string]interface{})
// // 		}
// // 		// Ensure metadata includes user_id and username.
// // 		if _, ok := metadata["user_id"]; !ok {
// // 			metadata["user_id"] = "user123"
// // 		}
// // 		if _, ok := metadata["username"]; !ok {
// // 			metadata["username"] = username
// // 		}
// // 		aiDoc := ai.DocumentFromText(doc.Content, metadata)
// // 		docs = append(docs, aiDoc)
// // 	}

// // 	return &ai.RetrieverResponse{
// // 		Documents: docs,
// // 	}, nil
// // }

// // ChatRequest 定义请求结构体，匹配单元测试的 curl 请求
// type ChatRequest struct {
// 	Model             string  `json:"model"`
// 	Messages          string  `json:"messages"`
// 	Temperature       float64 `json:"temperature"`
// 	TopP              float64 `json:"top_p"`
// 	TopK              int     `json:"top_k"`
// 	N                 int     `json:"n"`
// 	MaxTokens         int     `json:"max_tokens"`
// 	Stream            bool    `json:"stream"`
// 	RepetitionPenalty float64 `json:"repetition_penalty"`
// 	FrequencyPenalty  float64 `json:"frequency_penalty"`
// 	PresencePenalty   float64 `json:"presence_penalty"`
// 	ChatMode          string  `json:"chat_mode"`
// 	ChatParam         string  `json:"chat_param"`
// 	EnableVis         bool    `json:"enable_vis"`
// }

// // ChatResponse 定义响应结构体，匹配单元测试的 API 响应
// type ChatResponse struct {
// 	ID      string `json:"id"`
// 	Object  string `json:"object"`
// 	Created int64  `json:"created"`
// 	Model   string `json:"model"`
// 	Choices []struct {
// 		Index   int `json:"index"`
// 		Message struct {
// 			Role             string      `json:"role"`
// 			Content          string      `json:"content"`
// 			ReasoningContent interface{} `json:"reasoning_content"`
// 		} `json:"message"`
// 		FinishReason interface{} `json:"finish_reason"`
// 	} `json:"choices"`
// 	Usage struct {
// 		PromptTokens     int `json:"prompt_tokens"`
// 		TotalTokens      int `json:"total_tokens"`
// 		CompletionTokens int `json:"completion_tokens"`
// 	} `json:"usage"`
// }

// // Assuming ai.Part has a Text() method or Text field to get the string content
// func partsToString(parts []*ai.Part) string {
// 	var texts []string
// 	for _, part := range parts {
// 		// Adjust this based on the actual ai.Part structure
// 		// If ai.Part has a Text() method:
// 		texts = append(texts, part.Text)
// 		// OR if ai.Part has a Text field:
// 		// texts = append(texts, part.Text)
// 	}
// 	return strings.Join(texts, " ")
// }

// // Retrieve implements the Retriever.Retrieve method.
// func (ds *docStore) Retrieve(ctx context.Context, req *ai.RetrieverRequest) (*ai.RetrieverResponse, error) {

// 	// Format query for retrieval.
// 	queryContent := partsToString(req.Query.Content)
// 	queryText := fmt.Sprintf("Search for: %s", queryContent)

// 	if req.Query.Metadata == nil {
// 		// If ok, we don't use the User struct since the requirement is to error on non-nil
// 		return nil, fmt.Errorf("req.Query.Metadata  must be not nil, got type %T", req.Options)
// 	}

// 	for k, v := range req.Query.Metadata {
// 		fmt.Println("k", k, "v", v)
// 	}

// 	// Extract username and user_id from req.Query.Metadata
// 	userName, ok := req.Query.Metadata[util.UserNameKey].(string)
// 	if !ok {
// 		return nil, fmt.Errorf("req.Query.Metadata  must provide  username key")
// 	}
// 	userId, ok := req.Query.Metadata[util.UserIdKey].(string)
// 	if !ok {
// 		return nil, fmt.Errorf("req.Query.Metadata  must provide  user_id key")
// 	}

// 	// Prepare request for chat completions endpoint.
// 	url := fmt.Sprintf("%s/api/v2/chat/completions", ds.client.BaseURL)
// 	chatReq := ChatRequest{
// 		Model:             ds.modelName,
// 		Messages:          queryText,
// 		Temperature:       0.7,
// 		TopP:              1,
// 		TopK:              -1,
// 		N:                 1,
// 		MaxTokens:         0,
// 		Stream:            false,
// 		RepetitionPenalty: 1,
// 		FrequencyPenalty:  0,
// 		PresencePenalty:   0,
// 		ChatMode:          "chat_knowledge",
// 		ChatParam:         userId,
// 		EnableVis:         true,
// 	}

// 	body, err := json.Marshal(chatReq)
// 	if err != nil {
// 		return nil, fmt.Errorf("marshal chat request: %w", err)
// 	}

// 	httpReq, err := http.NewRequestWithContext(ctx, "POST", url, bytes.NewBuffer(body))
// 	if err != nil {
// 		return nil, fmt.Errorf("create chat request: %w", err)
// 	}
// 	httpReq.Header.Set("Accept", "application/json")
// 	httpReq.Header.Set("Content-Type", "application/json")

// 	client := &http.Client{}
// 	resp, err := client.Do(httpReq)
// 	if err != nil {
// 		return nil, fmt.Errorf("send chat request: %w", err)
// 	}
// 	defer resp.Body.Close()

// 	if resp.StatusCode != http.StatusOK {
// 		body, _ := io.ReadAll(resp.Body)
// 		return nil, fmt.Errorf("chat completion failed with status %d: %s", resp.StatusCode, string(body))
// 	}

// 	// Parse response
// 	var chatResp ChatResponse
// 	if err := json.NewDecoder(resp.Body).Decode(&chatResp); err != nil {
// 		body, _ := io.ReadAll(resp.Body)
// 		return nil, fmt.Errorf("decode chat response: %w, raw response: %s", err, string(body))
// 	}

// 	// Convert response to ai.Document
// 	var docs []*ai.Document
// 	if len(chatResp.Choices) > 0 {
// 		content := chatResp.Choices[0].Message.Content
// 		metadata := map[string]interface{}{
// 			util.UserIdKey:   userId,
// 			util.UserNameKey: userName,
// 		}
// 		aiDoc := ai.DocumentFromText(content, metadata)
// 		docs = append(docs, aiDoc)
// 	}

// 	return &ai.RetrieverResponse{
// 		Documents: docs,
// 	}, nil
// }
