// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

// Package milvus provides a Genkit plugin for Milvus vector database using milvus-sdk-go.
package milvus

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"os"
	"strings"
	"sync"

	"github.com/firebase/genkit/go/ai"
	"github.com/firebase/genkit/go/genkit"
	"github.com/milvus-io/milvus-sdk-go/v2/client"
	"github.com/milvus-io/milvus-sdk-go/v2/entity"
	"github.com/rs/zerolog/log"
	"github.com/wade-liwei/agentchat/util"
)

// The provider used in the registry.
const provider = "milvus"

// Field names for Milvus schema.
const (
    idField       = "id"
    vectorField   = "vector"
    textField     = "text"
    metadataField = "metadata"
)

// Milvus holds configuration for the plugin.
type Milvus struct {
    // Milvus server address (host:port, e.g., "localhost:19530").
    // Defaults to MILVUS_ADDRESS environment variable.
    Addr string
    // Username for authentication.
    // Defaults to MILVUS_USERNAME.
    Username string
    // Password for authentication.
    // Defaults to MILVUS_PASSWORD.
    Password string
    // Token for authentication (alternative to username/password).
    // Defaults to MILVUS_TOKEN.
    Token string

    client  client.Client // Milvus client.
    mu      sync.Mutex    // Mutex to control access.
    initted bool          // Whether the plugin has been initialized.
}

// Name returns the plugin name.
func (m *Milvus) Name() string {
    return provider
}

// Init initializes the Milvus plugin.
func (m *Milvus) Init(ctx context.Context, g *genkit.Genkit) (err error) {
    log.Info().Str("method", "Milvus.Init").Msg("Initializing Milvus plugin")
    if m == nil {
        m = &Milvus{}
    }

    m.mu.Lock()
    defer m.mu.Unlock()
    defer func() {
        if err != nil {
            log.Error().Err(err).Str("method", "Milvus.Init").Msg("Initialization failed")
            err = fmt.Errorf("milvus.Init: %w", err)
        } else {
            log.Info().Str("method", "Milvus.Init").Msg("Initialization successful")
        }
    }()

    if m.initted {
        return errors.New("plugin already initialized")
    }

    // Load configuration.
    addr := m.Addr
    if addr == "" {
        addr = os.Getenv("MILVUS_ADDRESS")
    }
    if addr == "" {
        return errors.New("milvus address required")
    }

    username := m.Username
    if username == "" {
        username = os.Getenv("MILVUS_USERNAME")
    }
    password := m.Password
    if password == "" {
        password = os.Getenv("MILVUS_PASSWORD")
    }
    token := m.Token
    if token == "" {
        token = os.Getenv("MILVUS_TOKEN")
    }

    // Initialize Milvus client.
    config := client.Config{
        Address:  addr,
        Username: username,
        Password: password,
        APIKey:   token,
    }
    client, err := client.NewClient(ctx, config)
    if err != nil {
        return fmt.Errorf("failed to initialize Milvus client: %v", err)
    }

    m.client = client
    m.initted = true
    return nil
}

// CollectionConfig holds configuration for an indexer/retriever pair.
type CollectionConfig struct {
    // Milvus collection name. Must not be empty.
    Collection string
    // Embedding vector dimension (e.g., 1536 for text-embedding-ada-002).
    Dimension int
    // Embedder for generating vectors.
    Embedder ai.Embedder
    // Embedder options.
    EmbedderOptions any
}

// DefineIndexerAndRetriever defines an Indexer and Retriever for a Milvus collection.
func DefineIndexerAndRetriever(ctx context.Context, g *genkit.Genkit, cfg CollectionConfig) (ai.Indexer, ai.Retriever, error) {
    log.Info().
        Str("method", "DefineIndexerAndRetriever").
        Str("collection", cfg.Collection).
        Int("dimension", cfg.Dimension).
        Msg("Defining indexer and retriever")

    if cfg.Embedder == nil {
        log.Error().Str("method", "DefineIndexerAndRetriever").Msg("Embedder required")
        return nil, nil, errors.New("milvus: Embedder required")
    }
    if cfg.Collection == "" {
        log.Error().Str("method", "DefineIndexerAndRetriever").Msg("Collection name required")
        return nil, nil, errors.New("milvus: collection name required")
    }
    if cfg.Dimension <= 0 {
        log.Error().Str("method", "DefineIndexerAndRetriever").Int("dimension", cfg.Dimension).Msg("Dimension must be positive")
        return nil, nil, errors.New("milvus: dimension must be positive")
    }

    m := genkit.LookupPlugin(g, provider)
    if m == nil {
        log.Error().Str("method", "DefineIndexerAndRetriever").Msg("Milvus plugin not found")
        return nil, nil, errors.New("milvus plugin not found; did you call genkit.Init with the milvus plugin?")
    }
    milvus := m.(*Milvus)

    ds, err := milvus.newDocStore(ctx, &cfg)
    if err != nil {
        log.Error().Err(err).Str("method", "DefineIndexerAndRetriever").Str("collection", cfg.Collection).Msg("Failed to create doc store")
        return nil, nil, err
    }

    indexer := genkit.DefineIndexer(g, provider, cfg.Collection, ds.Index)
    retriever := genkit.DefineRetriever(g, provider, cfg.Collection, ds.Retrieve)
    log.Info().Str("method", "DefineIndexerAndRetriever").Str("collection", cfg.Collection).Msg("Indexer and retriever defined successfully")
    return indexer, retriever, nil
}

// docStore defines an Indexer and a Retriever.
type docStore struct {
    client          client.Client
    collection      string
    dimension       int
    embedder        ai.Embedder
    embedderOptions map[string]interface{}
}

// newDocStore creates a docStore.
func (m *Milvus) newDocStore(ctx context.Context, cfg *CollectionConfig) (*docStore, error) {
    log.Info().
        Str("method", "Milvus.newDocStore").
        Str("collection", cfg.Collection).
        Int("dimension", cfg.Dimension).
        Msg("Creating new doc store")

    if m.client == nil {
        log.Error().Str("method", "Milvus.newDocStore").Msg("Milvus client not initialized")
        return nil, errors.New("milvus.Init not called")
    }

    // Check/create collection.
    exists, err := m.client.HasCollection(ctx, cfg.Collection)
    if err != nil {
        log.Error().Err(err).Str("method", "Milvus.newDocStore").Str("collection", cfg.Collection).Msg("Failed to check collection")
        return nil, fmt.Errorf("failed to check collection %q: %v", cfg.Collection, err)
    }
    if !exists {
        // Define schema with textField as primary key, plus user_id and username fields.
        schema := &entity.Schema{
            CollectionName: cfg.Collection,
            Fields: []*entity.Field{
                {
                    Name:     vectorField,
                    DataType: entity.FieldTypeFloatVector,
                    TypeParams: map[string]string{
                        "dim": fmt.Sprintf("%d", cfg.Dimension),
                    },
                },
                {
                    Name:       textField,
                    DataType:   entity.FieldTypeVarChar,
                    PrimaryKey: true,
                    TypeParams: map[string]string{
                        "max_length": "65535",
                    },
                },
                {
                    Name:     metadataField,
                    DataType: entity.FieldTypeJSON,
                },
                {
                    Name:     "user_id",
                    DataType: entity.FieldTypeVarChar,
                    TypeParams: map[string]string{
                        "max_length": "128",
                    },
                },
                {
                    Name:     "username",
                    DataType: entity.FieldTypeVarChar,
                    TypeParams: map[string]string{
                        "max_length": "128",
                    },
                },
            },
        }

        err = m.client.CreateCollection(ctx, schema, entity.DefaultShardNumber)
        if err != nil {
            log.Error().Err(err).Str("method", "Milvus.newDocStore").Str("collection", cfg.Collection).Msg("Failed to create collection")
            return nil, fmt.Errorf("failed to create collection %q: %v", cfg.Collection, err)
        }

        // Create HNSW index for vectorField.
        index, err := entity.NewIndexHNSW(
            entity.L2,
            8,  // M
            96, // efConstruction
        )
        if err != nil {
            log.Error().Err(err).Str("method", "Milvus.newDocStore").Str("collection", cfg.Collection).Msg("Failed to create HNSW index")
            return nil, fmt.Errorf("entity.NewIndexHNSW: %v", err)
        }

        err = m.client.CreateIndex(ctx, cfg.Collection, vectorField, index, false)
        if err != nil {
            log.Error().Str("method", "Milvus.newDocStore").Str("collection", cfg.Collection).Msgf("Failed to create index: %s",err.Error())
            return nil, fmt.Errorf("failed to create index: %v", err)
        }
    }

    // Load collection.
    err = m.client.LoadCollection(ctx, cfg.Collection, false)
    if err != nil {
        log.Error().Err(err).Str("method", "Milvus.newDocStore").Str("collection", cfg.Collection).Msg("Failed to load collection")
        return nil, fmt.Errorf("failed to load collection %q: %v", cfg.Collection, err)
    }

    // Convert EmbedderOptions to map[string]interface{}.
    var embedderOptions map[string]interface{}
    if cfg.EmbedderOptions != nil {
        opts, ok := cfg.EmbedderOptions.(map[string]interface{})
        if !ok {
            log.Error().
                Str("method", "Milvus.newDocStore").
                Str("type", fmt.Sprintf("%T", cfg.EmbedderOptions)).
                Msg("EmbedderOptions must be a map[string]interface{}")
            return nil, fmt.Errorf("EmbedderOptions must be a map[string]interface{}, got %T", cfg.EmbedderOptions)
        }
        embedderOptions = opts
    } else {
        embedderOptions = make(map[string]interface{})
    }

    log.Info().Str("method", "Milvus.newDocStore").Str("collection", cfg.Collection).Msg("Doc store created successfully")
    return &docStore{
        client:          m.client,
        collection:      cfg.Collection,
        dimension:       cfg.Dimension,
        embedder:        cfg.Embedder,
        embedderOptions: embedderOptions,
    }, nil
}

// Indexer returns the indexer for a collection.
func Indexer(g *genkit.Genkit, collection string) ai.Indexer {
    log.Info().Str("method", "Indexer").Str("collection", collection).Msg("Looking up indexer")
    indexer := genkit.LookupIndexer(g, provider, collection)
    if indexer == nil {
        log.Warn().Str("method", "Indexer").Str("collection", collection).Msg("Indexer not found")
    }
    return indexer
}

// Retriever returns the retriever for a collection.
func Retriever(g *genkit.Genkit, collection string) ai.Retriever {
    log.Info().Str("method", "Retriever").Str("collection", collection).Msg("Looking up retriever")
    retriever := genkit.LookupRetriever(g, provider, collection)
    if retriever == nil {
        log.Warn().Str("method", "Retriever").Str("collection", collection).Msg("Retriever not found")
    }
    return retriever
}

// Index implements the Indexer.Index method.
func (ds *docStore) Index(ctx context.Context, req *ai.IndexerRequest) error {
    log.Info().
        Str("method", "docStore.Index").
        Str("collection", ds.collection).
        Int("documents", len(req.Documents)).
        Msg("Starting index operation")

    if len(req.Documents) == 0 {
        log.Debug().Str("method", "docStore.Index").Str("collection", ds.collection).Msg("No documents to index")
        return nil
    }

    // Embed documents.
    ereq := &ai.EmbedRequest{
        Input:   req.Documents,
        Options: ds.embedderOptions,
    }
    eres, err := ds.embedder.Embed(ctx, ereq)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Index").Str("collection", ds.collection).Msg("Embedding failed")
        return fmt.Errorf("milvus index embedding failed: %w", err)
    }

    // Validate embedding count matches document count.
    if len(eres.Embeddings) != len(req.Documents) {
        log.Error().
            Str("method", "docStore.Index").
            Str("collection", ds.collection).
            Int("embeddings", len(eres.Embeddings)).
            Int("documents", len(req.Documents)).
            Msg("Mismatch in embedding and document count")
        return fmt.Errorf("mismatch: got %d embeddings for %d documents", len(eres.Embeddings), len(req.Documents))
    }

    // Prepare row-based data.
    var rows []interface{}
    for i, emb := range eres.Embeddings {
        doc := req.Documents[i]

        if doc.Metadata == nil {
            log.Error().Str("method", "docStore.Index").Int("index", i).Msg("Document metadata is nil")
            return fmt.Errorf("req.Query.Metadata must be not nil, got type %T", doc.Metadata)
        }

        // Extract username and user_id from req.Query.Metadata
        userName, ok := doc.Metadata[util.UserNameKey].(string)
        if !ok {
            log.Error().Str("method", "docStore.Index").Int("index", i).Msg("Missing username in metadata")
            return fmt.Errorf("req.Query.Metadata must provide username key")
        }
        userId, ok := doc.Metadata[util.UserIdKey].(string)
        if !ok {
            log.Error().Str("method", "docStore.Index").Int("index", i).Msg("Missing user_id in metadata")
            return fmt.Errorf("req.Query.Metadata must provide user_id key")
        }

        var sb strings.Builder
        for _, p := range doc.Content {
            if p.IsText() {
                sb.WriteString(p.Text)
            }
        }
        text := sb.String()
        metadata := doc.Metadata
        if metadata == nil {
            metadata = make(map[string]interface{})
        }

        // Create row with explicit metadata field.
        row := make(map[string]interface{})
        row["vector"] = emb.Embedding
        row["text"] = text
        row["user_id"] = userId
        row["username"] = userName
        row["metadata"] = metadata
        rows = append(rows, row)

        log.Debug().
            Str("method", "docStore.Index").
            Int("index", i).
            Str("collection", ds.collection).
            Int("vector_length", len(emb.Embedding)).
            Str("text", text).
            Str("user_id", userId).
            Str("username", userName).
            Interface("metadata", metadata).
            Msg("Prepared row for insertion")
    }

    log.Info().
        Str("method", "docStore.Index").
        Str("collection", ds.collection).
        Int("rows", len(rows)).
        Msg("Inserting rows into Milvus")

    // Insert rows into Milvus.
    _, err = ds.client.InsertRows(ctx, ds.collection, "", rows)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Index").Str("collection", ds.collection).Msg("Failed to insert rows")
        return fmt.Errorf("milvus insert rows failed: %w", err)
    }

    log.Info().Str("method", "docStore.Index").Str("collection", ds.collection).Int("rows", len(rows)).Msg("Index operation completed successfully")
    return nil
}

// RetrieverOptions for Milvus retrieval.
type RetrieverOptions struct {
    Count      int    `json:"count,omitempty"`       // Max documents to retrieve.
    MetricType string `json:"metric_type,omitempty"` // Similarity metric (e.g., "L2", "IP").
}

// Retrieve implements the Retriever.Retrieve method.
func (ds *docStore) Retrieve(ctx context.Context, req *ai.RetrieverRequest) (*ai.RetrieverResponse, error) {
    log.Info().
        Str("method", "docStore.Retrieve").
        Str("collection", ds.collection).
        Msg("Starting retrieve operation")

    if req.Query.Metadata == nil {
        log.Error().Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("Query metadata is nil")
        return nil, fmt.Errorf("req.Query.Metadata must be not nil, got type %T", req.Query.Metadata)
    }

    // Extract username and user_id from req.Query.Metadata
    userName, ok := req.Query.Metadata[util.UserNameKey].(string)
    if !ok {
        log.Error().Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("Missing username in metadata")
        return nil, fmt.Errorf("req.Query.Metadata must provide username key")
    }
    userId, ok := req.Query.Metadata[util.UserIdKey].(string)
    if !ok {
        log.Error().Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("Missing user_id in metadata")
        return nil, fmt.Errorf("req.Query.Metadata must provide user_id key")
    }

    count := 3 // Default.
    metricTypeStr := "L2"
    if req.Options != nil {
        ropt, ok := req.Options.(*RetrieverOptions)
        if !ok {
            log.Error().
                Str("method", "docStore.Retrieve").
                Str("collection", ds.collection).
                Str("options_type", fmt.Sprintf("%T", req.Options)).
                Msg("Invalid options type")
            return nil, fmt.Errorf("milvus.Retrieve options have type %T, want %T", req.Options, &RetrieverOptions{})
        }
        if ropt.Count > 0 {
            count = ropt.Count
        }
        if ropt.MetricType != "" {
            metricTypeStr = ropt.MetricType
        }
    }

    // Map string metric type to entity.MetricType.
    var metricType entity.MetricType
    switch metricTypeStr {
    case "L2":
        metricType = entity.L2
    case "IP":
        metricType = entity.IP
    default:
        log.Error().Str("method", "docStore.Retrieve").Str("metric_type", metricTypeStr).Msg("Unsupported metric type")
        return nil, fmt.Errorf("unsupported metric type: %s", metricTypeStr)
    }

    // Embed query.
    ereq := &ai.EmbedRequest{
        Input:   []*ai.Document{req.Query},
        Options: ds.embedderOptions,
    }
    eres, err := ds.embedder.Embed(ctx, ereq)
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("Embedding failed")
        return nil, fmt.Errorf("milvus retrieve embedding failed: %v", err)
    }
    if len(eres.Embeddings) == 0 {
        log.Error().Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("No embeddings generated")
        return nil, errors.New("no embeddings generated for query")
    }
    queryVector := entity.FloatVector(eres.Embeddings[0].Embedding)

    // Create search parameters.
    searchParams, err := entity.NewIndexHNSWSearchParam(64) // ef
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("Failed to create HNSW search parameters")
        return nil, fmt.Errorf("NewIndexHNSWSearchParam failed: %v", err)
    }

    // Define filter expression for user_id
    expr := fmt.Sprintf("user_id == %q", userId)

    log.Debug().
        Str("method", "docStore.Retrieve").
        Str("collection", ds.collection).
        Str("user_id", userId).
        Str("metric_type", metricTypeStr).
        Int("count", count).
        Msg("Performing vector search")

    // Perform vector search to get IDs, text, and metadata.
    results, err := ds.client.Search(
        ctx,
        ds.collection,
        []string{},
        expr,
        []string{textField, metadataField},
        []entity.Vector{queryVector},
        vectorField,
        metricType,
        count,
        searchParams,
    )
    if err != nil {
        log.Error().Err(err).Str("method", "docStore.Retrieve").Str("collection", ds.collection).Msg("Search failed")
        return nil, fmt.Errorf("milvus search failed: %v", err)
    }

    // Process search results.
    var docs []*ai.Document
    for _, result := range results {
        // Find text and metadata columns in search results.
        var textCol, metaCol entity.Column
        for _, col := range result.Fields {
            if col.Name() == textField {
                textCol = col
            }
            if col.Name() == metadataField {
                metaCol = col
            }
        }

        // Ensure text column exists.
        if textCol == nil {
            log.Error().
                Str("method", "docStore.Retrieve").
                Str("collection", ds.collection).
                Str("field", textField).
                Msg("Text column not found in search results")
            return nil, fmt.Errorf("text column %s not found in search results", textField)
        }

        // Iterate over rows.
        for i := 0; i < result.ResultCount; i++ {
            // Get text value.
            text, err := textCol.GetAsString(i)
            if err != nil {
                log.Error().
                    Err(err).
                    Str("method", "docStore.Retrieve").
                    Str("collection", ds.collection).
                    Int("index", i).
                    Msg("Failed to parse text")
                continue
            }

            // Get metadata value (optional).
            var metadata map[string]interface{}
            if metaCol != nil {
                metaStr, err := metaCol.GetAsString(i)
                if err == nil && metaStr != "" {
                    if err := json.Unmarshal([]byte(metaStr), &metadata); err != nil {
                        log.Error().
                            Err(err).
                            Str("method", "docStore.Retrieve").
                            Str("collection", ds.collection).
                            Int("index", i).
                            Msg("Failed to parse metadata")
                        continue
                    }
                } else if err != nil {
                    log.Error().
                        Err(err).
                        Str("method", "docStore.Retrieve").
                        Str("collection", ds.collection).
                        Int("index", i).
                        Msg("Failed to get metadata string")
                }
            }

            // Ensure metadata includes user_id and username from query.
            if metadata == nil {
                metadata = make(map[string]interface{})
            }
            metadata[util.UserIdKey] = userId
            metadata[util.UserNameKey] = userName

            // Create document.
            doc := ai.DocumentFromText(text, metadata)
            docs = append(docs, doc)

            log.Debug().
                Str("method", "docStore.Retrieve").
                Str("collection", ds.collection).
                Int("index", i).
                Str("text", text).
                Interface("metadata", metadata).
                Msg("Processed search result")
        }
    }

    log.Info().
        Str("method", "docStore.Retrieve").
        Str("collection", ds.collection).
        Int("documents", len(docs)).
        Msg("Retrieve operation completed successfully")

    return &ai.RetrieverResponse{
        Documents: docs,
    }, nil
}






























// // Copyright 2025 Google LLC
// //
// // Licensed under the Apache License, Version 2.0 (the "License");
// // you may not use this file except in compliance with the License.
// // You may obtain a copy of the License at
// //
// //     http://www.apache.org/licenses/LICENSE-2.0
// //
// // Unless required by applicable law or agreed to in writing, software
// // distributed under the License is distributed on an "AS IS" BASIS,
// // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// // See the License for the specific language governing permissions and
// // limitations under the License.
// //
// // SPDX-License-Identifier: Apache-2.0

// // Package milvus provides a Genkit plugin for Milvus vector database using milvus-sdk-go.
// package milvus

// import (
// 	"context"
// 	"encoding/json"
// 	"errors"
// 	"fmt"
// 	"os"
// 	"strings"
// 	"sync"

// 	"github.com/firebase/genkit/go/ai"
// 	"github.com/firebase/genkit/go/genkit"
// 	"github.com/milvus-io/milvus-sdk-go/v2/client"
// 	"github.com/milvus-io/milvus-sdk-go/v2/entity"
// 	"github.com/wade-liwei/agentchat/util"
// )

// // The provider used in the registry.
// const provider = "milvus"

// // Field names for Milvus schema.
// const (
// 	idField       = "id"
// 	vectorField   = "vector"
// 	textField     = "text"
// 	metadataField = "metadata"
// )

// // Milvus holds configuration for the plugin.
// type Milvus struct {
// 	// Milvus server address (host:port, e.g., "localhost:19530").
// 	// Defaults to MILVUS_ADDRESS environment variable.
// 	Addr string
// 	// Username for authentication.
// 	// Defaults to MILVUS_USERNAME.
// 	Username string
// 	// Password for authentication.
// 	// Defaults to MILVUS_PASSWORD.
// 	Password string
// 	// Token for authentication (alternative to username/password).
// 	// Defaults to MILVUS_TOKEN.
// 	Token string

// 	client  client.Client // Milvus client.
// 	mu      sync.Mutex    // Mutex to control access.
// 	initted bool          // Whether the plugin has been initialized.
// }

// // Name returns the plugin name.
// func (m *Milvus) Name() string {
// 	return provider
// }

// // Init initializes the Milvus plugin.
// func (m *Milvus) Init(ctx context.Context, g *genkit.Genkit) (err error) {
// 	if m == nil {
// 		m = &Milvus{}
// 	}

// 	m.mu.Lock()
// 	defer m.mu.Unlock()
// 	defer func() {
// 		if err != nil {
// 			err = fmt.Errorf("milvus.Init: %w", err)
// 		}
// 	}()

// 	if m.initted {
// 		return errors.New("plugin already initialized")
// 	}

// 	// Load configuration.
// 	addr := m.Addr
// 	if addr == "" {
// 		addr = os.Getenv("MILVUS_ADDRESS")
// 	}
// 	if addr == "" {
// 		return errors.New("milvus address required")
// 	}

// 	username := m.Username
// 	if username == "" {
// 		username = os.Getenv("MILVUS_USERNAME")
// 	}
// 	password := m.Password
// 	if password == "" {
// 		password = os.Getenv("MILVUS_PASSWORD")
// 	}
// 	token := m.Token
// 	if token == "" {
// 		token = os.Getenv("MILVUS_TOKEN")
// 	}

// 	// Initialize Milvus client.
// 	config := client.Config{
// 		Address:  addr,
// 		Username: username,
// 		Password: password,
// 		APIKey:   token,
// 	}
// 	client, err := client.NewClient(ctx, config)
// 	if err != nil {
// 		return fmt.Errorf("failed to initialize Milvus client: %v", err)
// 	}

// 	m.client = client
// 	m.initted = true
// 	return nil
// }

// // CollectionConfig holds configuration for an indexer/retriever pair.
// type CollectionConfig struct {
// 	// Milvus collection name. Must not be empty.
// 	Collection string
// 	// Embedding vector dimension (e.g., 1536 for text-embedding-ada-002).
// 	Dimension int
// 	// Embedder for generating vectors.
// 	Embedder ai.Embedder
// 	// Embedder options.
// 	EmbedderOptions any
// }

// // DefineIndexerAndRetriever defines an Indexer and Retriever for a Milvus collection.
// func DefineIndexerAndRetriever(ctx context.Context, g *genkit.Genkit, cfg CollectionConfig) (ai.Indexer, ai.Retriever, error) {
// 	if cfg.Embedder == nil {
// 		return nil, nil, errors.New("milvus: Embedder required")
// 	}
// 	if cfg.Collection == "" {
// 		return nil, nil, errors.New("milvus: collection name required")
// 	}
// 	if cfg.Dimension <= 0 {
// 		return nil, nil, errors.New("milvus: dimension must be positive")
// 	}

// 	m := genkit.LookupPlugin(g, provider)
// 	if m == nil {
// 		return nil, nil, errors.New("milvus plugin not found; did you call genkit.Init with the milvus plugin?")
// 	}
// 	milvus := m.(*Milvus)

// 	ds, err := milvus.newDocStore(ctx, &cfg)
// 	if err != nil {
// 		return nil, nil, err
// 	}

// 	indexer := genkit.DefineIndexer(g, provider, cfg.Collection, ds.Index)
// 	retriever := genkit.DefineRetriever(g, provider, cfg.Collection, ds.Retrieve)
// 	return indexer, retriever, nil
// }

// // docStore defines an Indexer and a Retriever.
// type docStore struct {
// 	client          client.Client
// 	collection      string
// 	dimension       int
// 	embedder        ai.Embedder
// 	embedderOptions map[string]interface{}
// }

// // newDocStore creates a docStore.
// func (m *Milvus) newDocStore(ctx context.Context, cfg *CollectionConfig) (*docStore, error) {
// 	if m.client == nil {
// 		return nil, errors.New("milvus.Init not called")
// 	}

// 	// Check/create collection.
// 	exists, err := m.client.HasCollection(ctx, cfg.Collection)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to check collection %q: %v", cfg.Collection, err)
// 	}
// 	if !exists {
// 		// Define schema with textField as primary key, plus user_id and username fields.
// 		schema := &entity.Schema{
// 			CollectionName: cfg.Collection,
// 			Fields: []*entity.Field{
// 				{
// 					Name:     vectorField,
// 					DataType: entity.FieldTypeFloatVector,
// 					TypeParams: map[string]string{
// 						"dim": fmt.Sprintf("%d", cfg.Dimension),
// 					},
// 				},
// 				{
// 					Name:       textField,
// 					DataType:   entity.FieldTypeVarChar,
// 					PrimaryKey: true, // Enforce unique constraint on text field
// 					TypeParams: map[string]string{
// 						"max_length": "65535", // Maximum length for VARCHAR
// 					},
// 				},
// 				{
// 					Name:     metadataField,
// 					DataType: entity.FieldTypeJSON,
// 				},
// 				{
// 					Name:     "user_id",
// 					DataType: entity.FieldTypeVarChar,
// 					TypeParams: map[string]string{
// 						"max_length": "128", // Reasonable length for user_id
// 					},
// 				},
// 				{
// 					Name:     "username",
// 					DataType: entity.FieldTypeVarChar,
// 					TypeParams: map[string]string{
// 						"max_length": "128", // Reasonable length for username
// 					},
// 				},
// 			},
// 		}

// 		err = m.client.CreateCollection(ctx, schema, entity.DefaultShardNumber)
// 		if err != nil {
// 			return nil, fmt.Errorf("failed to create collection %q: %v", cfg.Collection, err)
// 		}

// 		// Create HNSW index for vectorField.
// 		index, err := entity.NewIndexHNSW(
// 			entity.L2,
// 			8,  // M
// 			96, // efConstruction
// 		)
// 		if err != nil {
// 			return nil, fmt.Errorf("entity.NewIndexHNSW: %v", err)
// 		}

// 		err = m.client.CreateIndex(ctx, cfg.Collection, vectorField, index, false)
// 		if err != nil {
// 			return nil, fmt.Errorf("failed to create index: %v", err)
// 		}
// 	}

// 	// Load collection.
// 	err = m.client.LoadCollection(ctx, cfg.Collection, false)
// 	if err != nil {
// 		return nil, fmt.Errorf("failed to load collection %q: %v", cfg.Collection, err)
// 	}

// 	// Convert EmbedderOptions to map[string]interface{}.
// 	var embedderOptions map[string]interface{}
// 	if cfg.EmbedderOptions != nil {
// 		opts, ok := cfg.EmbedderOptions.(map[string]interface{})
// 		if !ok {
// 			return nil, fmt.Errorf("EmbedderOptions must be a map[string]interface{}, got %T", cfg.EmbedderOptions)
// 		}
// 		embedderOptions = opts
// 	} else {
// 		embedderOptions = make(map[string]interface{})
// 	}

// 	return &docStore{
// 		client:          m.client,
// 		collection:      cfg.Collection,
// 		dimension:       cfg.Dimension,
// 		embedder:        cfg.Embedder,
// 		embedderOptions: embedderOptions,
// 	}, nil
// }

// // Indexer returns the indexer for a collection.
// func Indexer(g *genkit.Genkit, collection string) ai.Indexer {
// 	return genkit.LookupIndexer(g, provider, collection)
// }

// // Retriever returns the retriever for a collection.
// func Retriever(g *genkit.Genkit, collection string) ai.Retriever {
// 	return genkit.LookupRetriever(g, provider, collection)
// }

// /*
//   更新 删除 很少用到；
// */

// // Index implements the Indexer.Index method.
// func (ds *docStore) Index(ctx context.Context, req *ai.IndexerRequest) error {
// 	if len(req.Documents) == 0 {
// 		return nil
// 	}

// 	// Embed documents.
// 	ereq := &ai.EmbedRequest{
// 		Input:   req.Documents,
// 		Options: ds.embedderOptions,
// 	}
// 	eres, err := ds.embedder.Embed(ctx, ereq)
// 	if err != nil {
// 		return fmt.Errorf("milvus index embedding failed: %w", err)
// 	}

// 	// Validate embedding count matches document count.
// 	if len(eres.Embeddings) != len(req.Documents) {
// 		return fmt.Errorf("mismatch: got %d embeddings for %d documents", len(eres.Embeddings), len(req.Documents))
// 	}

// 	// Prepare row-based data.
// 	var rows []interface{}
// 	for i, emb := range eres.Embeddings {
// 		doc := req.Documents[i]

// 		if doc.Metadata == nil {
// 			// If ok, we don't use the User struct since the requirement is to error on non-nil
// 			return fmt.Errorf("req.Query.Metadata  must be not nil, got type %T", req.Options)
// 		}

// 		// Extract username and user_id from req.Query.Metadata
// 		userName, ok := doc.Metadata[util.UserNameKey].(string)
// 		if !ok {
// 			return fmt.Errorf("req.Query.Metadata  must provide  username key")
// 		}
// 		userId, ok := doc.Metadata[util.UserIdKey].(string)
// 		if !ok {
// 			return fmt.Errorf("req.Query.Metadata  must provide  user_id key")
// 		}

// 		var sb strings.Builder
// 		for _, p := range doc.Content {
// 			if p.IsText() {
// 				sb.WriteString(p.Text)
// 			}
// 		}
// 		text := sb.String()
// 		metadata := doc.Metadata
// 		if metadata == nil {
// 			metadata = make(map[string]interface{})
// 		}

// 		// Create row with explicit metadata field.
// 		row := make(map[string]interface{})
// 		row["vector"] = emb.Embedding // []float32
// 		row["text"] = text
// 		row["user_id"] = userId
// 		row["username"] = userName
// 		row["metadata"] = metadata // Explicitly set metadata as JSON-compatible map
// 		rows = append(rows, row)

// 		// Debug: Log row contents.
// 		fmt.Printf("Row %d: vector_len=%d, text=%q,userId=%s,username=%s,metadata=%v\n", i, len(emb.Embedding), text, userId, userName, metadata)
// 	}

// 	// Debug: Log total rows.
// 	fmt.Printf("Inserting %d rows into collection %q\n", len(rows), ds.collection)

// 	// Insert rows into Milvus.
// 	_, err = ds.client.InsertRows(ctx, ds.collection, "", rows)
// 	if err != nil {
// 		return fmt.Errorf("milvus insert rows failed: %w", err)
// 	}

// 	return nil
// }

// // RetrieverOptions for Milvus retrieval.
// type RetrieverOptions struct {
// 	Count      int    `json:"count,omitempty"`       // Max documents to retrieve.
// 	MetricType string `json:"metric_type,omitempty"` // Similarity metric (e.g., "L2", "IP").
// }

// // Retrieve implements the Retriever.Retrieve method.
// func (ds *docStore) Retrieve(ctx context.Context, req *ai.RetrieverRequest) (*ai.RetrieverResponse, error) {
// 	if req.Query.Metadata == nil {
// 		return nil, fmt.Errorf("req.Query.Metadata must be not nil, got type %T", req.Query.Metadata)
// 	}

// 	// Extract username and user_id from req.Query.Metadata
// 	userName, ok := req.Query.Metadata[util.UserNameKey].(string)
// 	if !ok {
// 		return nil, fmt.Errorf("req.Query.Metadata must provide username key")
// 	}
// 	userId, ok := req.Query.Metadata[util.UserIdKey].(string)
// 	if !ok {
// 		return nil, fmt.Errorf("req.Query.Metadata must provide user_id key")
// 	}

// 	count := 3 // Default.
// 	metricTypeStr := "L2"
// 	if req.Options != nil {
// 		ropt, ok := req.Options.(*RetrieverOptions)
// 		if !ok {
// 			return nil, fmt.Errorf("milvus.Retrieve options have type %T, want %T", req.Options, &RetrieverOptions{})
// 		}
// 		if ropt.Count > 0 {
// 			count = ropt.Count
// 		}
// 		if ropt.MetricType != "" {
// 			metricTypeStr = ropt.MetricType
// 		}
// 	}

// 	// Map string metric type to entity.MetricType.
// 	var metricType entity.MetricType
// 	switch metricTypeStr {
// 	case "L2":
// 		metricType = entity.L2
// 	case "IP":
// 		metricType = entity.IP
// 	default:
// 		return nil, fmt.Errorf("unsupported metric type: %s", metricTypeStr)
// 	}

// 	// Embed query.
// 	ereq := &ai.EmbedRequest{
// 		Input:   []*ai.Document{req.Query},
// 		Options: ds.embedderOptions,
// 	}
// 	eres, err := ds.embedder.Embed(ctx, ereq)
// 	if err != nil {
// 		return nil, fmt.Errorf("milvus retrieve embedding failed: %v", err)
// 	}
// 	if len(eres.Embeddings) == 0 {
// 		return nil, errors.New("no embeddings generated for query")
// 	}
// 	queryVector := entity.FloatVector(eres.Embeddings[0].Embedding)

// 	// Create search parameters.
// 	searchParams, err := entity.NewIndexHNSWSearchParam(64) // ef
// 	if err != nil {
// 		return nil, fmt.Errorf("NewIndexHNSWSearchParam failed: %v", err)
// 	}

// 	// Define filter expression for user_id
// 	expr := fmt.Sprintf("user_id == %q", userId)

// 	// Perform vector search to get IDs, text, and metadata.
// 	results, err := ds.client.Search(
// 		ctx,
// 		ds.collection,
// 		[]string{},                         // partitions
// 		expr,                               // Filter by user_id
// 		[]string{textField, metadataField}, // Output fields: text and metadata
// 		[]entity.Vector{queryVector},
// 		vectorField,
// 		metricType,
// 		count,
// 		searchParams,
// 	)
// 	if err != nil {
// 		return nil, fmt.Errorf("milvus search failed: %v", err)
// 	}

// 	// Process search results.
// 	var docs []*ai.Document
// 	for _, result := range results {
// 		// Find text and metadata columns in search results.
// 		var textCol, metaCol entity.Column
// 		for _, col := range result.Fields {
// 			if col.Name() == textField {
// 				textCol = col
// 			}
// 			if col.Name() == metadataField {
// 				metaCol = col
// 			}
// 		}

// 		// Ensure text column exists.
// 		if textCol == nil {
// 			return nil, fmt.Errorf("text column %s not found in search results", textField)
// 		}

// 		// Iterate over rows (assuming columns have same length).
// 		for i := 0; i < result.ResultCount; i++ {
// 			// Get text value.
// 			text, err := textCol.GetAsString(i)
// 			if err != nil {
// 				fmt.Printf("Failed to parse text at index %d: %v\n", i, err)
// 				continue
// 			}

// 			// Get metadata value (optional, as metadata column may be missing).
// 			var metadata map[string]interface{}
// 			if metaCol != nil {
// 				metaStr, err := metaCol.GetAsString(i)
// 				if err == nil && metaStr != "" {
// 					if err := json.Unmarshal([]byte(metaStr), &metadata); err != nil {
// 						fmt.Printf("Failed to parse metadata at index %d: %v\n", i, err)
// 						continue
// 					}
// 				} else if err != nil {
// 					fmt.Printf("Failed to get metadata string at index %d: %v\n", i, err)
// 				}
// 			}

// 			// Ensure metadata includes user_id and username from query
// 			if metadata == nil {
// 				metadata = make(map[string]interface{})
// 			}
// 			metadata[util.UserIdKey] = userId
// 			metadata[util.UserNameKey] = userName

// 			// Create document.
// 			doc := ai.DocumentFromText(text, metadata)
// 			docs = append(docs, doc)
// 		}
// 	}

// 	return &ai.RetrieverResponse{
// 		Documents: docs,
// 	}, nil
// }
