package main

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"

	"github.com/firebase/genkit/go/ai"
	"github.com/firebase/genkit/go/genkit"
	"github.com/wade-liwei/agentchat/plugins/deepseek"
	"github.com/wade-liwei/agentchat/plugins/graphrag"
	"github.com/wade-liwei/agentchat/plugins/knowledge"
	"github.com/wade-liwei/agentchat/plugins/ollama"
	"github.com/wade-liwei/agentchat/plugins/question"
	"github.com/wade-liwei/agentchat/util"
	"google.golang.org/genai"

	// Import knowledge package
	"github.com/firebase/genkit/go/plugins/googlegenai"
	"github.com/rs/zerolog/log"
	_ "github.com/wade-liwei/agentchat/docs" // 导入生成的 Swagger 文档
)

// ProviderInfo 封装模型和嵌入器信息
type ProviderInfo struct {
	Models    map[string]ai.ModelInfo `json:"models,omitempty"`
	Embedders []string                `json:"embedders,omitempty"`
	Provider  string                  `json:"provider,omitempty"`
}

// DefineModelsFlow 定义 models 流
func DefineModelsFlow(g *genkit.Genkit) {
	genkit.DefineFlow(g, "models", func(ctx context.Context, provider string) ([]ProviderInfo, error) {
		log.Info().
			Str("method", "DefineModelsFlow").
			Str("provider", provider).
			Msg("Listing models and embedders for provider")

		infos := make([]ProviderInfo, 0, 3)

		switch provider {
		case googlegenai.GoogleAIProvider:
			googleModels, err := googlegenai.ListModels(googlegenai.GoogleAIProvider)
			if err != nil {
				log.Error().
					Str("method", "DefineModelsFlow").
					Str("provider", provider).
					Err(err).
					Msg("Failed to list Google AI provider info")
				return nil, err
			}
			googleEmbedders, err := googlegenai.ListEmbedders(genai.BackendGeminiAPI)
			if err != nil {
				log.Error().
					Str("method", "DefineModelsFlow").
					Str("provider", provider).
					Err(err).
					Msg("Failed to list Google AI embedders")
				return nil, err
			}

			googleInfo := ProviderInfo{
				Provider:  googlegenai.GoogleAIProvider,
				Models:    googleModels,
				Embedders: googleEmbedders,
			}

			infos = append(infos, googleInfo)
			return infos, nil

		case deepseek.Provider:
			dsModels := deepseek.ListModels()
			infos = append(infos, ProviderInfo{
				Provider:  deepseek.Provider,
				Models:    dsModels,
				Embedders: []string{}, // DeepSeek 不支持嵌入器
			})
			return infos, nil

		case ollama.Provider:
			ollamaModels, err := ollama.ListModels()
			if err != nil {
				log.Error().
					Str("method", "DefineModelsFlow").
					Str("provider", provider).
					Err(err).
					Msg("Failed to list Ollama provider info")
				return nil, err
			}
			infos = append(infos, ProviderInfo{
				Provider: ollama.Provider,
				Models:   ollamaModels,
			})
			return infos, nil

		default:
			// Google AI
			googleModels, err := googlegenai.ListModels(googlegenai.GoogleAIProvider)
			if err != nil {
				log.Error().
					Str("method", "DefineModelsFlow").
					Str("provider", googlegenai.GoogleAIProvider).
					Err(err).
					Msg("Failed to list Google AI provider info")
				return nil, err
			}
			googleEmbedders, err := googlegenai.ListEmbedders(genai.BackendGeminiAPI)
			if err != nil {
				log.Error().
					Str("method", "DefineModelsFlow").
					Str("provider", googlegenai.GoogleAIProvider).
					Err(err).
					Msg("Failed to list Google AI embedders")
				return nil, err
			}
			infos = append(infos, ProviderInfo{
				Provider:  googlegenai.GoogleAIProvider,
				Models:    googleModels,
				Embedders: googleEmbedders,
			})

			// DeepSeek
			dsModels := deepseek.ListModels()
			infos = append(infos, ProviderInfo{
				Provider:  deepseek.Provider,
				Models:    dsModels,
				Embedders: []string{}, // DeepSeek 不支持嵌入器
			})

			ollamaModels, err := ollama.ListModels()
			if err != nil {
				log.Error().
					Str("method", "DefineModelsFlow").
					Str("provider", provider).
					Err(err).
					Msg("Failed to list Ollama provider info")
				return nil, err
			}
			infos = append(infos, ProviderInfo{
				Provider: ollama.Provider,
				Models:   ollamaModels,
			})

			log.Info().
				Str("method", "DefineModelsFlow").
				Str("provider", provider).
				Msg("------default---")

			return infos, nil
		}
	})
}

func DefineGraphFlow(g *genkit.Genkit, graphIndexer ai.Indexer) {

	genkit.DefineFlow(g, "index/graph", func(ctx context.Context, input *GraphInput) (Response, error) {

		opt := graphrag.IndexReqOption{
			UserId:   input.UserID,
			UserName: input.Username,
		}

		resDocName := ""

		if v, ok := input.Metadata[graphrag.DocNameKey]; !ok {
			// Generate random docName.
			docName, err := graphrag.GenerateRandomDocName(8)
			if err != nil {
				return Response{
					Code: 500,
					Msg:  fmt.Sprintf("generate random docName for document  %w", err),
				}, nil
			}
			input.Metadata[graphrag.DocNameKey] = docName
			resDocName = docName
		} else {
			if str, isString := v.(string); isString {
				resDocName = str
			}
		}

		doc := ai.DocumentFromText(input.Content, input.Metadata)
		err := graphIndexer.Index(ctx, &ai.IndexerRequest{
			Documents: []*ai.Document{doc},
			Options:   &opt,
		})
		if err != nil {
			return Response{
				Code: 500,
				Msg:  fmt.Sprintf("index document:  %w", err),
			}, nil
		}

		return Response{
			Code: 200,
			Msg:  fmt.Sprintf("Document indexed successfully, docname %s", resDocName),
		}, nil
	})
}

func DefineDocumentFlow(g *genkit.Genkit, indexer ai.Indexer) {
	// 定义文档索引流
	genkit.DefineFlow(g, "index/document", func(ctx context.Context, input *DocumentInput) (Response, error) {

		if input.Metadata == nil {
			input.Metadata = make(map[string]any)
		}

		input.Metadata[util.UserIdKey] = input.UserID
		input.Metadata[util.UserNameKey] = input.Username

		doc := ai.DocumentFromText(input.Content, input.Metadata)
		err := indexer.Index(ctx, &ai.IndexerRequest{
			Documents: []*ai.Document{doc},
		})

		if err != nil {
			return Response{
				Code: 500,
				Msg:  err.Error(),
			}, nil
		}
		return Response{
			Code: 200,
			Msg:  "Document indexed successfully",
		}, nil
	})
}

func DefineChatFlow(g *genkit.Genkit, retriever ai.Retriever, graphRetriever ai.Retriever, pgConnString string, kc *knowledge.KnowledgeClient) {

	qa, err := question.InitQAStore(pgConnString)
	if err != nil {
		log.Fatal().Msgf("InitQAStore failed: %v", err)
	}

	// Define a simple flow that generates jokes about a given topic
	genkit.DefineFlow(g, "chat", func(ctx context.Context, input *ChatInput) (Response, error) {

		inputAsJson, err := json.Marshal(input)
		if err != nil {
			return Response{
				Code: 500,
				Msg:  fmt.Sprintf("json.Marshal:  %w", err),
			}, nil
		}

		log.Info().Msgf("input--------%s", string(inputAsJson))

		idx, lastQa, lastok, err := qa.WriteAndGetLatestQA(context.Background(), question.QA{
			FromID:   &input.FromID,
			From:     &input.From,
			Question: &input.Content,
			To:       &input.To,
			ToID:     &input.ToID,
			Milvus:   &input.Milvus,
			Graph:    &input.Graph,
		})

		if err != nil {
			return Response{
				Code: 500,
				Msg:  fmt.Sprintf("WriteAndGetLatestQA:  %w", err),
			}, nil
		}

		qaAsJson, err := json.Marshal(lastQa)

		if err != nil {
			return Response{
				Code: 500,
				Msg:  fmt.Sprintf("json.Marshal(lastQa):  %w", err),
			}, nil
		}

		log.Info().Msgf("qaAsJson--------%s", string(qaAsJson))

		promptInput := &simpleQaPromptInput{
			Query: input.Content,
		}

		if lastok && lastQa.Summary != nil {
			promptInput.Summary = *lastQa.Summary
		}

		metaData := make(map[string]any)
		metaData[util.UserIdKey] = input.ToID
		metaData[util.UserNameKey] = input.To
		dRequest := ai.DocumentFromText(input.Content, metaData)

		if input.Milvus {
			response, err := ai.Retrieve(ctx, retriever, ai.WithDocs(dRequest))
			if err != nil {
				log.Error().Msgf("milvus Retrieve err.Error() %s", err.Error())
			} else {
				var sb strings.Builder
				for _, d := range response.Documents {
					sb.WriteString(d.Content[0].Text)
					sb.WriteByte('\n')
				}

				promptInput.Context = sb.String()
				log.Info().Msgf("promptInput.Context: %s", promptInput.Context)
			}
		}

		if input.Graph {
			graphResponse, err := ai.Retrieve(ctx, graphRetriever, ai.WithDocs(dRequest))
			if err != nil {
				log.Error().Msgf("graph Retrieve err.Error() %s", err.Error())
			} else {
				var sb strings.Builder
				for _, d := range graphResponse.Documents {
					sb.WriteString(d.Content[0].Text)
					sb.WriteByte('\n')
				}
				promptInput.Graph = sb.String()
				log.Info().Msgf("promptInput.Graph : %s", promptInput.Graph)
			}
		}

		resp := &ai.ModelResponse{}
		var lastErr error
		for i, model := range input.Models {
			simpleQaPrompt, err := defineSimpleQaPrompt(g, model)
			if err != nil {
				// 打印错误日志
				log.Error().
					Str("model", model).
					Int("index", i).
					Err(err).
					Msg("Failed to define simple QA prompt")

				// 如果是最后一个模型，返回错误
				if i == len(input.Models)-1 {
					return Response{
						Code: 500,
						Msg:  fmt.Sprintf("index document: %w", err),
					}, nil
				}
				// 记录错误，继续下一个模型
				lastErr = err
				continue
			}

			respTemp, err := simpleQaPrompt.Execute(ctx, ai.WithInput(promptInput))
			if err != nil {
				// 打印错误日志
				log.Error().
					Str("model", model).
					Int("index", i).
					Err(err).
					Msg("Failed to execute prompt")

				// 如果是最后一个模型，返回错误
				if i == len(input.Models)-1 {
					return Response{
						Code: 500,
						Msg:  fmt.Sprintf("index document: %w", err),
					}, nil
				}
				// 记录错误，继续下一个模型
				lastErr = err
				continue
			}

			// 成功执行，更新 resp
			resp = respTemp
			break 
		}

		// 所有模型处理完成，检查是否全部失败
		if resp == nil && lastErr != nil {
			// 如果没有成功的结果，返回最后一个错误
			return Response{
				Code: 500,
				Msg:  fmt.Sprintf("index document: %w", lastErr),
			}, nil
		}

		if lastok {

			if promptInput.Summary == "" {
				promptInput.Summary = resp.Text()
			}

			log.Info().
				Str("from", input.From).
				Str("from_id", input.FromID).
				Str("to", input.To).
				Str("to_id", input.ToID).
				Str("promptInput.Query", promptInput.Query).
				Str("resp.Text()", resp.Text()).
				Str("promptInput.Summary", promptInput.Summary).Msg("QueryRewriteWithSummary")

			res, err := kc.QueryRewriteWithSummary(context.Background(), promptInput.Query, resp.Text(), promptInput.Summary)

			if err != nil {
				log.Error().Msg(err.Error())
			} else {
				qa.UpdateQAFields(context.Background(), idx, res.RewrittenQuery, resp.Text())
				/*
					{"RewrittenQuery":"Conversation summary: The available knowledge base does not contain information about the capital of the UK.","RawResponse":{"Response":{"Content":"Conversation summary: The available knowledge base does not contain information about the capital of the UK.","Usage":{"InputTokens":74,"OutputTokens":19,"TotalTokens":93},"RequestId":"15f1ce0c-a83f-4d95-af22-33a3bd829e8d"}}}
				*/
			}
		} else {
			qa.UpdateQAFields(context.Background(), idx, "", resp.Text())
		}

		log.Info().
			Str("from", input.From).
			Str("from_id", input.FromID).
			Str("to", input.To).
			Str("to_id", input.ToID).
			Str("question", promptInput.Query).
			Str("context", promptInput.Context).
			Str("graph", promptInput.Graph).
			Str("last summary", promptInput.Summary).
			Str("answer", resp.Text()).
			Msg("Question and answer pair recorded")

		return Response{
			Data: resp.Text(),
			Code: 200,
		}, nil
	})
}

// defineSimpleQaPrompt 加载或定义指定名称的 Prompt
func defineSimpleQaPrompt(g *genkit.Genkit, promptName string) (*ai.Prompt, error) {
	// 步骤 1：尝试查找现有的 Prompt
	log.Info().
		Str("method", "defineSimpleQaPrompt").
		Str("prompt_name", promptName).
		Msg("Attempting to lookup prompt")
	prompt := genkit.LookupPrompt(g, promptName)
	if prompt != nil {
		log.Info().
			Str("method", "defineSimpleQaPrompt").
			Str("prompt_name", promptName).
			Msg("Prompt found and loaded")
		return prompt, nil
	}
	log.Info().
		Str("method", "defineSimpleQaPrompt").
		Str("prompt_name", promptName).
		Msg("Prompt not found, defining new prompt")

	// 步骤 2：如果未找到，定义新的 Prompt
	simpleQaPrompt, err := genkit.DefinePrompt(g, promptName,
		// ai.WithModelName("deepseek/deepseek-chat"),
		ai.WithModelName(promptName),
		ai.WithPrompt(simpleQaPromptTemplate),
		ai.WithInputType(simpleQaPromptInput{}),
		ai.WithOutputFormat(ai.OutputFormatText),
	)
	if err != nil {
		log.Error().
			Str("method", "defineSimpleQaPrompt").
			Str("prompt_name", promptName).
			Err(err).
			Msg("Failed to define prompt")
		return nil, err
	}

	log.Info().
		Str("method", "defineSimpleQaPrompt").
		Str("prompt_name", promptName).
		Msg("Prompt defined successfully")
	return simpleQaPrompt, nil
}

func DefineSplitDocFlow(g *genkit.Genkit, kc *knowledge.KnowledgeClient) {
	// Define a simple flow that generates jokes about a given topic
	genkit.DefineFlow(g, "split/document", func(ctx context.Context, input string) (SplitDocResponse, error) {
		res, err := kc.ReconstructDocumentSSE(ctx, input)

		if err != nil {
			return SplitDocResponse{
				Code: 500,
				Msg:  err.Error(),
			}, nil
		}
		for k, v := range res {
			fmt.Println("process", k, *v.Progress)
		}
		return SplitDocResponse{
			Code: 200,
			Data: res,
		}, nil
	})
}

type Response struct {
	Data string `json:"data"`
	Code int    `json:"code"`
	Msg  string `json:"msg"`
}

type SplitDocResponse struct {
	Data []knowledge.ReconstructDocumentSSEResponse `json:"data"`
	Code int                                        `json:"code"`
	Msg  string                                     `json:"msg"`
}

const simpleQaPromptTemplate = `
You're a helpful agent that answers the user's questions based on the provided context.

Here is the user's query: {{query}}

Here is the context you should use: {{context}} from Milvus

Graph context: {{graph}}

Previous conversation summary: {{summary}}

Instructions:
- If the query is related to a character's personality, adopt the tone and style specified in the Personality context, and generate a response using the Milvus and Graph contexts to inform the personality-driven content.
- For all other queries, provide a clear and accurate response using the Milvus and Graph contexts, without emphasizing the Personality context.
- Ensure responses leverage the Previous conversation summary when relevant.
`
