/*

curl -X 'POST' \
  'http://54.92.111.204:5670/knowledge/space/add' \
  -H 'accept: application/json' \
  -H 'Content-Type: application/json' \
  -d '{
  "id": 0,
  "name": "string",
  "vector_type": "string",
  "domain_type": "Normal",
  "desc": "string",
  "owner": "string",
  "space_id": 0
}'



curl -X 'POST' \
  'http://54.92.111.204:5670/knowledge/111111/document/add' \
  -H 'accept: application/json' \
  -H 'Content-Type: application/json' \
  -d '{
  "doc_name": "string",
  "doc_id": 0,
  "doc_type": "string",
  "doc_token": "string",
  "content": "string",
  "source": "string",
  "labels": "string",
  "questions": [
    "string"
  ]
}'




curl -X 'POST' \
  'http://54.92.111.204:5670/knowledge/1111/document/sync_batch' \
  -H 'accept: application/json' \
  -H 'Content-Type: application/json' \
  -d '[
  {
    "doc_id": 0,
    "space_id": "string",
    "model_name": "string",
    "chunk_parameters": {
      "chunk_strategy": "string",
      "text_splitter": "string",
      "splitter_type": "user_define",
      "chunk_size": 512,
      "chunk_overlap": 50,
      "separator": "\n",
      "enable_merge": true
    }
  }
]'


*/

package knowledge

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
)

// Client 知识库客户端
type Client struct {
    BaseURL string // 基础URL，例如 "http://54.92.111.204:5670"
}

// SpaceRequest 创建空间的请求结构体
type SpaceRequest struct {
    ID         int    `json:"id"`
    Name       string `json:"name"`
    VectorType string `json:"vector_type"`
    DomainType string `json:"domain_type"`
    Desc       string `json:"desc"`
    Owner      string `json:"owner"`
    SpaceID    int    `json:"space_id"`
}

// DocumentRequest 添加文档的请求结构体
type DocumentRequest struct {
    DocName   string   `json:"doc_name"`
    DocID     int      `json:"doc_id"`
    DocType   string   `json:"doc_type"`
    DocToken  string   `json:"doc_token"`
    Content   string   `json:"content"`
    Source    string   `json:"source"`
    Labels    string   `json:"labels"`
    Questions []string `json:"questions"`
}

// ChunkParameters 分片参数
type ChunkParameters struct {
    ChunkStrategy string `json:"chunk_strategy"`
    TextSplitter  string `json:"text_splitter"`
    SplitterType  string `json:"splitter_type"`
    ChunkSize    int    `json:"chunk_size"`
    ChunkOverlap int    `json:"chunk_overlap"`
    Separator    string `json:"separator"`
    EnableMerge  bool   `json:"enable_merge"`
}

// SyncBatchRequest 同步批处理的请求结构体
type SyncBatchRequest struct {
    DocID          int             `json:"doc_id"`
    SpaceID        string          `json:"space_id"`
    ModelName      string          `json:"model_name"`
    ChunkParameters ChunkParameters `json:"chunk_parameters"`
}

// NewClient 创建新的客户端实例
func NewClient(ip string, port int) *Client {
    return &Client{
        BaseURL: fmt.Sprintf("http://%s:%d", ip, port),
    }
}

// AddSpace 创建知识空间
func (c *Client) AddSpace(req SpaceRequest) (*http.Response, error) {
    url := fmt.Sprintf("%s/knowledge/space/add", c.BaseURL)
    body, err := json.Marshal(req)
    if err != nil {
        return nil, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        return nil, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        return nil, fmt.Errorf("failed to send request: %w", err)
    }

    return resp, nil
}

// AddDocument 添加文档
func (c *Client) AddDocument(spaceID string, req DocumentRequest) (*http.Response, error) {
    url := fmt.Sprintf("%s/knowledge/%s/document/add", c.BaseURL, spaceID)
    body, err := json.Marshal(req)
    if err != nil {
        return nil, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        return nil, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        return nil, fmt.Errorf("failed to send request: %w", err)
    }

    return resp, nil
}

// SyncBatchDocument 同步批处理文档
func (c *Client) SyncBatchDocument(spaceID string, req []SyncBatchRequest) (*http.Response, error) {
    url := fmt.Sprintf("%s/knowledge/%s/document/sync_batch", c.BaseURL, spaceID)
    body, err := json.Marshal(req)
    if err != nil {
        return nil, fmt.Errorf("failed to marshal request: %w", err)
    }

    httpReq, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
    if err != nil {
        return nil, fmt.Errorf("failed to create request: %w", err)
    }

    httpReq.Header.Set("Accept", "application/json")
    httpReq.Header.Set("Content-Type", "application/json")

    client := &http.Client{}
    resp, err := client.Do(httpReq)
    if err != nil {
        return nil, fmt.Errorf("failed to send request: %w", err)
    }

    return resp, nil
}
