package main

import (
	"context"
	"flag"
	"net/http"

	"github.com/firebase/genkit/go/genkit"
	"github.com/rs/zerolog"
	"github.com/wade-liwei/agentchat/plugins/deepseek"
	"github.com/wade-liwei/agentchat/plugins/graphrag" // Import knowledge package
	"github.com/wade-liwei/agentchat/plugins/knowledge"
	"github.com/wade-liwei/agentchat/plugins/milvus"
	"github.com/wade-liwei/agentchat/plugins/ollama"

	"github.com/firebase/genkit/go/plugins/googlegenai"
	"github.com/firebase/genkit/go/plugins/server"

	httpSwagger "github.com/swaggo/http-swagger"
	_ "github.com/wade-liwei/agentchat/docs" // 导入生成的 Swagger 文档

	"github.com/rs/zerolog/log"
)

type ChatInput struct {
	Content string `json:"content,omitempty"`
	From    string `json:"from,omitempty"`    // 替换 Username
	FromID  string `json:"from_id,omitempty"` // 替换 UserID
	To      string `json:"to,"`
	ToID    string `json:"to_id,omitempty"`
	//
	Models []string `json:"models,omitempty"`
	APIKey string   `json:"apiKey,omitempty"`
	Milvus bool     `json:"milvus,omitempty"`
	Graph  bool     `json:"graph,omitempty"`
}

// DocumentInput 结构体用于文档索引接口
type DocumentInput struct {
	UserID   string                 `json:"user_id"`
	Username string                 `json:"username"`
	Content  string                 `json:"content"`
	Metadata map[string]interface{} `json:"metadata,omitempty"`
}

// GraphInput 结构体用于文档索引接口
type GraphInput struct {
	UserID   string                 `json:"user_id"`
	Username string                 `json:"username"`
	Content  string                 `json:"content"`
	Metadata map[string]interface{} `json:"metadata,omitempty"`
}

type simpleQaPromptInput struct {
	Query   string `json:"query"`
	Context string `json:"context"`
	Graph   string `json:"graph"`
	Summary string `json:"summary"`
}

func main() {

	// Define command-line flags with hardcoded values as defaults
	ollamaServerAddress := flag.String("ollama-server-address", "http://localhost:11434", "Ollama server address")
	deepseekAPIKey := flag.String("deepseek-api-key", "sk-9f70df871a7c4b8aa566a3c7a0603706", "DeepSeek API key")
	milvusAddr := flag.String("milvus-addr", "standalone:19530", "Milvus server address")
	graphragAddr := flag.String("graphrag-addr", "webserver:5670", "GraphRAG server address")
	googleAIApiKey := flag.String("googleai-api-key", "AIzaSyCoYBOmnwRWlH_-nT25lpn8pMg3T18Q0uI", "Google AI API key")
	pgConnString := flag.String("pg-conn-string", "postgresql://postgres.awcfgdodiuqnlsobcivq:P99IU9NEoDRPsBfb@aws-0-ap-southeast-1.pooler.supabase.com:5432/postgres", "PostgreSQL connection string")
	tencentcloudSecretID := flag.String("tencentcloud-secret-id", "AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF", "Tencent Cloud Secret ID")
    tencentcloudSecretKey := flag.String("tencentcloud-secret-key", "rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd", "Tencent Cloud Secret Key")
	debug := flag.Bool("debug", false, "sets log level to debug")
	flag.Parse()

	loggingInit()

	// Log all flag values
	log.Info().
    Str("ollama_server_address", *ollamaServerAddress).
    Str("deepseek_api_key", *deepseekAPIKey).
    Str("milvus_addr", *milvusAddr).
    Str("graphrag_addr", *graphragAddr).
    Str("googleai_api_key", *googleAIApiKey).
    Str("pg_conn_string", *pgConnString).
    Str("tencentcloud_secret_id", *tencentcloudSecretID).
    Str("tencentcloud_secret_key", *tencentcloudSecretKey).
    Bool("debug", *debug).
    Msg("Startup parameters")


	if *debug {
		zerolog.SetGlobalLevel(zerolog.DebugLevel)
	}

	ctx := context.Background()

	// Initialize genkit with plugins using flag/env values
	g, err := genkit.Init(ctx, genkit.WithPlugins(
		&ollama.Ollama{ServerAddress: *ollamaServerAddress},
		&deepseek.DeepSeek{APIKey: *deepseekAPIKey},
		&milvus.Milvus{Addr: *milvusAddr},
		&graphrag.GraphKnowledge{Addr: *graphragAddr},
		&googlegenai.GoogleAI{APIKey: *googleAIApiKey},
	))

	if err != nil {
		log.Fatal().Msg(err.Error())
	}

	embedder := googlegenai.GoogleAIEmbedder(g, "embedding-001")
	if embedder == nil {
		log.Fatal().Msg(err.Error())
	}

	// Configure collection
	cfg := milvus.CollectionConfig{
		Collection:      "chatRag1",
		Dimension:       768, // Match mock embedder dimension
		Embedder:        embedder,
		EmbedderOptions: map[string]interface{}{}, // Explicitly set as map
	}

	// Define indexer and retriever
	indexer, retriever, err := milvus.DefineIndexerAndRetriever(ctx, g, cfg)
	if err != nil {
		log.Fatal().Msgf("DefineIndexerAndRetriever failed: %v", err)
	}

	graphIndexer, graphRetriever, err := graphrag.DefineIndexerAndRetriever(ctx, g)

	if err != nil {
		log.Fatal().Msgf("graphrag.DefineIndexerAndRetriever  failed: %v", err)
	}

	DefineDocumentFlow(g, indexer)
	DefineGraphFlow(g, graphIndexer)
	DefineModelsFlow(g)

	// Initialize KnowledgeClient with test parameters
	kc := knowledge.NewKnowledgeClient(knowledge.ClientConfig{
		Endpoint: "lkeap.tencentcloudapi.com",
		Region:   "ap-guangzhou",
		SecretID: *tencentcloudSecretID,
		SecretKey: *tencentcloudSecretKey,
	})
	if err := kc.Init(context.Background()); err != nil {
		log.Fatal().Msgf("Failed to initialize KnowledgeClient: %v", err)
	}
	log.Info().Msg("KnowledgeClient initialized successfully")

	DefineChatFlow(g, retriever, graphRetriever, *pgConnString, kc)
	DefineSplitDocFlow(g, kc)

	// 配置限速器：每秒 10 次请求，突发容量 20，最大并发 5
	rl := NewRateLimiter(10, 20, 5)

	// 创建 Genkit HTTP 处理器
	mux := http.NewServeMux()

	for _, a := range genkit.ListFlows(g) {
		handler := rl.Middleware(genkit.Handler(a))
		mux.Handle("POST /"+a.Name(), handler)
	}

	// 暴露 Swagger UI，使用 swagger.yaml
	mux.HandleFunc("/swagger/", httpSwagger.Handler(
		httpSwagger.URL("/docs/swagger.yaml"), // 指定 YAML 文件路径
	))

	// 确保 docs 目录可通过 HTTP 访问
	mux.Handle("/docs/", http.StripPrefix("/docs/", http.FileServer(http.Dir("docs"))))

	// 启动服务器，监听
	log.Printf("Server starting on 0.0.0.0:8000")
	if err := server.Start(ctx, "0.0.0.0:8000", mux); err != nil {
		log.Fatal().Msgf("Server failed: %v", err)
	}
}
