package knowledge

import (
	"context"
	"os"
	"testing"

	"github.com/rs/zerolog"
	"github.com/rs/zerolog/log"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
)

func TestMain(m *testing.M) {
	// Configure zerolog for human-readable console output during tests
	zerolog.TimeFieldFormat = zerolog.TimeFormatUnix
	log.Logger = log.Output(zerolog.ConsoleWriter{Out: os.Stderr})

	// Run tests
	os.Exit(m.Run())
}

func TestKnowledgeClient_QueryRewrite(t *testing.T) {
	// Warning: Do not hardcode credentials in production code. Use environment variables or a secure vault.
	// The credentials below are placeholders for testing purposes.
	os.Setenv("TENCENTCLOUD_SECRET_ID", "AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF")
	os.Setenv("TENCENTCLOUD_SECRET_KEY", "rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd")
	defer func() {
		os.Unsetenv("TENCENTCLOUD_SECRET_ID")
		os.Unsetenv("TENCENTCLOUD_SECRET_KEY")
	}()

	// Create client configuration
	config := ClientConfig{
		Endpoint: "lkeap.tencentcloudapi.com",
		Region:   "ap-guangzhou",
	}

	// Initialize client
	client := NewKnowledgeClient(config)
	ctx := context.Background()

	// Test cases
	tests := []struct {
		name        string
		messages    []*lkeap.Message
		model       string
		expectError bool
	}{
		{
			name: "CurlPayload",
			messages: []*lkeap.Message{
				{
					Role:    common.StringPtr("user"),
					Content: common.StringPtr("你的家在哪里"),
				},
				{
					Role:    common.StringPtr("assistant"),
					Content: common.StringPtr("国内"),
				},
				{
					Role:    common.StringPtr("user"),
					Content: common.StringPtr("国内哪里"),
				},
			},
			model:       "lke-query-rewrite-base",
			expectError: true, // Expect error due to potentially invalid credentials
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Initialize client for each test
			if err := client.Init(ctx); err != nil {
				t.Fatalf("Failed to initialize KnowledgeClient: %v", err)
			}

			// Perform query rewrite
			req := QueryRewriteRequest{
				Messages: tt.messages,
				Model:    tt.model,
			}
			resp, err := client.QueryRewrite(ctx, req)

			// Check error expectation
			if tt.expectError {
				if err == nil {
					t.Error("Expected error, got none")
				} else {
					log.Debug().
						Str("method", "TestKnowledgeClient_QueryRewrite").
						Str("test_name", tt.name).
						Err(err).
						Msg("Received expected error")
				}
				return
			}

			// Check response (only for non-error cases)
			if err != nil {
				t.Fatalf("QueryRewrite failed: %v", err)
			}
			if resp.RewrittenQuery == "" {
				t.Error("Expected non-empty rewritten query")
			}

			log.Info().
				Str("method", "TestKnowledgeClient_QueryRewrite").
				Str("test_name", tt.name).
				Str("rewritten_query", resp.RewrittenQuery).
				Msg("Query rewrite successful")
		})
	}
}

// func TestKnowledgeClient_ReconstructDocumentSSE(t *testing.T) {
//     // Warning: Do not hardcode credentials in production code. Use environment variables or a secure vault.
//     os.Setenv("TENCENTCLOUD_SECRET_ID", "AKID64oLfmfLtESUJ6i8LPSM4gCVbiniQuBF")
//     os.Setenv("TENCENTCLOUD_SECRET_KEY", "rX2JMBnBMJ2YqulOo37xa5OUMSN4Xnpd")
//     defer func() {
//         os.Unsetenv("TENCENTCLOUD_SECRET_ID")
//         os.Unsetenv("TENCENTCLOUD_SECRET_KEY")
//     }()

//     // Create client configuration
//     config := ClientConfig{
//         Endpoint: "lkeap.tencentcloudapi.com",
//         Region:   "ap-guangzhou",
//     }

//     // Initialize client
//     client := NewKnowledgeClient(config)
//     ctx := context.Background()

//     // Test cases
//     tests := []struct {
//         name        string
//         text        string
//         initClient  bool
//         expectError bool
//     }{
//         {
//             name:        "ValidText",
//             text:        "Sample text for reconstruction.\nAnother line.",
//             initClient:  true,
//             expectError: true, // Expect error due to potentially invalid credentials
//         },
//         {
//             name:        "EmptyText",
//             text:        "",
//             initClient:  true,
//             expectError: true,
//         },
//         {
//             name:        "LargeText",
//             text:        strings.Repeat("a", 9*1024*1024), // ~9MB
//             initClient:  true,
//             expectError: true,
//         },
//         {
//             name:        "UninitializedClient",
//             text:        "Sample text.",
//             initClient:  false,
//             expectError: true,
//         },
//     }

//     for _, tt := range tests {
//         t.Run(tt.name, func(t *testing.T) {
//             // Initialize client if required
//             if tt.initClient {
//                 if err := client.Init(ctx); err != nil {
//                     t.Fatalf("Failed to initialize KnowledgeClient: %v", err)
//                 }
//             }

//             // Perform document reconstruction
//             result, err := client.ReconstructDocumentSSE(ctx, tt.text)

//             // Check error expectation
//             if tt.expectError {
//                 if err == nil {
//                     t.Error("Expected error, got none")
//                 } else {
//                     log.Debug().
//                         Str("method", "TestKnowledgeClient_ReconstructDocumentSSE").
//                         Str("test_name", tt.name).
//                         Str("error", err.Error()).
//                         Msg("Received expected error")
//                 }
//                 return
//             }

//             // Check response
//             if err != nil {
//                 t.Errorf("ReconstructDocumentSSE failed: %v", err)
//             }
//             if result == "" {
//                 t.Error("Expected non-empty response JSON")
//             }

//             log.Info().
//                 Str("method", "TestKnowledgeClient_ReconstructDocumentSSE").
//                 Str("test_name", tt.name).
//                 Str("response_json", result[:min(100, len(result))]).
//                 Msg("Document reconstruction successful")
//         })
//     }
// }
