package deepseek

import (
	"context"
	"fmt"
	"strings"
	"sync"

	"github.com/firebase/genkit/go/ai"
	"github.com/firebase/genkit/go/genkit"
	"github.com/rs/zerolog/log"

	deepseek "github.com/cohesion-org/deepseek-go"
)

const provider = "deepseek"
const Provider = "deepseek"

// 常量定义支持的模型
const (
	// DeepSeekChat     = "deepseek-chat"
	// DeepSeekCoder    = "deepseek-coder"
	// DeepSeekReasoner = "deepseek-reasoner"
	DeepSeekChat     = deepseek.DeepSeekChat
	DeepSeekCoder    = deepseek.DeepSeekCoder
	DeepSeekReasoner = deepseek.DeepSeekReasoner
)

var (
	mediaSupportedModels = []string{DeepSeekChat, DeepSeekCoder, DeepSeekReasoner}
	roleMapping          = map[ai.Role]string{
		ai.RoleUser:   deepseek.ChatMessageRoleUser,
		ai.RoleModel:  deepseek.ChatMessageRoleAssistant,
		ai.RoleSystem: deepseek.ChatMessageRoleSystem,
		ai.RoleTool:   deepseek.ChatMessageRoleTool,
	}
)

// modelInfoMap 定义支持的模型及其信息
var modelInfoMap = map[string]ai.ModelInfo{
	DeepSeekChat: {
		Label: "DeepSeek Chat",
		Supports: &ai.ModelSupports{
			Multiturn:  true,
			SystemRole: true,
			Media:      false,
			Tools:      true,
		},
		Versions: []string{},
	},
	DeepSeekCoder: {
		Label: "DeepSeek Coder",
		Supports: &ai.ModelSupports{
			Multiturn:  true,
			SystemRole: true,
			Media:      false,
			Tools:      true,
		},
		Versions: []string{},
	},
	DeepSeekReasoner: {
		Label: "DeepSeek Reasoner",
		Supports: &ai.ModelSupports{
			Multiturn:  true,
			SystemRole: true,
			Media:      false,
			Tools:      true,
		},
		Versions: []string{},
	},
}

func ListModels() map[string]ai.ModelInfo {
	return modelInfoMap
}

// DeepSeek holds configuration for the plugin.
type DeepSeek struct {
	APIKey string // DeepSeek API key

	mu      sync.Mutex // Mutex to control access.
	initted bool       // Whether the plugin has been initialized.
}

// Name returns the provider name.
func (d *DeepSeek) Name() string {
	return provider
}

// ModelDefinition represents a model with its name and type.
type ModelDefinition struct {
	Name string
	Type string
}

// Init initializes the DeepSeek plugin and registers all known models.
func (d *DeepSeek) Init(ctx context.Context, g *genkit.Genkit) error {
	log.Info().Str("method", "DeepSeek.Init").Msg("Initializing DeepSeek plugin")
	d.mu.Lock()
	defer d.mu.Unlock()
	if d.initted {
		log.Error().Str("method", "DeepSeek.Init").Msg("Plugin already initialized")
		return fmt.Errorf("deepseek.Init already called")
	}

	if d == nil || d.APIKey == "" {
		log.Error().Str("method", "DeepSeek.Init").Msg("APIKey is required")
		return fmt.Errorf("deepseek: need APIKey")
	}

	d.initted = true

	//d.mu.Unlock()
	// 自动注册所有支持的模型
	for _, modelName := range mediaSupportedModels {
		modelDef := ModelDefinition{
			Name: modelName,
			Type: "chat", // 假设所有模型都支持 chat 类型
		}
		d.DefineModel(g, modelDef, nil)
	}

	log.Info().Str("method", "DeepSeek.Init").Msg("Initialization successful")
	return nil
}

// DefineModel defines a DeepSeek model in Genkit.
func (d *DeepSeek) DefineModel(g *genkit.Genkit, model ModelDefinition, info *ai.ModelInfo) ai.Model {
	log.Info().
		Str("method", "DeepSeek.DefineModel").
		Str("model_name", model.Name).
		Msg("Defining DeepSeek model")
	// d.mu.Lock()
	// defer d.mu.Unlock()
	if !d.initted {
		log.Error().Str("method", "DeepSeek.DefineModel").Msg("DeepSeek not initialized")
		return nil
	}

	// 获取模型信息
	var mi ai.ModelInfo
	if info != nil {
		mi = *info
	} else {
		var ok bool
		mi, ok = modelInfoMap[model.Name]
		if !ok {
			log.Error().
				Str("method", "DeepSeek.DefineModel").
				Str("model_name", model.Name).
				Msg("Unknown model and no ModelInfo provided")
			return nil
		}
	}

	// 定义生成器
	gen := &generator{model: model, apiKey: d.APIKey}
	modelDef := genkit.DefineModel(g, provider, model.Name, &mi, gen.generate)
	log.Info().
		Str("method", "Deep lantern").
		Str("model_name", model.Name).
		Msg("Model defined successfully")
	return modelDef
}

// generator handles model generation.
type generator struct {
	model  ModelDefinition
	apiKey string
}

// generate implements the Genkit model generation interface.
func (g *generator) generate(ctx context.Context, input *ai.ModelRequest, cb func(context.Context, *ai.ModelResponseChunk) error) (*ai.ModelResponse, error) {
	log.Info().
		Str("method", "generator.generate").
		Str("model_name", g.model.Name).
		Int("messages", len(input.Messages)).
		Msg("Starting model generation")

	if len(input.Messages) == 0 {
		log.Error().Str("method", "generator.generate").Msg("Prompt or messages required")
		return nil, fmt.Errorf("prompt or messages required")
	}

	// Initialize DeepSeek client.
	client := deepseek.NewClient(g.apiKey)
	log.Debug().Str("method", "generator.generate").Msg("DeepSeek client initialized")

	// Create a chat completion request
	request := &deepseek.ChatCompletionRequest{
		Model: g.model.Name,
	}

	for _, msg := range input.Messages {
		role, ok := roleMapping[msg.Role]
		if !ok {
			log.Error().
				Str("method", "generator.generate").
				Str("role", string(msg.Role)).
				Msg("Unsupported role")
			return nil, fmt.Errorf("unsupported role: %s", msg.Role)
		}
		content := concatMessageParts(msg.Content)
		request.Messages = append(request.Messages, deepseek.ChatCompletionMessage{
			Role:    role,
			Content: content,
		})
		log.Debug().
			Str("method", "generator.generate").
			Str("role", role).
			Str("content", content).
			Msg("Added message to request")
	}

	// Send the request and handle the response
	response, err := client.CreateChatCompletion(ctx, request)
	if err != nil {
		log.Error().
			Err(err).
			Str("method", "generator.generate").
			Msg("Failed to create chat completion")
		return nil, fmt.Errorf("create chat completion: %w", err)
	}

	log.Debug().
		Str("method", "generator.generate").
		Int("choices", len(response.Choices)).
		Msg("Received chat completion response")

	// Create a final response with the merged chunks
	finalResponse := &ai.ModelResponse{
		Request:      input,
		FinishReason: ai.FinishReason("stop"),
		Message: &ai.Message{
			Role: ai.RoleModel,
		},
	}

	for _, chunk := range response.Choices {
		log.Debug().
			Str("method", "generator.generate").
			Int("index", chunk.Index).
			Str("content", chunk.Message.Content).
			Msg("Processing response chunk")
		p := ai.Part{
			Text: chunk.Message.Content,
			Kind: ai.PartKind(chunk.Index),
		}
		finalResponse.Message.Content = append(finalResponse.Message.Content, &p)
	}

	log.Info().
		Str("method", "generator.generate").
		Str("model_name", g.model.Name).
		Int("content_parts", len(finalResponse.Message.Content)).
		Msg("Model generation completed successfully")
	return finalResponse, nil
}

// concatMessageParts concatenates message parts into a single string.
func concatMessageParts(parts []*ai.Part) string {
	log.Debug().
		Str("method", "concatMessageParts").
		Int("parts", len(parts)).
		Msg("Concatenating message parts")
	var sb strings.Builder
	for _, part := range parts {
		if part.IsText() {
			sb.WriteString(part.Text)
		}
		// Ignore non-text parts (e.g., media, tools) as DeepSeek API doesn't support them.
	}
	result := sb.String()
	log.Debug().
		Str("method", "concatMessageParts").
		Str("result", result).
		Msg("Concatenation complete")
	return result
}
