import { useEffect, useState } from 'react'

export enum TimePeriod {
  hour = 'hour',
  day = 'day',
  week = 'week',
  month = 'month',
  year = 'year',
  all = 'all',
}

export type TokenData = {
  [address: string]: {
    sparkline: string // svg string
    price: number // usdc price
    delta: number // basis points
    marketCap: number
    volume: Record<TimePeriod, number>
  }
}

interface UseTopTokensResult {
  data: TokenData | null
  error: string | null
  loading: boolean
}

const FAKE_TOP_TOKENS_RESULT = {
  '0x03ab458634910aad20ef5f1c8ee96f1d6ac54919': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 41.31694740177657 0.7988165680473372, 43.71568774238813 1.5976331360946745, 39.214791338596406 2.396449704142012, 46.76084696273317 3.195266272189349, 43.31848313855741 3.994082840236686, 50.0 4.792899408284024, 40.060343907618424 5.591715976331361, 40.68681013944878 6.390532544378698, 37.651448884438565 7.189349112426036, 44.837538286562726 7.988165680473372, 38.306409244466074 8.78698224852071, 38.191046116043644 9.585798816568047, 41.3410198277808 10.384615384615385, 42.67560277872357 11.183431952662723, 21.35038521188064 11.982248520710058, 9.708488423027035 12.781065088757396, 13.778306655876818 13.579881656804734, 9.09881194043739 14.378698224852071, 14.553394610391221 15.177514792899409, 10.408112010595799 15.976331360946745, 17.047111184927942 16.775147928994084, 22.438051145929947 17.57396449704142, 19.964721634868514 18.37278106508876, 12.570563556826544 19.171597633136095, 22.03776042126629 19.970414201183434, 18.925083357584395 20.76923076923077, 19.35072444083898 21.56804733727811, 21.40981002299806 22.366863905325445, 12.57674643462727 23.16568047337278, 20.162083141098453 23.964497041420117, 26.805621410668586 24.763313609467456, 22.414137516260947 25.562130177514792, 19.663247314553193 26.36094674556213, 30.343506101654167 27.159763313609467, 24.161948258403015 27.958579881656807, 26.057272098519324 28.757396449704142, 24.7313530966639 29.55621301775148, 26.665421037317756 30.355029585798817, 28.24452295248621 31.153846153846157, 22.087821726812084 31.95266272189349, 31.152446803141654 32.751479289940825, 13.906724922741992 33.55029585798817, 25.06576548942801 34.349112426035504, 27.904617659558312 35.14792899408284, 22.58989734125344 35.94674556213018, 18.77812362735527 36.74556213017752, 24.142831611371175 37.54437869822485, 22.97670333857772 38.34319526627219, 22.517687341971424 39.142011834319526, 15.005964139598122 39.94082840236687, 18.900129174969173 40.7396449704142, 21.18586088594469 41.53846153846154, 20.89929446076557 42.337278106508876, 21.798715655784196 43.13609467455622, 23.35029050040679 43.93491124260355, 29.342421964701487 44.73372781065089, 26.83234396439684 45.532544378698226, 20.21614893446483 46.33136094674556, 21.26355218350165 47.1301775147929, 19.724914326600633 47.928994082840234, 11.001032266342122 48.72781065088758, 19.445604145545026 49.52662721893491, 12.769383962251144 50.32544378698225, 11.339363541177807 51.124260355029584, 11.080971820705395 51.92307692307693, 9.871930076129743 52.72189349112426, 15.540234111629548 53.5207100591716, 19.385735929318766 54.319526627218934, 13.32912973357844 55.11834319526627, 14.998143085248122 55.91715976331361, 15.629741828355643 56.71597633136094, 12.063583180029193 57.514792899408285, 11.686542180307798 58.31360946745562, 9.765592857045856 59.11242603550296, 6.132503919488515 59.91124260355029, 6.291365447152408 60.710059171597635, 7.230900543395919 61.50887573964497, 8.226997644289916 62.307692307692314, 5.9415683085940145 63.10650887573964, 10.786357615134714 63.90532544378698, 6.300468254556363 64.70414201183432, 7.01673617024902 65.50295857988165, 7.513055645715544 66.30177514792899, 13.800956634581379 67.10059171597634, 14.746601659782756 67.89940828402366, 13.364390374024238 68.69822485207101, 13.376479400656974 69.49704142011834, 13.711030413235411 70.29585798816568, 20.243771793473858 71.09467455621302, 20.91915056070402 71.89349112426036, 10.524335562376507 72.6923076923077, 16.668841964995053 73.49112426035504, 16.077200187658498 74.28994082840237, 17.48790382331564 75.0887573964497, 8.25760657272661 75.88757396449704, 20.254923976468227 76.68639053254438, 11.31969662059836 77.48520710059172, 14.741879607014027 78.28402366863905, 13.592156573495034 79.0828402366864, 12.044467623434308 79.88165680473374, 8.50316426838233 80.68047337278107, 13.715774590913338 81.4792899408284, 0.0 82.27810650887574, 11.258495022417087 83.07692307692308, 9.542338145813623 83.87573964497041, 21.309625836502537 84.67455621301775, 16.894634431091397 85.4733727810651, 19.09116634354636 86.27218934911244, 18.423626148884804 87.07100591715975, 22.62720151567559 87.8698224852071, 20.594704144612486 88.66863905325444, 24.72044813699155 89.46745562130178, 23.675367002680513 90.26627218934911, 29.630501080695737 91.06508875739645, 26.318532584575365 91.8639053254438, 22.925131092870238 92.66272189349112, 20.076296983388282 93.46153846153845, 21.190330565544745 94.2603550295858, 16.717777911328476 95.05917159763314, 17.19594401396835 95.85798816568047, 28.972172514044804 96.65680473372781, 37.06141265702684 97.45562130177515, 27.393125868800766 98.2544378698225, 32.80367795886616 99.05325443786982, 31.71874849007873 99.85207100591715, 36.20244358226534 100.6508875739645, 34.090729813166604 101.44970414201183, 27.720311190850243 102.24852071005917, 38.09137264210808 103.04733727810651, 31.19514608856407 103.84615384615385, 22.16619584103189 104.64497041420118, 24.41549859938342 105.44378698224853, 35.85416325537771 106.24260355029585, 28.109900746130716 107.0414201183432, 25.83677939911684 107.84023668639053, 29.656237212703363 108.63905325443787, 25.298079680452734 109.43786982248521, 31.332534967570563 110.23668639053254, 23.9295492273324 111.03550295857988, 27.551483112483766 111.83431952662723, 29.094020265667904 112.63313609467457, 31.932887919324116 113.43195266272188, 20.225031345917436 114.23076923076923, 23.748829985960207 115.02958579881657, 23.965773347387046 115.82840236686391, 29.563403800646515 116.62721893491124, 28.048157186575423 117.42603550295858, 36.162170094309104 118.22485207100593, 32.55456795210098 119.02366863905324, 29.808843410000822 119.82248520710058, 25.941152811757785 120.62130177514793, 30.14795627238393 121.42011834319527, 22.317970221444114 122.2189349112426, 25.720600007582895 123.01775147928994, 32.418740053848005 123.81656804733728, 33.19142085157435 124.61538461538463, 37.57023431488021 125.41420118343194, 38.627613245013514 126.21301775147928, 28.115191130447247 127.01183431952663, 34.45835259388878 127.81065088757396, 32.09447784016973 128.60946745562129, 33.159107359049045 129.40828402366864, 32.462580786853785 130.20710059171597, 35.98824982668003 131.0059171597633, 37.797823430134706 131.80473372781066, 33.30767710620397 132.60355029585799, 27.317432176467175 133.40236686390534, 27.287847065956853 134.20118343195267, 29.882671744026894" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 16_326_205,
    price: 3.05,
    delta: 12,
    volume: {
      [TimePeriod.hour]: 25_000,
      [TimePeriod.day]: 619_000,
      [TimePeriod.week]: 16_800_000,
      [TimePeriod.month]: 58_920_000,
      [TimePeriod.year]: 690_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x0cec1a9154ff802e7934fc916ed7ca50bde6844e': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 12_987_355,
    price: 9.26543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x6B175474E89094C44Da98b954EedeAC495271d0F': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 11_987_355,
    price: 0.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0xdac17f958d2ee523a2206206994597c13d831ec7': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 5_987_355,
    price: 0.66543,
    delta: -79,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 10_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0xf629cbd94d3791c9250152bd8dfbdf380e2a3b9c': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 10.66543,
    delta: 56,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 7_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x95ad61b0a150d79219dcf64e1e6cc01f0b64c4ce': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: 60,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x8a2279d4a90b6fe1c4b30fa660cc9f926797baa2': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: 560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 108_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x84ca8bc7997272c7cfb4d0cd3d55cd942b3c9419': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: -50,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 10_000_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x3845badAde8e6dFF049820680d1F14bD3903a5d0': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 4.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x4c19596f5aaff459fa38b0f7ed92f11ae6543784': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: 60,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 2_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x71Ab77b7dbB4fa7e017BC15090b2163221420282': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0xccc8cb5229b0ac8069c51fd58367fd1e622afd97': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x03be5c903c727ee2c8c4e9bc0acc860cca4715e2': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x4674672bcddda2ea5300f5207e1158185c944bc0': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: 70,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0xdf801468a808a32656d2ed2d2d80b72a129739f4': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 1_987_355,
    price: 0.66543,
    delta: -50,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0xaDB2437e6F65682B85F814fBc12FeC0508A7B1D0': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 187_355,
    price: 0.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
  '0x1796ae0b0fa4862485106a0de9b654eFE301D0b2': {
    sparkline: `<svg width="135" height="50" viewBox="0 0 135 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <polyline points="0.0, 19.069794193924274 0.7988165680473372, 22.49421204178107 1.5976331360946745, 19.17009251279608 2.396449704142012, 21.88399814415597 3.195266272189349, 26.452314288582524 3.994082840236686, 22.28229723394612 4.792899408284024, 9.129019392277007 5.591715976331361, 13.362494884505907 6.390532544378698, 13.141638169690026 7.189349112426036, 16.20889017509926 7.988165680473372, 20.57785115867745 8.78698224852071, 24.885384162352622 9.585798816568047, 25.967077863157744 10.384615384615385, 24.574799923859146 11.183431952662723, 31.485622374073124 11.982248520710058, 38.81570800977322 12.781065088757396, 39.83458414280211 13.579881656804734, 35.15798230163302 14.378698224852071, 28.469739240535898 15.177514792899409, 27.80611066475612 15.976331360946745, 21.475416932054085 16.775147928994084, 25.383754575566215 17.57396449704142, 24.65435602483133 18.37278106508876, 30.9846243719124 19.171597633136095, 29.927868632474393 19.970414201183434, 15.622245847433284 20.76923076923077, 16.709285581063654 21.56804733727811, 16.748007109489496 22.366863905325445, 8.735020590484721 23.16568047337278, 10.842779773261826 23.964497041420117, 9.49856498810183 24.763313609467456, 8.94305283614397 25.562130177514792, 9.18575921702126 26.36094674556213, 11.158843666172713 27.159763313609467, 11.551941690440524 27.958579881656807, 12.272132962363939 28.757396449704142, 15.514345443592633 29.55621301775148, 11.789989824567304 30.355029585798817, 13.704593559596123 31.153846153846157, 15.373891613838753 31.95266272189349, 21.52640063079335 32.751479289940825, 22.004206171420165 33.55029585798817, 22.742526173823475 34.349112426035504, 18.002783264308967 35.14792899408284, 21.2249289927994 35.94674556213018, 21.988984023324264 36.74556213017752, 23.064738752369202 37.54437869822485, 18.134820287799535 38.34319526627219, 21.12868218849487 39.142011834319526, 21.198870994835048 39.94082840236687, 20.310285237239565 40.7396449704142, 20.945527198877016 41.53846153846154, 19.104748186437735 42.337278106508876, 24.399792129857314 43.13609467455622, 24.588701112081512 43.93491124260355, 23.987151219783247 44.73372781065089, 20.586332725122777 45.532544378698226, 20.9235683566436 46.33136094674556, 21.25720267245278 47.1301775147929, 19.624422258098903 47.928994082840234, 19.53216853269991 48.72781065088758, 19.492389847597767 49.52662721893491, 20.22638437300668 50.32544378698225, 21.530150510604397 51.124260355029584, 19.252491984021347 51.92307692307693, 17.815973873976873 52.72189349112426, 20.08992541508944 53.5207100591716, 22.594368046814832 54.319526627218934, 20.209393795918146 55.11834319526627, 20.510054800945305 55.91715976331361, 21.76969305247549 56.71597633136094, 22.912890406918486 57.514792899408285, 22.461251813793446 58.31360946745562, 23.037916542799906 59.11242603550296, 21.335708105969847 59.91124260355029, 19.765187466447966 60.710059171597635, 20.36159017396665 61.50887573964497, 21.631427360596646 62.307692307692314, 20.405145980977903 63.10650887573964, 17.877044338643973 63.90532544378698, 19.04604236166333 64.70414201183432, 17.66637741200566 65.50295857988165, 20.005921665525943 66.30177514792899, 18.37668150965999 67.10059171597634, 17.496269675629826 67.89940828402366, 26.279255746694076 68.69822485207101, 30.21786908678642 69.49704142011834, 32.20795194993793 70.29585798816568, 31.27910722797048 71.09467455621302, 33.858255434250545 71.89349112426036, 37.48098297128089 72.6923076923077, 35.5337752830347 73.49112426035504, 35.3543074621965 74.28994082840237, 38.030229694437985 75.0887573964497, 38.58223444528792 75.88757396449704, 40.08090273865682 76.68639053254438, 40.78586472492559 77.48520710059172, 44.08316644760627 78.28402366863905, 46.88850908653757 79.0828402366864, 50.0 79.88165680473374, 43.41898415656491 80.68047337278107, 40.50998575915518 81.4792899408284, 39.66018936946957 82.27810650887574, 35.226203588195936 83.07692307692308, 39.262900792743125 83.87573964497041, 44.59131706925604 84.67455621301775, 45.262358021067215 85.4733727810651, 47.490015931725196 86.27218934911244, 46.14258785162995 87.07100591715975, 45.730718337875246 87.8698224852071, 45.41260168363503 88.66863905325444, 43.77907175042644 89.46745562130178, 40.18609424554483 90.26627218934911, 32.34940925236911 91.06508875739645, 32.3984249774505 91.8639053254438, 35.1310465964791 92.66272189349112, 36.86732481242087 93.46153846153845, 34.81613866355567 94.2603550295858, 30.595514686212848 95.05917159763314, 27.723311757622 95.85798816568047, 29.3507064837596 96.65680473372781, 22.74859958680643 97.45562130177515, 22.21791442130814 98.2544378698225, 16.06092738708202 99.05325443786982, 17.815901352566485 99.85207100591715, 17.173905855620124 100.6508875739645, 18.72700512984933 101.44970414201183, 23.84716671935132 102.24852071005917, 24.786242824894686 103.04733727810651, 26.152762869768686 103.84615384615385, 24.412806252141575 104.64497041420118, 28.25087329702438 105.44378698224853, 24.564473631098803 106.24260355029585, 23.536773410356833 107.0414201183432, 24.121244677692633 107.84023668639053, 19.440891980198575 108.63905325443787, 17.46887522562029 109.43786982248521, 12.071600752949866 110.23668639053254, 15.998570675497376 111.03550295857988, 17.231048134711425 111.83431952662723, 18.95972949212449 112.63313609467457, 15.32906703535028 113.43195266272188, 19.81023464784774 114.23076923076923, 21.780895501660343 115.02958579881657, 21.098190847767786 115.82840236686391, 21.7442923631506 116.62721893491124, 14.275138797583352 117.42603550295858, 16.67205082097438 118.22485207100593, 17.44234956782037 119.02366863905324, 18.19282505123356 119.82248520710058, 12.129808323316546 120.62130177514793, 17.518845715269038 121.42011834319527, 20.54871349710698 122.2189349112426, 19.963743030498478 123.01775147928994, 20.852099847897556 123.81656804733728, 17.730968265456966 124.61538461538463, 15.634479101638119 125.41420118343194, 14.607701975391258 126.21301775147928, 17.914989469475216 127.01183431952663, 19.714165082749226 127.81065088757396, 13.83594671807976 128.60946745562129, 13.826720741519672 129.40828402366864, 16.264066896521168 130.20710059171597, 7.2334600088473895 131.0059171597633, 3.4296066914210193 131.80473372781066, 0.0 132.60355029585799, 1.2480644337023625 133.40236686390534, 2.182098383268708 134.20118343195267, 6.248503993171877" fill="none" stroke-width="1.25"/>
    </svg>`,
    marketCap: 87_355,
    price: 0.66543,
    delta: -560,
    volume: {
      [TimePeriod.hour]: 5_000,
      [TimePeriod.day]: 100_000,
      [TimePeriod.week]: 800_000,
      [TimePeriod.month]: 4_920_000,
      [TimePeriod.year]: 100_920_000,
      [TimePeriod.all]: 690_920_000,
    },
  },
}

const useTopTokens = (): UseTopTokensResult => {
  const [data, setData] = useState<TokenData | null>(null)
  const [error, setError] = useState<string | null>(null)
  const [loading, setLoading] = useState(false)

  const fetchTopTokens = async (): Promise<TokenData | void> => {
    const waitRandom = (min: number, max: number): Promise<void> =>
      new Promise((resolve) => setTimeout(resolve, min + Math.round(Math.random() * Math.max(0, max - min))))
    try {
      setLoading(true)
      setError(null)
      await waitRandom(250, 2000)
      if (Math.random() < 0.05) {
        throw new Error('fake error')
      }
      return FAKE_TOP_TOKENS_RESULT
    } catch (e) {
      setError('something went wrong')
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    setLoading(true)
    setError(null)
    fetchTopTokens()
      .then((data) => {
        if (data) setData(data)
      })
      .catch((e) => setError(e))
      .finally(() => setLoading(false))
  }, [])

  return { data, error, loading }
}

export default useTopTokens
