import { Flex, Text, TouchableArea, useSporeColors } from 'ui/src'
import { ArrowDown } from 'ui/src/components/icons/ArrowDown'
import { iconSizes, validColor } from 'ui/src/theme'
import { CurrencyLogo } from 'uniswap/src/components/CurrencyLogo/CurrencyLogo'
import { NetworkLogo } from 'uniswap/src/components/CurrencyLogo/NetworkLogo'
import { UNIVERSE_CHAIN_INFO } from 'uniswap/src/constants/chains'
import { toSupportedChainId } from 'uniswap/src/features/chains/utils'
import { CurrencyInfo } from 'uniswap/src/features/dataApi/types'
import { useLocalizationContext } from 'uniswap/src/features/language/LocalizationContext'
import { useCurrencyInfo } from 'uniswap/src/features/tokens/useCurrencyInfo'
import { BridgeTransactionInfo } from 'uniswap/src/features/transactions/types/transactionDetails'
import { UniverseChainId } from 'uniswap/src/types/chains'
import { useNetworkColors } from 'uniswap/src/utils/colors'
import { getSymbolDisplayText } from 'uniswap/src/utils/currency'
import { ValueText } from 'wallet/src/features/transactions/SummaryCards/DetailsModal/SwapTransactionDetails'
import { useTokenDetailsNavigation } from 'wallet/src/features/transactions/SummaryCards/DetailsModal/hooks'
import { useFormattedCurrencyAmountAndUSDValue } from 'wallet/src/features/transactions/SummaryCards/DetailsModal/utils'

export function BridgeTransactionDetails({
  typeInfo,
  onClose,
  disableClick,
}: {
  typeInfo: BridgeTransactionInfo
  onClose?: () => void
  disableClick?: boolean
}): JSX.Element {
  const colors = useSporeColors()
  const formatter = useLocalizationContext()

  const inputCurrency = useCurrencyInfo(typeInfo.inputCurrencyId)
  const outputCurrency = useCurrencyInfo(typeInfo.outputCurrencyId)

  const { amount: inputAmount, value: inputValue } = useFormattedCurrencyAmountAndUSDValue({
    currency: inputCurrency?.currency,
    currencyAmountRaw: typeInfo.inputCurrencyAmountRaw,
    formatter,
    isApproximateAmount: false,
  })
  const { amount: outputAmount, value: outputValue } = useFormattedCurrencyAmountAndUSDValue({
    currency: outputCurrency?.currency,
    currencyAmountRaw: typeInfo.outputCurrencyAmountRaw,
    formatter,
    isApproximateAmount: false,
  })

  // This should never happen. It's just to keep TS happy.
  if (!inputCurrency || !outputCurrency) {
    throw new Error('Missing required `currencyAmount` to render `TransactionAmountsReview`')
  }

  const onPressInputToken = useTokenDetailsNavigation(inputCurrency, onClose)
  const onPressOutputToken = useTokenDetailsNavigation(outputCurrency, onClose)

  return (
    <Flex $short={{ gap: '$spacing8' }} gap="$spacing16" px="$spacing8" py="$spacing12">
      <TouchableArea onPress={disableClick ? undefined : onPressInputToken}>
        <CurrencyValueWithIcon
          currencyInfo={inputCurrency}
          formattedFiatAmount={inputValue}
          formattedTokenAmount={inputAmount}
        />
      </TouchableArea>

      <ArrowDown color={colors.neutral3.get()} size={20} />

      <TouchableArea onPress={disableClick ? undefined : onPressOutputToken}>
        <CurrencyValueWithIcon
          currencyInfo={outputCurrency}
          formattedFiatAmount={outputValue}
          formattedTokenAmount={outputAmount}
        />
      </TouchableArea>
    </Flex>
  )
}

export function CurrencyValueWithIcon({
  currencyInfo,
  formattedFiatAmount,
  formattedTokenAmount,
}: {
  currencyInfo: CurrencyInfo
  formattedFiatAmount: string
  formattedTokenAmount: string
}): JSX.Element {
  const chainId = toSupportedChainId(currencyInfo.currency.chainId) ?? UniverseChainId.Mainnet
  const networkColors = useNetworkColors(chainId)
  const networkLabel = UNIVERSE_CHAIN_INFO[chainId].label
  const networkColor = validColor(networkColors.foreground)

  return (
    <Flex centered grow row>
      <Flex grow gap="$spacing4">
        <Flex row gap="$spacing4" alignItems="center">
          <NetworkLogo chainId={currencyInfo.currency.chainId} size={iconSizes.icon16} />
          <Text color={networkColor} variant="buttonLabel3">
            {networkLabel}
          </Text>
        </Flex>
        <Text color="$neutral1" variant="heading3">
          {formattedTokenAmount} {getSymbolDisplayText(currencyInfo.currency.symbol)}
        </Text>
        <ValueText value={formattedFiatAmount} />
      </Flex>

      <CurrencyLogo currencyInfo={currencyInfo} size={iconSizes.icon40} />
    </Flex>
  )
}
