import { SVGProps } from 'components/Logo/UniIcon'
import { useMemo } from 'react'
import { useTranslation } from 'react-i18next'
import { Flex, styled, useSporeColors } from 'ui/src'

function Logo({ color, onClick }: { color: string; onClick?: () => void }) {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      width="20"
      height="22"
      viewBox="0 0 20 22"
      fill="none"
      onClick={onClick}
      cursor="pointer"
    >
      <path
        d="M6.04898 1.5328C5.77502 1.49053 5.76347 1.48557 5.89239 1.46587C6.13945 1.42808 6.72283 1.47958 7.12484 1.57466C8.06336 1.79654 8.91735 2.36494 9.82894 3.37442L10.0711 3.64261L10.4176 3.58721C11.8771 3.35393 13.3619 3.53933 14.6038 4.10994C14.9454 4.26692 15.4841 4.5794 15.5514 4.6597C15.5728 4.6853 15.6122 4.85003 15.6389 5.02582C15.7311 5.63398 15.6849 6.10014 15.4979 6.44831C15.3962 6.63778 15.3905 6.69782 15.4589 6.85997C15.5135 6.98936 15.6658 7.08513 15.8166 7.08492C16.1252 7.08451 16.4574 6.58831 16.6113 5.89789L16.6724 5.62364L16.7935 5.76009C17.4579 6.5087 17.9796 7.52962 18.0693 8.25631L18.0926 8.44578L17.981 8.27353C17.7888 7.97714 17.5957 7.77537 17.3484 7.61264C16.9027 7.31931 16.4314 7.21948 15.1833 7.15406C14.056 7.09498 13.418 6.99921 12.7854 6.79404C11.709 6.445 11.1664 5.98015 9.88789 4.31174C9.31999 3.57068 8.96899 3.16067 8.61984 2.83048C7.82646 2.08022 7.04689 1.68675 6.04898 1.5328Z"
        fill={color}
      />
      <path
        d="M15.8056 3.1874C15.834 2.69082 15.9017 2.36329 16.0379 2.06417C16.0917 1.94577 16.1422 1.84887 16.15 1.84887C16.1578 1.84887 16.1344 1.93626 16.0979 2.04305C15.999 2.33335 15.9827 2.73041 16.0509 3.19236C16.1374 3.77851 16.1866 3.86308 16.8095 4.49624C17.1017 4.79322 17.4415 5.16778 17.5647 5.32859L17.7887 5.62099L17.5647 5.41202C17.2908 5.15648 16.6608 4.65812 16.5216 4.58688C16.4283 4.5391 16.4145 4.53992 16.3569 4.5969C16.3039 4.6494 16.2927 4.72829 16.2853 5.10123C16.2739 5.68248 16.1942 6.05556 16.002 6.4286C15.898 6.63037 15.8816 6.58731 15.9757 6.35957C16.046 6.18953 16.0531 6.11478 16.0526 5.55209C16.0515 4.42152 15.9165 4.14972 15.1251 3.68412C14.9247 3.56616 14.5943 3.39606 14.3911 3.30608C14.1878 3.2161 14.0264 3.13773 14.0322 3.13187C14.0547 3.10969 14.8265 3.33374 15.1371 3.45259C15.5992 3.62938 15.6754 3.65229 15.7316 3.63096C15.7692 3.61667 15.7874 3.5077 15.8056 3.1874Z"
        fill={color}
      />
      <path
        d="M6.58113 5.12149C6.02497 4.35993 5.68086 3.19229 5.75536 2.31943L5.77839 2.04932L5.90499 2.0723C6.14272 2.11543 6.55263 2.26718 6.74457 2.38313C7.2713 2.7013 7.49933 3.12019 7.73132 4.19585C7.79928 4.51092 7.88843 4.86746 7.92946 4.98817C7.9955 5.18246 8.24507 5.63629 8.44797 5.93103C8.59412 6.1433 8.49704 6.24389 8.17398 6.21488C7.68059 6.17058 7.01227 5.71183 6.58113 5.12149Z"
        fill={color}
      />
      <path
        d="M15.1311 10.7894C12.532 9.74852 11.6165 8.84507 11.6165 7.32069C11.6165 7.09636 11.6243 6.91281 11.6338 6.91281C11.6432 6.91281 11.7438 6.98684 11.8572 7.07734C12.3843 7.49779 12.9745 7.67736 14.6083 7.91444C15.5698 8.05396 16.1109 8.16664 16.61 8.33127C18.1963 8.85454 19.1778 9.91646 19.4119 11.3629C19.4799 11.7831 19.44 12.5713 19.3297 12.9867C19.2427 13.3147 18.977 13.9061 18.9066 13.9288C18.8871 13.9351 18.8679 13.8606 18.8629 13.7593C18.8361 13.2162 18.5602 12.6874 18.0968 12.2913C17.5699 11.841 16.862 11.4825 15.1311 10.7894Z"
        fill={color}
      />
      <path
        d="M13.3064 11.2218C13.2738 11.029 13.2174 10.7829 13.1809 10.6748L13.1146 10.4782L13.2377 10.6158C13.4082 10.8061 13.5429 11.0496 13.6571 11.3739C13.7442 11.6215 13.754 11.6951 13.7534 12.0973C13.7527 12.4922 13.7418 12.575 13.6614 12.7978C13.5345 13.1492 13.377 13.3983 13.1128 13.6657C12.638 14.1463 12.0276 14.4124 11.1468 14.5228C10.9937 14.5419 10.5474 14.5743 10.1551 14.5945C9.16633 14.6457 8.51558 14.7514 7.93085 14.9556C7.84678 14.985 7.77172 15.0028 7.7641 14.9952C7.74044 14.9718 8.13855 14.7358 8.46739 14.5782C8.93106 14.3561 9.39262 14.2348 10.4268 14.0636C10.9376 13.9789 11.4652 13.8763 11.5991 13.8354C12.8642 13.4496 13.5145 12.454 13.3064 11.2218Z"
        fill={color}
      />
      <path
        d="M14.4979 13.3263C14.1525 12.588 14.0732 11.8751 14.2624 11.2103C14.2827 11.1392 14.3152 11.0811 14.3348 11.0811C14.3544 11.0811 14.4359 11.1249 14.5159 11.1784C14.675 11.285 14.9941 11.4644 15.8444 11.9255C16.9054 12.5009 17.5104 12.9464 17.9218 13.4554C18.2821 13.9012 18.505 14.4089 18.6123 15.028C18.6731 15.3787 18.6375 16.2225 18.547 16.5757C18.2617 17.6891 17.5987 18.5637 16.6531 19.0741C16.5145 19.1488 16.3901 19.2102 16.3767 19.2105C16.3632 19.2108 16.4137 19.0831 16.4889 18.9268C16.807 18.2654 16.8432 17.622 16.6027 16.9059C16.4554 16.4674 16.1552 15.9324 15.5489 15.0282C14.8441 13.9768 14.6713 13.6971 14.4979 13.3263Z"
        fill={color}
      />
      <path
        d="M4.73535 17.3101C5.69986 16.5001 6.89994 15.9246 7.9931 15.748C8.46422 15.6719 9.24904 15.7021 9.68529 15.8132C10.3846 15.9912 11.0101 16.3898 11.3355 16.8648C11.6534 17.329 11.7898 17.7336 11.9318 18.6336C11.9878 18.9887 12.0488 19.3453 12.0672 19.426C12.1739 19.8924 12.3814 20.2653 12.6386 20.4526C13.0471 20.7499 13.7505 20.7684 14.4424 20.5C14.5598 20.4544 14.6618 20.4229 14.669 20.43C14.694 20.4548 14.3456 20.6868 14.0998 20.8089C13.7691 20.9732 13.5061 21.0367 13.1566 21.0367C12.5229 21.0367 11.9967 20.7161 11.5577 20.0623C11.4713 19.9336 11.2771 19.5482 11.1262 19.2059C10.6626 18.1543 10.4337 17.834 9.89554 17.4834C9.42717 17.1784 8.82312 17.1237 8.3687 17.3453C7.77179 17.6364 7.60525 18.3951 8.03277 18.8759C8.20269 19.067 8.51955 19.2318 8.77865 19.2639C9.26337 19.3239 9.67993 18.9571 9.67993 18.4703C9.67993 18.1543 9.5578 17.9739 9.25033 17.8359C8.83039 17.6475 8.379 17.8678 8.38116 18.2601C8.3821 18.4274 8.45535 18.5325 8.62398 18.6084C8.73216 18.6571 8.73467 18.6609 8.64646 18.6427C8.26115 18.5632 8.17088 18.1012 8.48068 17.7945C8.85263 17.4263 9.62176 17.5888 9.88587 18.0914C9.99684 18.3025 10.0097 18.7229 9.91297 18.9768C9.69646 19.545 9.06517 19.8438 8.42476 19.6812C7.98875 19.5705 7.81122 19.4506 7.28553 18.9121C6.37207 17.9762 6.01745 17.7949 4.70055 17.5904L4.44819 17.5512L4.73535 17.3101Z"
        fill={color}
      />
      <path
        fillRule="evenodd"
        clipRule="evenodd"
        d="M0.44926 0.55979C3.4998 4.24228 5.60086 5.76161 5.83435 6.0826C6.02713 6.34765 5.95457 6.58593 5.62431 6.77268C5.44065 6.87651 5.06307 6.98171 4.87401 6.98171C4.66018 6.98171 4.58677 6.89967 4.58677 6.89967C4.46279 6.78271 4.39296 6.80317 3.75628 5.67912C2.87236 4.31495 2.13263 3.18331 2.11245 3.16437C2.06579 3.12055 2.06659 3.12203 3.66615 5.96862C3.92459 6.56192 3.71756 6.7797 3.71756 6.86421C3.71756 7.03611 3.67041 7.12646 3.4572 7.36299C3.10178 7.75736 2.9429 8.20047 2.82821 9.11753C2.69963 10.1455 2.33809 10.8717 1.33613 12.1146C0.749626 12.8421 0.653656 12.9754 0.505663 13.2687C0.319254 13.6379 0.267998 13.8447 0.247224 14.311C0.225267 14.804 0.268031 15.1225 0.419469 15.5939C0.552047 16.0065 0.690435 16.279 1.04422 16.824C1.34953 17.2944 1.52533 17.6439 1.52533 17.7806C1.52533 17.8894 1.54621 17.8895 2.01931 17.7833C3.15151 17.529 4.07085 17.0817 4.58791 16.5337C4.9079 16.1944 4.98303 16.0071 4.98547 15.5422C4.98707 15.2381 4.97631 15.1745 4.89367 14.9995C4.75914 14.7148 4.51424 14.4781 3.97447 14.1111C3.26721 13.6302 2.96514 13.2431 2.88169 12.7107C2.81325 12.2738 2.89265 11.9656 3.28391 11.15C3.68888 10.3058 3.78924 9.94602 3.85713 9.09507C3.90097 8.5453 3.96169 8.32848 4.12051 8.15445C4.28614 7.97297 4.43525 7.91151 4.84517 7.85581C5.51345 7.765 5.93898 7.59304 6.28876 7.27246C6.5922 6.99435 6.71917 6.72638 6.73866 6.32298L6.75345 6.01722L6.58388 5.82059C5.96981 5.10846 0.0380236 0 0.000233728 0C-0.00783924 0 0.194231 0.251923 0.44926 0.55979ZM1.87003 14.8689C2.00887 14.6243 1.9351 14.3099 1.70287 14.1563C1.48343 14.0112 1.14256 14.0795 1.14256 14.2687C1.14256 14.3264 1.17464 14.3684 1.24695 14.4054C1.36871 14.4677 1.37754 14.5378 1.28175 14.681C1.18473 14.826 1.19256 14.9535 1.30384 15.0402C1.48319 15.1799 1.73707 15.103 1.87003 14.8689Z"
        fill={color}
      />
      <path
        fillRule="evenodd"
        clipRule="evenodd"
        d="M7.17519 8.0162C6.86146 8.11204 6.55649 8.44275 6.46209 8.78951C6.4045 9.00105 6.43718 9.37214 6.52344 9.48675C6.6628 9.67184 6.79757 9.72061 7.16249 9.71807C7.87695 9.71311 8.49805 9.40834 8.57025 9.02734C8.62944 8.71503 8.35666 8.28221 7.98092 8.0922C7.78703 7.99419 7.37468 7.9553 7.17519 8.0162ZM8.01039 8.66577C8.12056 8.51006 8.07237 8.34178 7.88498 8.22796C7.52814 8.01124 6.9885 8.19058 6.9885 8.52587C6.9885 8.69277 7.26991 8.87487 7.52786 8.87487C7.69955 8.87487 7.9345 8.77304 8.01039 8.66577Z"
        fill="#FC74FE"
      />
    </svg>
  )
}

function HolidayLogo({ color, onClick }: { color: string; onClick?: () => void }) {
  const { t } = useTranslation()
  const size = 32

  return (
    <svg
      width={size}
      height={size}
      viewBox="0 0 41 41"
      fill="none"
      xmlns="http://www.w3.org/2000/svg"
      onClick={onClick}
      cursor="pointer"
      aria-label={t('common.happyHolidays')}
    >
      <g clipPath="url(#clip0_38_2)">
        <path
          d="M16.4448 0.5C16.4448 0.22386 16.221 0 15.9448 0C15.6687 0 15.4448 0.22386 15.4448 0.5V2.0673L14.0876 1.2837C13.8484 1.14562 13.5426 1.22756 13.4046 1.46671C13.2665 1.70585 13.3484 2.01165 13.5876 2.14972L14.9453 2.9336L13.5888 3.7168C13.3496 3.85487 13.2677 4.16067 13.4058 4.39981C13.5438 4.63896 13.8496 4.7209 14.0888 4.58283L15.4448 3.79991V5.36622C15.4448 5.64236 15.6687 5.86622 15.9448 5.86622C16.221 5.86622 16.4448 5.64236 16.4448 5.36622V3.79935L17.8018 4.58283C18.041 4.7209 18.3468 4.63896 18.4849 4.39981C18.6229 4.16067 18.541 3.85487 18.3018 3.7168L16.9453 2.9336L18.303 2.14972C18.5422 2.01165 18.6241 1.70585 18.486 1.46671C18.348 1.22756 18.0422 1.14562 17.803 1.2837L16.4448 2.06786V0.5Z"
          fill={color}
        />
        <path
          d="M17.2884 17.0901C17.2286 17.3208 17.1243 17.5374 16.9813 17.7277C16.7151 18.0747 16.3628 18.3453 15.9597 18.5125C15.5972 18.6694 15.2132 18.7709 14.8207 18.8134C14.7417 18.8234 14.6599 18.8297 14.5804 18.8358L14.5621 18.8372C14.3135 18.8467 14.0745 18.9359 13.88 19.0917C13.6855 19.2476 13.5458 19.4619 13.4813 19.7032C13.4518 19.8233 13.4298 19.9451 13.4156 20.068C13.3931 20.2525 13.3815 20.4413 13.3689 20.6455L13.3688 20.6482C13.3598 20.7954 13.3502 20.9507 13.3358 21.118C13.2514 21.7996 13.0551 22.4624 12.755 23.0795C12.6937 23.2092 12.6312 23.335 12.5697 23.4585C12.2408 24.1203 11.9434 24.7186 12.0287 25.5194C12.0955 26.1365 12.4102 26.55 12.8283 26.9765C13.0267 27.1802 13.2896 27.3544 13.5626 27.5352L13.5635 27.5358C14.3285 28.0422 15.1719 28.6006 14.894 30.0074C14.6666 31.1473 12.7852 32.3435 10.1408 32.7613C10.3971 32.7222 9.83296 31.755 9.76966 31.6465L9.76576 31.6398C9.69456 31.5277 9.62156 31.4173 9.54876 31.3071L9.54066 31.2948C9.32646 30.9705 9.11326 30.6477 8.94686 30.29C8.50506 29.3515 8.30026 28.2657 8.48126 27.2373C8.64516 26.3068 9.25746 25.5635 9.84706 24.8478C9.94326 24.7311 10.0393 24.6146 10.1322 24.4987C10.921 23.5147 11.7486 22.2254 11.9317 20.9481C11.9472 20.8371 11.961 20.7161 11.9755 20.5888L11.976 20.5844C12.0018 20.3577 12.03 20.1112 12.074 19.8656C12.1397 19.4387 12.2729 19.025 12.4684 18.6402C12.6018 18.3879 12.7775 18.1605 12.9878 17.968C13.0974 17.8658 13.1697 17.7296 13.1932 17.5812C13.2166 17.4329 13.1898 17.2809 13.1171 17.1496L8.90156 9.53322L14.9565 17.0392C15.0255 17.1262 15.1126 17.1969 15.2118 17.2462C15.311 17.2955 15.4198 17.3223 15.5304 17.3247C15.6411 17.3271 15.7509 17.305 15.8521 17.2599C15.9533 17.2149 16.0434 17.148 16.116 17.0641C16.1927 16.9743 16.2362 16.8606 16.2391 16.7422C16.2421 16.6239 16.2043 16.5082 16.1321 16.4146C15.855 16.0589 15.5668 15.6984 15.2797 15.3394L15.266 15.3222C15.148 15.1747 15.0301 15.0272 14.9134 14.8807L13.3897 12.9864L10.3315 9.20412L6.93576 5.16588L10.7238 8.86532L13.9791 12.4808L15.6031 14.2929C15.7511 14.4603 15.899 14.6262 16.0469 14.7921L16.0506 14.7962C16.4402 15.2329 16.8298 15.6698 17.2194 16.1332L17.3078 16.2414L17.3272 16.4092C17.3534 16.6367 17.3403 16.8671 17.2884 17.0901Z"
          fill={color}
        />
        <path
          d="M26.9818 16.7721C26.7603 17.0018 26.6213 17.2939 26.5773 17.6018C26.3601 17.5581 26.1436 17.5063 25.9284 17.4447C25.7129 17.383 25.4974 17.3158 25.2894 17.2347C25.1816 17.1957 25.0814 17.1535 24.9748 17.108C24.8681 17.0625 24.7592 17.0106 24.6515 16.9554C24.2461 16.7323 23.8706 16.4584 23.534 16.1403C22.9226 15.5689 22.4319 14.9269 21.9498 14.2962L21.826 14.1344C21.3333 13.4498 20.8046 12.7922 20.242 12.1643C19.687 11.5492 19.0309 11.0344 18.3024 10.6423C17.5481 10.2606 16.7287 10.0256 15.8875 9.94952C16.7602 9.85432 17.643 9.96252 18.4673 10.2656C19.299 10.59 20.0594 11.075 20.7054 11.6934C21.1272 12.0907 21.523 12.5148 21.8906 12.9631C23.8137 12.5828 25.4815 12.707 26.895 13.1381C26.6531 13.4878 26.5606 13.9375 26.6793 14.3804C26.7168 14.5202 26.7728 14.6501 26.844 14.7681C26.7227 14.9227 26.63 15.1039 26.5759 15.3057C26.43 15.8505 26.6036 16.4056 26.9818 16.7721Z"
          fill={color}
        />
        <path
          d="M33.9255 27.3398C34.7143 26.3096 35.174 23.9847 34.1192 22.0407C33.8755 22.2119 33.5784 22.3125 33.2579 22.3125C32.5739 22.3125 31.9968 21.8547 31.8164 21.2287C31.31 21.3728 30.7426 21.2456 30.3438 20.8469C30.2104 20.7135 30.1073 20.5612 30.0346 20.3986C29.8574 20.417 29.6738 20.404 29.4914 20.3551C28.9461 20.209 28.5518 19.7805 28.4239 19.2691C27.8455 19.4129 27.2228 19.1989 26.8587 18.7082C25.7254 18.5075 24.7209 18.2058 24.0193 17.4534C23.5876 20.771 26.4874 21.9531 29.2596 23.0831C31.69 24.0738 34.0227 25.0247 33.9255 27.3398Z"
          fill={color}
        />
        <path
          d="M18.0908 21.4226C18.8225 21.3522 20.3818 20.9701 19.6846 19.7371C19.5346 19.4863 19.3172 19.2831 19.0574 19.1507C18.7976 19.0184 18.5059 18.9624 18.2158 18.9891C17.9215 19.0209 17.6439 19.1428 17.4209 19.3384C17.1979 19.534 17.0401 19.7937 16.9691 20.0824C16.7525 20.8889 16.982 21.5308 18.0908 21.4226Z"
          fill={color}
        />
        <path
          d="M17.8806 11.6673C17.4216 11.1347 16.7093 10.8554 16.011 10.7537C15.9849 10.9284 15.9684 11.1044 15.9615 11.2809C15.9302 12.7346 16.4442 14.3313 17.4388 15.4409C17.757 15.7996 18.1407 16.0935 18.5692 16.3069C18.817 16.4281 19.4744 16.729 19.7179 16.4584C19.7364 16.4336 19.7481 16.4043 19.7517 16.3735C19.7553 16.3427 19.7507 16.3115 19.7384 16.2831C19.698 16.1672 19.6201 16.062 19.5427 15.9576C19.4878 15.8836 19.4332 15.81 19.3925 15.7331C19.3513 15.6557 19.3072 15.5801 19.2632 15.5047C19.1805 15.3629 19.098 15.2214 19.0347 15.0674C18.8678 14.6646 18.7809 14.2364 18.6943 13.8094L18.6941 13.8084C18.6768 13.723 18.6594 13.637 18.6414 13.5519C18.5089 12.8927 18.3397 12.1999 17.8806 11.6673Z"
          fill={color}
        />
        <path
          d="M27.6881 23.2639C26.976 25.2602 28.1245 26.9463 29.2663 28.6225C30.5434 30.4973 31.8121 32.3599 30.4585 34.6301C33.0889 33.539 34.3378 30.2428 33.2462 27.6275C32.5584 25.9733 30.9006 25.0765 29.3618 24.2442C28.7648 23.9213 28.1856 23.608 27.6881 23.2639Z"
          fill={color}
        />
        <path
          d="M19.991 28.0292C19.5143 28.2246 19.0648 28.4811 18.6537 28.7924C19.5885 28.4518 20.5697 28.2573 21.5632 28.2154C21.741 28.2048 21.92 28.1965 22.1005 28.1881L22.1111 28.1876C22.4256 28.173 22.7456 28.1583 23.0718 28.131C23.6087 28.0944 24.1383 27.9851 24.6461 27.8062C25.1783 27.6192 25.6626 27.316 26.0642 26.9186C26.47 26.5083 26.7546 25.9931 26.8864 25.4301C27.0024 24.8983 26.9861 24.346 26.839 23.8221C26.6918 23.2981 26.4183 22.8187 26.0427 22.4262C26.2239 22.888 26.3354 23.3743 26.3735 23.8692C26.4063 24.3299 26.3439 24.7924 26.1903 25.2277C26.0403 25.6403 25.7951 26.0113 25.4748 26.3102C25.1439 26.6121 24.7608 26.8505 24.3444 27.0138C23.7658 27.2479 23.1114 27.3437 22.4299 27.4435C22.119 27.489 21.8022 27.5355 21.4845 27.5962C20.9736 27.6902 20.4732 27.8353 19.991 28.0292Z"
          fill={color}
        />
        <path
          fillRule="evenodd"
          clipRule="evenodd"
          d="M28.2544 36.2705L28.2034 36.3117C28.0847 36.4076 27.9639 36.5051 27.834 36.5927C27.6659 36.7039 27.4898 36.8024 27.3071 36.8871C26.9266 37.0736 26.5084 37.1688 26.0851 37.1653C24.9386 37.1437 24.1282 36.2864 23.6541 35.3175C23.5298 35.0636 23.4203 34.8019 23.3108 34.5401C23.1355 34.1211 22.9602 33.702 22.7241 33.3149C22.1757 32.4154 21.2371 31.6912 20.138 31.8254C19.6897 31.8817 19.2694 32.0841 19.0205 32.4749C18.3653 33.4957 19.3061 34.9257 20.5054 34.7232C20.6074 34.7076 20.7072 34.68 20.8028 34.641C20.8981 34.6002 20.987 34.5459 21.0668 34.4797C21.2344 34.3395 21.3606 34.1562 21.4321 33.9493C21.5109 33.7335 21.5283 33.5 21.4827 33.2749C21.4335 33.0395 21.2952 32.8327 21.097 32.6979C21.3275 32.8063 21.5071 33.0004 21.5981 33.2391C21.6924 33.4849 21.7167 33.7521 21.6681 34.011C21.621 34.2806 21.5026 34.5326 21.3253 34.7406C21.2312 34.8475 21.1224 34.9405 21.0022 35.0166C20.883 35.0918 20.7552 35.1525 20.6217 35.1974C20.351 35.2903 20.063 35.3203 19.7791 35.2851C19.3805 35.228 19.0046 35.0642 18.6908 34.8109C18.6328 34.7649 18.5772 34.7167 18.5233 34.6667C18.3098 34.4817 18.116 34.2746 17.9451 34.0488C17.8684 33.964 17.7905 33.8803 17.7091 33.7999C17.3249 33.3949 16.8716 33.0624 16.3707 32.8181C16.0253 32.6657 15.6637 32.5531 15.2931 32.4825C15.1066 32.4435 14.9182 32.4154 14.7296 32.3916C14.7102 32.3896 14.6741 32.3833 14.6289 32.3755L14.6208 32.3741C14.4634 32.3466 14.2069 32.3019 14.1617 32.3439C14.7446 31.8049 15.3799 31.3262 16.0582 30.915C16.7547 30.4999 17.5025 30.1785 18.2824 29.9592C19.0909 29.7306 19.9367 29.6654 20.7705 29.7676C21.1997 29.8194 21.6201 29.9286 22.0205 30.0923C22.44 30.2607 22.8272 30.501 23.1649 30.8025C23.499 31.1187 23.7691 31.497 23.9601 31.9163C24.1326 32.309 24.2612 32.7196 24.3437 33.1406C24.3879 33.3672 24.4213 33.6161 24.4557 33.8738C24.6128 35.0481 24.7943 36.4031 26.1401 36.6392C26.2256 36.6559 26.3119 36.6682 26.3987 36.6761L26.667 36.6825C26.8515 36.6693 27.0348 36.6429 27.2155 36.6035C27.5899 36.5151 27.9548 36.39 28.3049 36.2301L28.2544 36.2705ZM18.1137 34.2377C18.0807 34.2014 18.0481 34.1648 18.0159 34.1278L18.1137 34.2377Z"
          fill={color}
        />
        <path
          d="M27.6304 11.5632C27.4924 11.8023 27.5743 12.1081 27.8135 12.2462L29.3664 13.1428L27.9983 13.5095C27.7315 13.581 27.5732 13.8551 27.6447 14.1219C27.7162 14.3886 27.9904 14.5469 28.2571 14.4754L30.5911 13.8499L32.2589 14.8128L30.4867 15.8359L28.1537 15.2111C27.887 15.1396 27.6129 15.2979 27.5414 15.5647C27.47 15.8314 27.6283 16.1056 27.895 16.177L29.2618 16.5431L27.812 17.3802C27.5728 17.5182 27.4909 17.824 27.6289 18.0632C27.767 18.3023 28.0728 18.3843 28.312 18.2462L29.7628 17.4086L29.3961 18.7771C29.3247 19.0439 29.483 19.318 29.7497 19.3895C30.0165 19.4609 30.2906 19.3026 30.3621 19.0359L30.9875 16.7015L32.7574 15.6796V17.7254L31.0503 19.4329C30.855 19.6282 30.8551 19.9448 31.0504 20.14C31.2456 20.3353 31.5622 20.3352 31.7575 20.1399L32.7574 19.1398V20.8128C32.7574 21.0889 32.9813 21.3128 33.2574 21.3128C33.5335 21.3128 33.7574 21.0889 33.7574 20.8128V19.1377L34.7598 20.14C34.955 20.3352 35.2716 20.3352 35.4669 20.14C35.6621 19.9447 35.6621 19.6281 35.4668 19.4329L33.7574 17.7236V15.6779L35.5303 16.7015L36.1556 19.0359C36.2271 19.3026 36.5013 19.4609 36.768 19.3895C37.0347 19.318 37.193 19.0439 37.1216 18.7771L36.7549 17.4086L38.2058 18.2462C38.4449 18.3843 38.7507 18.3023 38.8888 18.0632C39.0268 17.824 38.9449 17.5182 38.7058 17.3802L37.2559 16.5431L38.6227 16.177C38.8894 16.1056 39.0477 15.8314 38.9763 15.5647C38.9049 15.2979 38.6307 15.1396 38.364 15.2111L36.031 15.8359L34.2589 14.8128L35.9266 13.8499L38.2606 14.4754C38.5273 14.5469 38.8015 14.3886 38.873 14.1219C38.9445 13.8551 38.7862 13.581 38.5195 13.5095L37.1513 13.1428L38.7043 12.2462C38.9434 12.1081 39.0253 11.8023 38.8873 11.5632C38.7492 11.324 38.4434 11.2421 38.2043 11.3802L36.6518 12.2765L37.0183 10.9094C37.0898 10.6427 36.9315 10.3685 36.6648 10.297C36.3981 10.2255 36.1239 10.3838 36.0524 10.6505L35.4269 12.9837L33.7574 13.9476V12.0212L35.4668 10.3119C35.6621 10.1166 35.6621 9.80002 35.4669 9.60482C35.2716 9.40952 34.955 9.40952 34.7598 9.60472L33.7574 10.607V8.81282C33.7574 8.53662 33.5335 8.31282 33.2574 8.31282C32.9813 8.31282 32.7574 8.53662 32.7574 8.81282V10.6049L31.7575 9.60482C31.5622 9.40952 31.2456 9.40952 31.0504 9.60472C30.8551 9.79992 30.855 10.1165 31.0503 10.3118L32.7574 12.0193V13.9459L31.0908 12.9837L30.4653 10.6505C30.3939 10.3838 30.1197 10.2255 29.8529 10.297C29.5862 10.3685 29.428 10.6427 29.4995 10.9094L29.8659 12.2765L28.3135 11.3802C28.0743 11.2421 27.7685 11.324 27.6304 11.5632Z"
          fill={color}
        />
        <path
          d="M7.86625 38.6284C7.86625 38.9046 7.64235 39.1284 7.36625 39.1284H5.79836L6.58252 40.4867C6.72059 40.7258 6.63866 41.0316 6.39951 41.1697C6.16036 41.3077 5.85457 41.2258 5.7165 40.9867L4.93261 39.6289L4.14942 40.9855C4.01135 41.2246 3.70555 41.3066 3.4664 41.1685C3.22726 41.0304 3.14532 40.7246 3.28339 40.4855L4.06687 39.1284H2.5C2.22386 39.1284 2 38.9046 2 38.6284C2 38.3523 2.22386 38.1284 2.5 38.1284H4.06631L3.28339 36.7724C3.14532 36.5332 3.22726 36.2275 3.4664 36.0894C3.70555 35.9513 4.01135 36.0332 4.14942 36.2724L4.93262 37.6289L5.7165 36.2712C5.85457 36.0321 6.16036 35.9501 6.39951 36.0882C6.63866 36.2263 6.72059 36.5321 6.58252 36.7712L5.79892 38.1284H7.36625C7.64235 38.1284 7.86625 38.3523 7.86625 38.6284Z"
          fill={color}
        />
      </g>
      <defs>
        <clipPath id="clip0_38_2">
          <rect width="41" height="41" fill="white" />
        </clipPath>
      </defs>
    </svg>
  )
}

const Container = styled(Flex, {
  position: 'relative',
  justifyContent: 'center',
  alignItems: 'center',
  cursor: 'auto',
  variants: {
    clickable: {
      true: { cursor: 'pointer' },
    },
  },
})

type NavIconProps = SVGProps & {
  clickable?: boolean
  onClick?: () => void
}

export const NavIcon = ({ clickable, onClick }: NavIconProps) => {
  const colors = useSporeColors()
  const showHolidayUni = useMemo(() => {
    const date = new Date()
    // months in javascript are 0 indexed...
    const month = date.getMonth() + 1
    const day = date.getDate()
    return month === 12 || (month === 1 && day <= 7)
  }, [])

  return (
    <Container clickable={clickable}>
      {showHolidayUni ? (
        <HolidayLogo color={colors.accent1.val} onClick={onClick} />
      ) : (
        <Logo color={colors.accent1.val} onClick={onClick} />
      )}
    </Container>
  )
}
