import React from 'react'
import styled, { useTheme } from 'styled-components/macro'

import { themeVars, vars } from '../css/sprinkles.css'

// ESLint reports `fill` is missing, whereas it exists on an SVGProps type
type SVGProps = React.SVGProps<SVGSVGElement> & { fill?: string; height?: string | number; width?: string | number }

export const UniIcon = (props: SVGProps) => (
  <svg {...props} fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M20.3526 19.9242C20.2928 20.1549 20.1885 20.3715 20.0455 20.5618C19.7793 20.9088 19.427 21.1794 19.0239 21.3466C18.6614 21.5035 18.2774 21.605 17.8849 21.6475C17.8042 21.6578 17.7206 21.6641 17.6395 21.6703L17.6263 21.6713C17.3777 21.6808 17.1387 21.77 16.9442 21.9258C16.7497 22.0817 16.61 22.296 16.5455 22.5373C16.516 22.6574 16.494 22.7792 16.4798 22.9021C16.4572 23.0874 16.4456 23.2771 16.433 23.4823C16.424 23.6295 16.4144 23.7848 16.4 23.9521C16.3156 24.6337 16.1193 25.2965 15.8192 25.9136C15.7579 26.0433 15.6954 26.1691 15.6339 26.2926C15.3049 26.9546 15.0076 27.5526 15.0929 28.3535C15.1597 28.9706 15.4744 29.3841 15.8925 29.8106C16.0911 30.0146 16.3544 30.1889 16.6277 30.3699C17.3927 30.8763 18.2361 31.4347 17.9582 32.8415C17.7308 33.9814 15.8494 35.1776 13.205 35.5954C13.4615 35.5563 12.8965 34.5879 12.8338 34.4804L12.83 34.4739C12.7571 34.3592 12.6824 34.2462 12.6079 34.1334L12.6079 34.1334L12.6078 34.1334C12.3926 33.8076 12.1783 33.4835 12.0111 33.1241C11.5693 32.1856 11.3645 31.0998 11.5455 30.0714C11.7095 29.1407 12.3221 28.3972 12.9118 27.6813L12.9118 27.6813C13.008 27.5646 13.1035 27.4487 13.1964 27.3328C13.9852 26.3488 14.8128 25.0595 14.9959 23.7822C15.0114 23.6712 15.0252 23.5502 15.0397 23.4229L15.0397 23.4229L15.0397 23.4229C15.0657 23.195 15.0939 22.9469 15.1382 22.6997C15.2039 22.2728 15.3371 21.8591 15.5326 21.4743C15.666 21.222 15.8417 20.9946 16.052 20.8021C16.1616 20.6999 16.2339 20.5637 16.2574 20.4153C16.2808 20.267 16.254 20.115 16.1813 19.9837L11.9658 12.3673L18.0207 19.8733C18.0897 19.9603 18.1768 20.031 18.276 20.0803C18.3752 20.1296 18.484 20.1564 18.5946 20.1588C18.7053 20.1612 18.8151 20.1391 18.9163 20.094C19.0175 20.049 19.1076 19.9821 19.1802 19.8982C19.2569 19.8084 19.3004 19.6947 19.3033 19.5763C19.3063 19.458 19.2685 19.3423 19.1963 19.2487C18.9148 18.8873 18.6218 18.521 18.3302 18.1563L18.33 18.156C18.212 18.0085 18.0943 17.8613 17.9776 17.7148L16.4539 15.8205L13.3957 12.0382L10 8L13.788 11.6994L17.0433 15.3149L18.6673 17.127C18.8165 17.2958 18.9657 17.4631 19.1148 17.6303C19.5044 18.0671 19.894 18.5039 20.2836 18.9673L20.372 19.0755L20.3914 19.2433C20.4176 19.4708 20.4045 19.7012 20.3526 19.9242ZM35.9247 22.4697L35.9311 22.4795C35.93 21.6719 35.4322 20.3394 34.4247 19.0577L34.401 19.0263C34.0906 18.641 33.7524 18.2792 33.3891 17.9438C33.3212 17.8778 33.248 17.8128 33.1736 17.7479C32.7081 17.3401 32.1993 16.985 31.6564 16.6892L31.6176 16.6697C29.9118 15.7366 27.692 15.256 24.9549 15.7972C24.5873 15.3489 24.1914 14.9248 23.7696 14.5275C23.1236 13.9091 22.3632 13.4241 21.5315 13.0997C20.7072 12.7966 19.8244 12.6884 18.9517 12.7836C19.7929 12.8597 20.6123 13.0947 21.3666 13.4764C22.0951 13.8685 22.7512 14.3833 23.3062 14.9984C23.8688 15.6263 24.3976 16.2839 24.8903 16.9685L25.0139 17.1302C25.496 17.7609 25.9868 18.403 26.5982 18.9744C26.9348 19.2925 27.3103 19.5664 27.7157 19.7895C27.8234 19.8447 27.9323 19.8966 28.039 19.9421C28.1456 19.9876 28.2458 20.0298 28.3536 20.0688C28.5616 20.1499 28.7771 20.2171 28.9926 20.2788C29.8547 20.5256 30.7372 20.6143 31.5993 20.662C31.7192 20.6683 31.8389 20.6742 31.9581 20.6801L31.9583 20.6801C32.2662 20.6954 32.5713 20.7105 32.8719 20.7323C33.2837 20.7569 33.6922 20.8214 34.0917 20.925C34.6918 21.0822 35.2201 21.4413 35.5885 21.9425C35.7137 22.1095 35.8261 22.2857 35.9247 22.4697ZM33.4013 17.9451C33.358 17.9049 33.3139 17.8651 33.27 17.8254L33.27 17.8254C33.2418 17.7999 33.2136 17.7745 33.1858 17.7491C33.2073 17.7688 33.2288 17.7887 33.2503 17.8087C33.3009 17.8556 33.3514 17.9025 33.4013 17.9451ZM32.3238 25.9172C29.5516 24.7872 26.6518 23.6051 27.0835 20.2875C28.0091 21.2802 29.462 21.4884 31.0422 21.7149C32.4757 21.9203 34.0139 22.1407 35.3583 22.9753C38.5306 24.9433 38.0673 28.7666 36.9897 30.1739C37.0869 27.8587 34.7544 26.908 32.3238 25.9172ZM21.1551 24.2567C21.8868 24.1863 23.4461 23.8042 22.7489 22.5712C22.5989 22.3204 22.3815 22.1172 22.1217 21.9848C21.8619 21.8525 21.5702 21.7965 21.2801 21.8232C20.9858 21.855 20.7082 21.9769 20.4852 22.1725C20.2622 22.3681 20.1044 22.6278 20.0334 22.9165C19.8168 23.723 20.0463 24.3649 21.1551 24.2567ZM20.9448 14.5014C20.4858 13.9688 19.7735 13.6895 19.0752 13.5878C19.0491 13.7625 19.0326 13.9385 19.0257 14.115C18.9944 15.5687 19.5084 17.1654 20.503 18.275C20.8212 18.6337 21.2049 18.9276 21.6334 19.141C21.8812 19.2622 22.5386 19.5631 22.7821 19.2925C22.8006 19.2677 22.8123 19.2384 22.8159 19.2076C22.8195 19.1768 22.8149 19.1456 22.8026 19.1172C22.7622 19.0013 22.6843 18.8961 22.6069 18.7917C22.552 18.7177 22.4974 18.6441 22.4567 18.5672C22.4155 18.4898 22.3714 18.4142 22.3274 18.3388L22.3274 18.3388C22.2447 18.197 22.1622 18.0555 22.0989 17.9015C21.9319 17.4984 21.845 17.0698 21.7583 16.6425L21.7583 16.6424L21.7583 16.6424L21.7583 16.6423L21.7582 16.6422L21.7582 16.6421L21.7582 16.642C21.7409 16.5566 21.7236 16.4712 21.7056 16.386C21.5731 15.7268 21.4039 15.034 20.9448 14.5014ZM30.7524 26.098C30.0403 28.0943 31.1888 29.7804 32.3306 31.4566C33.6077 33.3315 34.8764 35.194 33.5228 37.4642C36.1532 36.3731 37.4021 33.0769 36.3105 30.4616C35.6227 28.8074 33.9649 27.9106 32.4261 27.0783L32.4261 27.0783L32.426 27.0782C31.829 26.7553 31.2499 26.4421 30.7524 26.098ZM23.0552 30.8633C22.5785 31.0587 22.129 31.3152 21.7179 31.6265C22.6527 31.2859 23.6339 31.0914 24.6274 31.0495C24.8074 31.0388 24.9887 31.0304 25.1715 31.0219L25.1717 31.0219L25.172 31.0219C25.4878 31.0073 25.8085 30.9925 26.136 30.9651C26.6729 30.9285 27.2025 30.8192 27.7103 30.6403C28.2425 30.4533 28.7268 30.1501 29.1284 29.7527C29.5342 29.3424 29.8188 28.8272 29.9506 28.2642C30.0666 27.7324 30.0503 27.1801 29.9032 26.6562C29.756 26.1322 29.4825 25.6528 29.1069 25.2603C29.2881 25.7221 29.3996 26.2084 29.4377 26.7033C29.4705 27.164 29.4081 27.6265 29.2545 28.0618C29.1045 28.4744 28.8593 28.8454 28.539 29.1443C28.2081 29.4462 27.825 29.6846 27.4086 29.8479C26.8299 30.0821 26.1755 30.1779 25.4939 30.2777C25.183 30.3232 24.8664 30.3696 24.5487 30.4303C24.0378 30.5243 23.5374 30.6694 23.0552 30.8633ZM31.3187 39.1046L31.273 39.1415L31.273 39.1416C31.1525 39.2389 31.0301 39.3379 30.8982 39.4268C30.7301 39.538 30.554 39.6365 30.3713 39.7212C29.9908 39.9077 29.5726 40.0029 29.1493 39.9994C28.0028 39.9778 27.1924 39.1205 26.7183 38.1516C26.594 37.8977 26.4845 37.6359 26.375 37.3741L26.375 37.3741C26.1997 36.9551 26.0244 36.5361 25.7883 36.149C25.2399 35.2495 24.3013 34.5253 23.2022 34.6595C22.7539 34.7158 22.3336 34.9182 22.0847 35.309C21.4295 36.3298 22.3703 37.7598 23.5696 37.5573C23.6716 37.5417 23.7714 37.5141 23.867 37.4751C23.9623 37.4343 24.0512 37.38 24.131 37.3138C24.2986 37.1736 24.4248 36.9903 24.4963 36.7834C24.5751 36.5676 24.5926 36.3341 24.547 36.109C24.4978 35.8736 24.3594 35.6668 24.1612 35.532C24.3917 35.6404 24.5713 35.8345 24.6623 36.0732C24.7566 36.319 24.7809 36.5862 24.7323 36.8451C24.6852 37.1147 24.5669 37.3667 24.3896 37.5747C24.2955 37.6816 24.1866 37.7746 24.0664 37.8507C23.9472 37.9259 23.8195 37.9866 23.686 38.0315C23.4153 38.1244 23.1272 38.1544 22.8433 38.1192C22.4447 38.0621 22.0688 37.8983 21.755 37.645C21.697 37.599 21.6414 37.5508 21.5875 37.5008C21.374 37.3158 21.1802 37.1087 21.0093 36.8829C20.9326 36.7981 20.8547 36.7144 20.7733 36.634C20.3891 36.229 19.9358 35.8965 19.4349 35.6522C19.0895 35.4998 18.728 35.3872 18.3574 35.3166C18.1709 35.2776 17.9824 35.2495 17.7938 35.2257C17.7733 35.2236 17.734 35.2167 17.685 35.2082L17.6847 35.2081L17.6846 35.2081L17.6846 35.2081L17.6846 35.2081L17.6845 35.2081C17.5271 35.1806 17.2711 35.136 17.2259 35.178C17.8088 34.639 18.4441 34.1603 19.1224 33.7491C19.8189 33.334 20.5667 33.0126 21.3466 32.7933C22.1551 32.5647 23.0009 32.4995 23.8347 32.6017C24.2639 32.6535 24.6843 32.7627 25.0847 32.9264C25.5042 33.0948 25.8914 33.3351 26.2291 33.6366C26.5632 33.9528 26.8333 34.3311 27.0243 34.7504C27.1968 35.1431 27.3254 35.5537 27.4079 35.9747C27.4521 36.2013 27.4855 36.4502 27.5199 36.7079L27.52 36.7079L27.52 36.708L27.52 36.708C27.6771 37.8823 27.8585 39.2372 29.2043 39.4733C29.2898 39.49 29.3761 39.5023 29.4629 39.5102L29.7312 39.5166C29.9157 39.5034 30.099 39.477 30.2797 39.4376C30.6541 39.3492 31.019 39.2241 31.3691 39.0642L31.3187 39.1046ZM21.0801 36.9619C21.1123 36.9989 21.1449 37.0355 21.1779 37.0718C21.1644 37.0567 21.151 37.0415 21.1375 37.0264L21.1375 37.0264L21.1375 37.0264L21.1375 37.0264C21.1184 37.0049 21.0993 36.9834 21.0801 36.9619Z"
      fill="currentColor"
    />
  </svg>
)

export const ChevronUpIcon = ({
  secondaryColor,
  secondaryWidth,
  secondaryHeight,
  ...props
}: SVGProps & { secondaryWidth?: string; secondaryHeight?: string; secondaryColor?: string }) => (
  <svg
    width={secondaryWidth || '29'}
    height={secondaryHeight || '28'}
    viewBox="0 0 29 28"
    fill="none"
    xmlns="http://www.w3.org/2000/svg"
    {...props}
  >
    <g clipPath="url(#clip0_564_11230)">
      <path
        d="M7.2207 16.0615L13.9092 9.22363C14.1377 8.97754 14.4102 8.86328 14.7178 8.86328C15.0254 8.86328 15.3066 8.98633 15.5352 9.22363L22.2148 16.0615C22.4082 16.2549 22.5137 16.501 22.5137 16.791C22.5137 17.3799 22.0566 17.8369 21.4766 17.8369C21.1953 17.8369 20.9229 17.7314 20.7207 17.5205L14.7266 11.3594L8.71484 17.5205C8.52148 17.7227 8.24902 17.8369 7.95898 17.8369C7.37891 17.8369 6.92188 17.3799 6.92188 16.791C6.92188 16.5098 7.02734 16.2549 7.2207 16.0615Z"
        fill={secondaryColor || themeVars.colors.textSecondary}
      />
    </g>
    <defs>
      <clipPath id="clip0_564_11230">
        <rect width="28" height="28" fill="white" transform="translate(0.716797)" />
      </clipPath>
    </defs>
  </svg>
)

export const BackArrowIcon = (props: SVGProps) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 28 28" {...props}>
    <path d="M5.676 13.645c0 .263.114.527.316.72l5.801 5.792c.21.202.44.3.694.3.554 0 .958-.396.958-.933a.934.934 0 00-.29-.694l-1.977-2.004-2.55-2.329 2.049.123h10.652c.58 0 .985-.404.985-.975 0-.58-.405-.985-.985-.985H10.677l-2.04.123 2.54-2.329 1.978-2.004a.934.934 0 00.29-.694c0-.536-.404-.932-.958-.932-.255 0-.492.097-.72.317l-5.775 5.774a1.012 1.012 0 00-.316.73z" />
  </svg>
)

export const VerifiedIcon = (props: SVGProps) => {
  const theme = useTheme()
  return (
    <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
      <path
        d="M4.52795 13.8056C4.52719 14.4043 4.6712 14.8474 4.95997 15.135C5.24798 15.4233 5.68496 15.5651 6.27091 15.5605H7.57497C7.62945 15.5585 7.68379 15.5676 7.73463 15.5873C7.78547 15.607 7.83176 15.6369 7.87062 15.6752L8.79884 16.5928C9.22054 17.0142 9.63382 17.2237 10.0387 17.2214C10.4436 17.2191 10.8569 17.0096 11.2786 16.5928L12.1954 15.6752C12.2356 15.6365 12.2832 15.6063 12.3354 15.5866C12.3876 15.5669 12.4433 15.558 12.499 15.5605H13.7951C14.3871 15.5613 14.8283 15.4171 15.1186 15.1281C15.4089 14.839 15.5541 14.3959 15.5541 13.7987V12.5014C15.5511 12.389 15.5923 12.2799 15.6687 12.1974L16.5854 11.2798C17.0125 10.86 17.2245 10.4467 17.2214 10.0399C17.2184 9.63305 17.0064 9.21935 16.5854 8.79878L15.6687 7.88115C15.592 7.79886 15.5509 7.68965 15.5541 7.57719V6.2799C15.5533 5.68191 15.4093 5.23878 15.1221 4.95049C14.8348 4.66221 14.3925 4.51806 13.7951 4.51806H12.499C12.4433 4.52036 12.3877 4.51138 12.3355 4.49168C12.2834 4.47197 12.2357 4.44193 12.1954 4.40336L11.2786 3.48574C10.8569 3.06439 10.4436 2.85487 10.0387 2.85717C9.63382 2.85946 9.22054 3.06898 8.79884 3.48574L7.87062 4.40336C7.83164 4.44148 7.78536 4.4713 7.73454 4.49101C7.68373 4.51072 7.62943 4.51993 7.57497 4.51806H6.27091C5.67961 4.51883 5.23995 4.66182 4.95194 4.94705C4.66393 5.23228 4.51992 5.67656 4.51992 6.2799V7.58063C4.52314 7.69309 4.48197 7.80229 4.40533 7.88459L3.48859 8.80222C3.06765 9.22203 2.85718 9.63572 2.85718 10.0433C2.85718 10.4509 3.07033 10.8653 3.49662 11.2867L4.41336 12.2043C4.48979 12.2867 4.53092 12.3958 4.52795 12.5083V13.8056Z"
        fill={theme.accentAction}
      />
      <path
        d="M9.99737 12.4943C9.86205 12.7005 9.6623 12.8164 9.43032 12.8164C9.19191 12.8164 9.00504 12.7198 8.83106 12.4943L7.31036 10.6385C7.20082 10.5032 7.14282 10.3614 7.14282 10.2068C7.14282 9.88458 7.38768 9.63327 7.70342 9.63327C7.89673 9.63327 8.05138 9.70415 8.20603 9.90391L9.40455 11.4311L11.9498 7.34577C12.0851 7.12669 12.2591 7.02359 12.4524 7.02359C12.7553 7.02359 13.0388 7.23623 13.0388 7.55197C13.0388 7.70017 12.9615 7.85482 12.8777 7.99014L9.99737 12.4943Z"
        fill="white"
      />
    </svg>
  )
}

export const PoolIcon = ({ width, height, ...props }: SVGProps) => (
  <svg
    width={width}
    height={height}
    viewBox={`0 0 ${width} ${height}`}
    {...props}
    fill="none"
    xmlns="http://www.w3.org/2000/svg"
  >
    <path
      d="M5.19993 7.5915H12.3999V3.59138C12.3999 1.82406 13.8326 0.39143 15.5999 0.39143C17.3672 0.39143 18.8 1.82406 18.8 3.59138H17.1999C17.1999 2.70774 16.4835 1.99148 15.5999 1.99148C14.7162 1.99148 14 2.70774 14 3.59138V12.3915H12.3999V9.19155H5.19993V10.7915H3.59988V3.59145C3.59988 1.82413 5.03265 0.391498 6.79998 0.391498C8.5673 0.391498 9.99993 1.82413 9.99993 3.59145H8.39988C8.39988 2.70781 7.68362 1.99154 6.79998 1.99154C5.91633 1.99154 5.19993 2.70781 5.19993 3.59145V7.5915ZM1.49707 14.7342L0.902832 13.2487C3.75352 12.1084 6.88055 12.1084 10.2529 13.2325C13.2806 14.2417 16.0201 14.2417 18.5027 13.2487L19.097 14.7342C16.2463 15.8745 13.1192 15.8745 9.7469 14.7503C6.71916 13.7411 3.9797 13.7411 1.49707 14.7342ZM1.49707 18.734L0.902832 17.2485C3.75352 16.1082 6.88055 16.1082 10.2529 17.2324C13.2806 18.2416 16.0201 18.2416 18.5027 17.2485L19.097 18.734C16.2463 19.8743 13.1192 19.8743 9.7469 18.7502C6.71916 17.7411 3.9797 17.7411 1.49707 18.734Z"
      fill="currentColor"
    />
  </svg>
)

export const XMarkIcon = (props: SVGProps) => (
  <svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fill={props.fill ?? themeVars.colors.textSecondary}
      d="M10.2706 27.2148C9.74321 27.7421 9.7181 28.6838 10.2831 29.2362C10.8356 29.7887 11.7773 29.7761 12.3046 29.2488L19.9887 21.5521L27.6855 29.2488C28.2254 29.7887 29.1545 29.7887 29.7069 29.2362C30.2468 28.6712 30.2594 27.7547 29.7069 27.2148L22.0228 19.518L29.7069 11.8339C30.2594 11.294 30.2594 10.3649 29.7069 9.81241C29.1419 9.27251 28.2254 9.25995 27.6855 9.79985L19.9887 17.4966L12.3046 9.79985C11.7773 9.27251 10.823 9.2474 10.2831 9.81241C9.73066 10.3649 9.74321 11.3065 10.2706 11.8339L17.9673 19.518L10.2706 27.2148Z"
    />
  </svg>
)

export const ArrowRightIcon = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" {...props} xmlns="http://www.w3.org/2000/svg">
    <path
      d="M12.7494 7.79771C12.7494 7.64202 12.6842 7.49135 12.5686 7.38086L9.26897 4.08119C9.13839 3.95564 9.00279 3.90039 8.86216 3.90039C8.54074 3.90039 8.30971 4.1264 8.30971 4.43276C8.30971 4.59347 8.375 4.72907 8.47545 4.82952L9.60547 5.97461L11.0619 7.30552L9.89676 7.23521H3.80971C3.47321 7.23521 3.24219 7.46624 3.24219 7.79771C3.24219 8.12416 3.47321 8.35519 3.80971 8.35519H9.89676L11.0619 8.28488L9.60547 9.61579L8.47545 10.7609C8.375 10.8613 8.30971 10.9969 8.30971 11.1576C8.30971 11.464 8.54074 11.69 8.86216 11.69C9.00279 11.69 9.13839 11.6348 9.25893 11.5193L12.5686 8.20954C12.6842 8.09905 12.7494 7.94838 12.7494 7.79771Z"
      fill="currentColor"
    />
  </svg>
)

export const ExternalIcon = (props: SVGProps) => (
  <svg {...props} viewBox="0 0 28 28" xmlns="http://www.w3.org/2000/svg">
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M4.91993 13.9991C4.91993 18.9722 9.03078 23.0758 13.9946 23.0782C13.9953 23.0782 13.9959 23.0782 13.9966 23.0782C13.997 23.0782 13.9974 23.0782 13.9978 23.0782C13.9982 23.0782 13.9986 23.0782 13.999 23.0782C13.9997 23.0782 14.0003 23.0782 14.001 23.0782C18.9648 23.0758 23.0757 18.9722 23.0757 13.9991C23.0757 9.43823 19.5985 5.59653 15.1841 5.00029C14.8058 4.94735 14.4136 4.92 14.0078 4.92C14.0045 4.92 14.0011 4.92 13.9978 4.92C13.9945 4.92 13.9911 4.92 13.9878 4.92C9.36477 4.92 5.49758 8.47078 4.97903 12.9708C4.97912 12.9708 4.97921 12.9707 4.9793 12.9707C4.94009 13.3086 4.91993 13.6518 4.91993 13.9991ZM18.4807 13.3839L18.4814 13.3663C18.4375 12.2764 18.2969 11.2745 18.0596 10.3692C18.7499 10.1516 19.3569 9.87637 19.8424 9.55441C19.9241 9.50192 20.0028 9.4483 20.0786 9.39363C20.9488 10.5186 21.52 11.8985 21.6343 13.3839H18.4807ZM17.2222 14.6143H14.6143V17.0928C14.6176 17.093 14.6211 17.0931 14.6245 17.0933C15.3934 17.1203 16.1361 17.2166 16.8267 17.3477C17.0464 16.5128 17.187 15.5899 17.2222 14.6143ZM14.6232 18.3238C14.6707 18.3262 14.7263 18.3287 14.7883 18.3313C15.3825 18.3611 15.9463 18.4207 16.4663 18.5167C15.9922 19.7809 15.3428 20.7031 14.6143 21.0468V18.3234C14.6172 18.3235 14.6202 18.3237 14.6232 18.3238ZM16.2881 21.304C16.2563 21.3075 16.2267 21.311 16.1996 21.3145C16.7877 20.7176 17.3063 19.8565 17.688 18.8155C18.3032 19.0089 18.8218 19.2462 19.2525 19.545C18.4189 20.3445 17.4042 20.9548 16.2881 21.304ZM13.3814 18.3234V21.0468C12.6529 20.7031 12.0034 19.7809 11.5293 18.5167C12.0999 18.4113 12.7231 18.3498 13.3814 18.3234ZM13.3814 14.6143H10.7734C10.8086 15.5899 10.9492 16.5128 11.1689 17.3477C11.8595 17.2166 12.6022 17.1203 13.3711 17.0933C13.3746 17.0931 13.378 17.093 13.3814 17.0928V14.6143ZM9.51492 13.3839H6.36133C6.47559 11.8985 7.04688 10.5186 7.917 9.39363C7.99279 9.4483 8.07156 9.50191 8.15322 9.5544C8.6387 9.87637 9.24574 10.1516 9.93606 10.3692C9.69875 11.2745 9.55813 12.2764 9.51418 13.3663L9.51492 13.3839ZM18.631 8.85777C18.8652 8.74186 19.0785 8.61524 19.27 8.47957C18.4126 7.63961 17.3548 7.0174 16.166 6.66499C16.7786 7.26197 17.3125 8.138 17.7069 9.22351C17.7978 9.1952 17.8866 9.16558 17.9734 9.13471C18.2063 9.04964 18.4255 8.95751 18.631 8.85777ZM10.2888 9.22351C10.6831 8.138 11.217 7.26197 11.8296 6.66499C10.6408 7.0174 9.58305 7.63961 8.72559 8.47957C8.91713 8.61524 9.13042 8.74186 9.36462 8.85777C9.57011 8.95752 9.78933 9.04965 10.0223 9.13472C10.109 9.16559 10.1979 9.1952 10.2888 9.22351ZM14.6143 6.96014V9.71873C15.2726 9.69231 15.9045 9.63081 16.4839 9.52547C16.0098 8.23491 15.3428 7.28636 14.6143 6.96014ZM13.3814 6.96014C12.6528 7.28636 11.9858 8.23491 11.5117 9.52547C12.0911 9.63081 12.7231 9.69231 13.3814 9.71873V6.96014ZM17.2222 13.3839C17.1871 12.4267 17.0467 11.5046 16.836 10.6878C16.1446 10.8272 15.3925 10.9231 14.6143 10.9493V13.3839H17.2222ZM13.3814 13.3839V10.9493C12.6031 10.9231 11.851 10.8272 11.1596 10.6878C10.949 11.5046 10.8086 12.4267 10.7734 13.3839H13.3814ZM9.52539 14.6143C9.56055 15.7218 9.70996 16.7501 9.94727 17.6641C9.14746 17.919 8.46192 18.2442 7.94336 18.6397C7.05567 17.5059 6.47559 16.1173 6.36133 14.6143H9.52539ZM10.3076 18.8155C10.6943 19.8702 11.2217 20.7403 11.8193 21.338C10.6592 20.9952 9.6045 20.3712 8.74317 19.545C9.17383 19.2462 9.69239 19.0089 10.3076 18.8155ZM18.4702 14.6143C18.4351 15.7218 18.2857 16.7501 18.0484 17.6641C18.8482 17.919 19.5337 18.2442 20.0523 18.6397C20.94 17.5059 21.52 16.1173 21.6343 14.6143H18.4702Z"
      fill={props.fill}
    />
  </svg>
)

export const InstagramIcon = (props: SVGProps) => (
  <svg role="img" viewBox="-3 -3 23 23" xmlns="http://www.w3.org/2000/svg" {...props} fill="none">
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M4.16295 0.285156H11.2695C13.5526 0.285156 15.4305 2.17023 15.4305 4.45603V11.5429C15.4305 13.8313 13.5499 15.7137 11.2695 15.7137H4.16295C1.87988 15.7137 0.00195312 13.8313 0.00195312 11.5429V4.45603C0.00195312 2.16754 1.87988 0.285156 4.16295 0.285156ZM11.859 2.97813C12.0924 2.98133 12.3154 3.0757 12.4804 3.24117C12.6455 3.40663 12.7397 3.63013 12.7429 3.86411C12.7395 4.098 12.6453 4.32136 12.4803 4.48671C12.3152 4.65206 12.0923 4.74636 11.859 4.74956C11.7425 4.75115 11.6269 4.72934 11.519 4.68542C11.4111 4.64149 11.313 4.57634 11.2306 4.49381C11.1482 4.41127 11.0832 4.31303 11.0393 4.20488C10.9954 4.09673 10.9736 3.98086 10.9751 3.86411C10.9735 3.74732 10.9953 3.63139 11.0391 3.52318C11.083 3.41497 11.148 3.31666 11.2304 3.23407C11.3128 3.15147 11.4109 3.08628 11.5188 3.04232C11.6268 2.99836 11.7425 2.97653 11.859 2.97813ZM7.69797 3.79409H7.73451C10.0176 3.79409 11.9326 5.71364 11.9326 8.00213C11.9326 10.3278 10.0176 12.2102 7.73451 12.2102H7.69797C5.4149 12.2102 3.53698 10.3278 3.53698 8.00213C3.53698 5.71364 5.4149 3.79409 7.69797 3.79409ZM7.69797 5.23376H7.73451C9.24438 5.23376 10.4963 6.48867 10.4963 8.00213C10.4963 9.5522 9.24438 10.8071 7.73451 10.8071H7.69797C6.1881 10.8071 4.93615 9.5522 4.93615 8.00213C4.93615 6.48867 6.1881 5.23376 7.69797 5.23376ZM4.19949 1.61386H11.233C11.9945 1.61486 12.7245 1.91856 13.2629 2.45834C13.8013 2.99811 14.1041 3.7299 14.105 4.49319V11.5062C14.104 12.2694 13.8011 13.0011 13.2627 13.5407C12.7243 14.0804 11.9944 14.384 11.233 14.385H4.19949C3.4381 14.384 2.70818 14.0804 2.1698 13.5407C1.63141 13.0011 1.32852 12.2694 1.32752 11.5062V4.49427C1.32837 3.73098 1.63121 2.99919 2.16961 2.45942C2.70801 1.91964 3.43801 1.61594 4.19949 1.61494V1.61386Z"
      fill={props.fill}
    />
  </svg>
)

export const EllipsisIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M10 10.8334C10.4603 10.8334 10.8334 10.4603 10.8334 10C10.8334 9.53978 10.4603 9.16669 10 9.16669C9.5398 9.16669 9.16671 9.53978 9.16671 10C9.16671 10.4603 9.5398 10.8334 10 10.8334Z"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M15.8334 10.8334C16.2936 10.8334 16.6667 10.4603 16.6667 10C16.6667 9.53978 16.2936 9.16669 15.8334 9.16669C15.3731 9.16669 15 9.53978 15 10C15 10.4603 15.3731 10.8334 15.8334 10.8334Z"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M4.16671 10.8334C4.62694 10.8334 5.00004 10.4603 5.00004 10C5.00004 9.53978 4.62694 9.16669 4.16671 9.16669C3.70647 9.16669 3.33337 9.53978 3.33337 10C3.33337 10.4603 3.70647 10.8334 4.16671 10.8334Z"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const LightningBoltIcon = (props: SVGProps) => (
  <svg {...props} width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M6.51562 12.6003C6.51562 12.9092 6.7567 13.1427 7.08817 13.1427H11.4576L9.15988 19.3126C8.851 20.1338 9.70229 20.5707 10.2447 19.9078L17.281 11.1991C17.4166 11.0258 17.4844 10.8676 17.4844 10.6943C17.4844 10.3779 17.2433 10.1519 16.9118 10.1519H12.5424L14.8401 3.97447C15.149 3.15332 14.2977 2.71638 13.7553 3.38686L6.72656 12.0956C6.59096 12.2613 6.51562 12.4195 6.51562 12.6003Z"
      fill="#FB118E"
    />
    <defs>
      <linearGradient
        id="paint0_linear_1845_11847"
        x1="17.4844"
        y1="3.05859"
        x2="1.90385"
        y2="13.0107"
        gradientUnits="userSpaceOnUse"
      >
        <stop stopColor="#4673FA" />
        <stop offset="1" stopColor="#9646FA" />
      </linearGradient>
    </defs>
  </svg>
)

export const SweepIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M13.4177 11.9534C12.3508 11.6675 11.2541 12.3006 10.9682 13.3676C9.90129 13.0817 8.80461 13.7148 8.51873 14.7818L8.25991 15.7477M13.4177 11.9534C14.4846 12.2392 15.1178 13.3359 14.8319 14.4028C15.8989 14.6887 16.532 15.7855 16.2461 16.8524L15.9873 17.8183M13.4177 11.9534L16.0059 2.2941M8.25991 15.7477L15.9873 17.8183M8.25991 15.7477C8.25991 15.7477 7.74227 17.6796 7.48345 18.6455C7.22463 19.6114 5.99989 20.3185 5.99989 20.3185C9.86359 21.3538 12.3131 19.9396 12.3131 19.9396L11.7954 21.8714C13.4053 22.3028 14.9197 21.8027 15.2109 20.716L15.9873 17.8183"
      stroke="currentColor"
      strokeWidth="2"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M5.18229 6.58808C5.25706 6.38601 5.54287 6.38601 5.61764 6.58808C5.99377 7.60457 6.79521 8.406 7.8117 8.78214C8.01377 8.85691 8.01377 9.14272 7.8117 9.21749C6.79521 9.59363 5.99377 10.3951 5.61764 11.4116C5.54286 11.6136 5.25706 11.6136 5.18229 11.4116C4.80615 10.3951 4.00471 9.59363 2.98822 9.21749C2.78615 9.14272 2.78615 8.85691 2.98822 8.78214C4.00471 8.406 4.80615 7.60457 5.18229 6.58808Z"
      fill="currentColor"
    />
  </svg>
)

export const CrossIcon = (props: SVGProps) => (
  <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M9.0304 17.9341C8.66126 18.3032 8.64369 18.9624 9.03919 19.3403C9.41712 19.7358 10.0851 19.7183 10.4542 19.3491L14.7169 15.0864L18.9796 19.3491C19.3576 19.7271 20.0079 19.7358 20.3859 19.3403C20.7814 18.9624 20.7726 18.3032 20.3947 17.9253L16.132 13.6626L20.3947 9.40869C20.7726 9.02197 20.7814 8.37158 20.3859 7.99365C20.0079 7.59814 19.3576 7.60693 18.9796 7.98486L14.7169 12.2476L10.4542 7.98486C10.0851 7.61572 9.41712 7.59814 9.03919 7.99365C8.64369 8.37158 8.66126 9.03076 9.0304 9.3999L13.2931 13.6626L9.0304 17.9341Z"
      fill="currentColor"
    />
  </svg>
)

export const ArrowsIcon = () => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M8.5166 5.71191C8.91211 5.29883 9.52734 5.30762 9.91406 5.71191L14.3438 10.2471C14.5195 10.4229 14.6338 10.6953 14.6338 10.9326C14.6338 11.4863 14.2471 11.8643 13.7021 11.8643C13.4385 11.8643 13.2275 11.7764 13.0518 11.5918L11.2412 9.71094L10.0811 8.375L10.1514 10.2383L10.1514 21.6465C10.1514 22.1914 9.75586 22.5869 9.21094 22.5869C8.66602 22.5869 8.2793 22.1914 8.2793 21.6465L8.2793 10.2383L8.34082 8.375L7.18945 9.71094L5.37891 11.5918C5.20313 11.7764 4.9834 11.8643 4.71973 11.8643C4.18359 11.8643 3.79688 11.4863 3.79688 10.9326C3.79688 10.6953 3.91113 10.4229 4.08691 10.2471L8.5166 5.71191ZM20.1533 22.2793C19.7578 22.6924 19.1426 22.6836 18.7559 22.2793L14.3262 17.7529C14.1504 17.5684 14.0361 17.2959 14.0361 17.0586C14.0361 16.5137 14.4229 16.1357 14.9678 16.1357C15.2227 16.1357 15.4424 16.2236 15.6182 16.3994L17.4287 18.2803L18.5801 19.6162L18.5186 17.7529L18.5186 6.34473C18.5186 5.80859 18.9141 5.4043 19.459 5.4043C19.9951 5.4043 20.3906 5.80859 20.3906 6.34473L20.3906 17.7529L20.3291 19.6162L21.4805 18.2803L23.291 16.3994C23.4668 16.2236 23.6865 16.1357 23.9414 16.1357C24.4863 16.1357 24.873 16.5137 24.873 17.0586C24.873 17.2959 24.7588 17.5684 24.583 17.7529L20.1533 22.2793Z"
      fill="currentColor"
    />
  </svg>
)

export const ReversedArrowsIcon = () => (
  <svg width="20" height="20" viewBox="0 3 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M19.4834 5.71191C19.0879 5.29883 18.4727 5.30762 18.0859 5.71191L13.6562 10.2471C13.4805 10.4229 13.3662 10.6953 13.3662 10.9326C13.3662 11.4863 13.7529 11.8643 14.2979 11.8643C14.5615 11.8643 14.7725 11.7764 14.9482 11.5918L16.7588 9.71094L17.9189 8.375L17.8486 10.2383L17.8486 21.6465C17.8486 22.1914 18.2441 22.5869 18.7891 22.5869C19.334 22.5869 19.7207 22.1914 19.7207 21.6465L19.7207 10.2383L19.6592 8.375L20.8105 9.71094L22.6211 11.5918C22.7969 11.7764 23.0166 11.8643 23.2803 11.8643C23.8164 11.8643 24.2031 11.4863 24.2031 10.9326C24.2031 10.6953 24.0889 10.4229 23.9131 10.2471L19.4834 5.71191ZM7.84668 22.2793C8.24218 22.6924 8.85742 22.6836 9.24414 22.2793L13.6738 17.7529C13.8496 17.5684 13.9639 17.2959 13.9639 17.0586C13.9639 16.5137 13.5771 16.1357 13.0322 16.1357C12.7773 16.1357 12.5576 16.2236 12.3818 16.3994L10.5713 18.2803L9.41992 19.6162L9.48144 17.7529L9.48144 6.34473C9.48144 5.80859 9.08594 5.4043 8.54101 5.4043C8.00488 5.4043 7.60937 5.80859 7.60937 6.34473L7.60937 17.7529L7.6709 19.6162L6.51953 18.2803L4.70898 16.3994C4.5332 16.2236 4.31347 16.1357 4.05859 16.1357C3.51367 16.1357 3.12695 16.5137 3.12695 17.0586C3.12695 17.2959 3.24121 17.5684 3.41699 17.7529L7.84668 22.2793Z"
      fill="currentColor"
    />
  </svg>
)

export const RarityVerifiedIcon = () => (
  <svg width="17" height="17" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M4.5686 3.53439C4.74662 3.33093 5.0038 3.21423 5.27414 3.21423L10.1451 3.21423C10.4154 3.21423 10.6726 3.33093 10.8506 3.53438L13.0437 6.04077C13.3654 6.40839 13.3509 6.96139 13.0104 7.31163L8.38181 12.0724C8.01367 12.4511 7.40558 12.4511 7.03744 12.0724L2.40887 7.31163C2.06836 6.96139 2.05385 6.40839 2.37552 6.04077L4.5686 3.53439Z"
      fill="#4673FA"
    />
    <path
      d="M7.50864 9.74988C7.31466 9.74988 7.15641 9.6669 7.01348 9.47501L5.76283 7.92953C5.67094 7.81544 5.625 7.69097 5.625 7.56131C5.625 7.29682 5.82919 7.079 6.09463 7.079C6.25288 7.079 6.37539 7.14123 6.50301 7.30719L7.48822 8.5778L9.60157 5.14456C9.70877 4.96823 9.85681 4.87488 10.0151 4.87488C10.2703 4.87488 10.5 5.05639 10.5 5.32607C10.5 5.44536 10.4438 5.57501 10.3673 5.69429L7.97827 9.47501C7.86597 9.65653 7.69751 9.74988 7.50864 9.74988Z"
      fill="white"
    />
  </svg>
)

export const BrokenLinkIcon = (props: SVGProps) => (
  <svg width="21" height="21" viewBox="0 0 21 21" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <g clipPath="url(#clip0_79_4612)">
      <path
        d="M14.4344 11.3181L16.9344 8.81813C17.6934 8.03229 18.1133 6.97978 18.1039 5.8873C18.0944 4.79481 17.6562 3.74976 16.8836 2.97722C16.1111 2.20469 15.066 1.76649 13.9735 1.75699C12.8811 1.7475 11.8286 2.16748 11.0427 2.92647L9.60938 4.35147"
        stroke="#98A1C0"
        strokeWidth="1.5"
        strokeLinecap="round"
        strokeLinejoin="round"
      />
      <path
        d="M5.20088 8.75098L2.70088 11.251C1.94189 12.0368 1.52191 13.0893 1.53141 14.1818C1.5409 15.2743 1.9791 16.3194 2.75164 17.0919C3.52417 17.8644 4.56922 18.3026 5.66171 18.3121C6.7542 18.3216 7.80671 17.9016 8.59255 17.1426L10.0175 15.7176"
        stroke="#98A1C0"
        strokeWidth="1.5"
        strokeLinecap="round"
        strokeLinejoin="round"
      />
      <path
        d="M5 3.24316L14.7368 16.6952"
        stroke="#98A1C0"
        strokeWidth="1.5"
        strokeLinecap="round"
        strokeLinejoin="round"
      />
    </g>
    <defs>
      <clipPath id="clip0_79_4612">
        <rect width="20" height="20" fill="white" transform="translate(0.128906 0.0341797)" />
      </clipPath>
    </defs>
  </svg>
)

export const RowsCollpsedIcon = () => (
  <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M13.9912 5.42773C13.71 5.42773 13.4639 5.55957 13.1738 5.78809L7.24121 10.5869C7.03027 10.7627 6.89844 11.0264 6.89844 11.3516C6.89844 11.9492 7.38184 12.3887 7.93555 12.3887C8.19043 12.3887 8.44531 12.292 8.7002 12.0898L14 7.81836L19.291 12.0898C19.5371 12.292 19.8008 12.3887 20.0557 12.3887C20.6094 12.3887 21.0928 11.9492 21.0928 11.3516C21.0928 11.0264 20.9609 10.7627 20.75 10.5869L14.8086 5.78809C14.5273 5.55957 14.2812 5.42773 13.9912 5.42773ZM13.9912 21.7139C14.2812 21.7139 14.5273 21.5908 14.8086 21.3623L20.75 16.5635C20.9609 16.3789 21.0928 16.1152 21.0928 15.7988C21.0928 15.1924 20.6094 14.7529 20.0557 14.7529C19.8008 14.7529 19.5371 14.8584 19.291 15.0605L14 19.332L8.7002 15.0605C8.44531 14.8584 8.19043 14.7529 7.93555 14.7529C7.38184 14.7529 6.89844 15.1924 6.89844 15.7988C6.89844 16.1152 7.03027 16.3789 7.24121 16.5635L13.1738 21.3535C13.4639 21.5908 13.71 21.7139 13.9912 21.7139Z"
      fill="#70757A"
    />
  </svg>
)

export const RowsExpandedIcon = () => (
  <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M13.9912 11.9609C14.2812 11.9609 14.5273 11.8379 14.8086 11.6094L20.75 6.81055C20.9609 6.62598 21.0928 6.3623 21.0928 6.0459C21.0928 5.43945 20.6094 5 20.0557 5C19.8008 5 19.5371 5.10547 19.291 5.30762L14 9.5791L8.7002 5.30762C8.44531 5.10547 8.19043 5 7.93555 5C7.38184 5 6.89844 5.43945 6.89844 6.0459C6.89844 6.3623 7.03027 6.62598 7.24121 6.81055L13.1738 11.6006C13.4639 11.8379 13.71 11.9609 13.9912 11.9609Z"
      fill="#70757A"
    />
    <path
      d="M14.0928 15C13.8115 15 13.5654 15.1318 13.2754 15.3604L7.34277 20.1592C7.13184 20.335 7 20.5986 7 20.9238C7 21.5215 7.4834 21.9609 8.03711 21.9609C8.29199 21.9609 8.54688 21.8643 8.80176 21.6621L14.1016 17.3906L19.3926 21.6621C19.6387 21.8643 19.9023 21.9609 20.1572 21.9609C20.7109 21.9609 21.1943 21.5215 21.1943 20.9238C21.1943 20.5986 21.0625 20.335 20.8516 20.1592L14.9102 15.3604C14.6289 15.1318 14.3828 15 14.0928 15Z"
      fill="#70757A"
    />
  </svg>
)

export const ClockIcon = () => (
  <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M10.7474 4.99984C10.7474 4.58562 10.4116 4.24984 9.9974 4.24984C9.58318 4.24984 9.2474 4.58562 9.2474 4.99984H10.7474ZM9.9974 9.99984H9.2474C9.2474 10.2839 9.4079 10.5436 9.66199 10.6707L9.9974 9.99984ZM12.9953 12.3373C13.3658 12.5226 13.8163 12.3724 14.0015 12.0019C14.1868 11.6314 14.0366 11.1809 13.6661 10.9957L12.9953 12.3373ZM17.5807 9.99984C17.5807 14.188 14.1856 17.5832 9.9974 17.5832V19.0832C15.014 19.0832 19.0807 15.0164 19.0807 9.99984H17.5807ZM9.9974 17.5832C5.80924 17.5832 2.41406 14.188 2.41406 9.99984H0.914062C0.914062 15.0164 4.98081 19.0832 9.9974 19.0832V17.5832ZM2.41406 9.99984C2.41406 5.81168 5.80924 2.4165 9.9974 2.4165V0.916504C4.98081 0.916504 0.914062 4.98325 0.914062 9.99984H2.41406ZM9.9974 2.4165C14.1856 2.4165 17.5807 5.81168 17.5807 9.99984H19.0807C19.0807 4.98325 15.014 0.916504 9.9974 0.916504V2.4165ZM9.2474 4.99984V9.99984H10.7474V4.99984H9.2474ZM9.66199 10.6707L12.9953 12.3373L13.6661 10.9957L10.3328 9.32902L9.66199 10.6707Z"
      fill={vars.color.gray300}
    />
  </svg>
)

export const ApprovedCheckmarkIcon = (props: SVGProps) => (
  <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M13.7464 21.1101C13.3792 21.6796 12.8372 22 12.2076 22C11.5607 22 11.0536 21.733 10.5814 21.1101L6.45464 15.9844C6.15738 15.6107 6 15.2191 6 14.792C6 13.9021 6.66448 13.208 7.52131 13.208C8.0459 13.208 8.46557 13.4038 8.88525 13.9555L12.1377 18.1735L19.0448 6.88988C19.412 6.28476 19.8842 6 20.4087 6C21.2306 6 22 6.58732 22 7.4594C22 7.86874 21.7902 8.29588 21.5628 8.66963L13.7464 21.1101Z"
      fill="currentColor"
    />
  </svg>
)

export const FilterIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="1 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M18.3332 2.5H1.6665L8.33317 10.3833V15.8333L11.6665 17.5V10.3833L18.3332 2.5Z"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const MagnifyingGlassIcon = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M15 15L11.2439 11.2439M12.3821 6.69106C12.3821 9.83414 9.83414 12.3821 6.69106 12.3821C3.54797 12.3821 1 9.83414 1 6.69106C1 3.54797 3.54797 1 6.69106 1C9.83414 1 12.3821 3.54797 12.3821 6.69106Z"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const NavMagnifyingGlassIcon = () => (
  <svg
    viewBox="0 0 24 24"
    width="24"
    height="24"
    fill="none"
    xmlns="http://www.w3.org/2000/svg"
    data-cy="magnifying-icon"
  >
    <path
      d="M10.5 18C14.6421 18 18 14.6421 18 10.5C18 6.35786 14.6421 3 10.5 3C6.35786 3 3 6.35786 3 10.5C3 14.6421 6.35786 18 10.5 18Z"
      stroke="currentColor"
      strokeWidth="2"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path d="M21 21L16 16" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
  </svg>
)

export const BagIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" {...props}>
    <path
      d="M19.2893 5H4.5115C3.34557 5 2.40039 5.89543 2.40039 7V21C2.40039 22.1046 3.34557 23 4.5115 23H19.2893C20.4552 23 21.4004 22.1046 21.4004 21V7C21.4004 5.89543 20.4552 5 19.2893 5Z"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M9 5L9 4.25C9 3.45435 9.31607 2.69129 9.87868 2.12868C10.4413 1.56607 11.2044 1.25 12 1.25C12.7957 1.25 13.5587 1.56607 14.1213 2.12868C14.6839 2.69129 15 3.45435 15 4.25L15 5"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const HundredsOverflowIcon = () => (
  <svg width="8" height="12" viewBox="0 0 8 12" fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M0.781304 6.5554C0.565868 6.5554 0.381209 6.47964 0.227327 6.32812C0.073444 6.17661 -0.00231359 5.99195 5.38039e-05 5.77415C-0.00231359 5.56108 0.073444 5.37879 0.227327 5.22727C0.381209 5.07576 0.565868 5 0.781304 5C0.989637 5 1.17075 5.07576 1.32463 5.22727C1.48088 5.37879 1.56019 5.56108 1.56255 5.77415C1.56019 5.91856 1.52231 6.04995 1.44892 6.16832C1.37789 6.2867 1.2832 6.38139 1.16483 6.45242C1.04882 6.52107 0.920982 6.5554 0.781304 6.5554Z"
      fill="white"
    />
    <path
      d="M3.68169 6.5554C3.46626 6.5554 3.2816 6.47964 3.12772 6.32812C2.97383 6.17661 2.89808 5.99195 2.90044 5.77415C2.89808 5.56108 2.97383 5.37879 3.12772 5.22727C3.2816 5.07576 3.46626 5 3.68169 5C3.89003 5 4.07114 5.07576 4.22502 5.22727C4.38127 5.37879 4.46058 5.56108 4.46294 5.77415C4.46058 5.91856 4.4227 6.04995 4.34931 6.16832C4.27829 6.2867 4.18359 6.38139 4.06522 6.45242C3.94921 6.52107 3.82137 6.5554 3.68169 6.5554Z"
      fill="white"
    />
    <path
      d="M6.58209 6.5554C6.36665 6.5554 6.18199 6.47964 6.02811 6.32812C5.87423 6.17661 5.79847 5.99195 5.80084 5.77415C5.79847 5.56108 5.87423 5.37879 6.02811 5.22727C6.18199 5.07576 6.36665 5 6.58209 5C6.79042 5 6.97153 5.07576 7.12541 5.22727C7.28166 5.37879 7.36097 5.56108 7.36333 5.77415C7.36097 5.91856 7.32309 6.04995 7.2497 6.16832C7.17868 6.2867 7.08398 6.38139 6.96561 6.45242C6.8496 6.52107 6.72176 6.5554 6.58209 6.5554Z"
      fill="white"
    />
  </svg>
)

export const TagIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" {...props}>
    <path
      d="M8.33333 8.33333H8.34097M20.7908 14.2092L14.2183 20.7817C14.0481 20.9521 13.8459 21.0873 13.6233 21.1796C13.4007 21.2719 13.1622 21.3194 12.9213 21.3194C12.6803 21.3194 12.4418 21.2719 12.2192 21.1796C11.9966 21.0873 11.7944 20.9521 11.6242 20.7817L3.75 12.9167V3.75H12.9167L20.7908 11.6242C21.1323 11.9677 21.324 12.4323 21.324 12.9167C21.324 13.401 21.1323 13.8657 20.7908 14.2092Z"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const BarChartIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path d="M12 20V10" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    <path d="M6 20V4" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    <path d="M18 20V16" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
  </svg>
)

export const DiscordIcon = (props: SVGProps) => (
  <svg role="img" width="44" height="44" viewBox="0 0 44 44" xmlns="http://www.w3.org/2000/svg" {...props} fill="none">
    <path
      d="M27.2113 25.4989C25.6574 25.4989 24.3869 24.1067 24.3869 22.3837C24.3869 20.6607 25.63 19.2685 27.2113 19.2685C27.5983 19.2814 27.9789 19.3726 28.331 19.5367C28.6831 19.7007 28.9998 19.9345 29.2625 20.2243C29.5253 20.5141 29.729 20.8543 29.8617 21.225C29.9944 21.5956 30.0536 21.9895 30.0358 22.3837C30.0358 24.0927 28.7892 25.4989 27.2113 25.4989ZM16.7708 25.4989C15.2203 25.4989 13.9464 24.1067 13.9464 22.3837C13.9464 20.6607 15.1929 19.2685 16.7708 19.2685C17.1578 19.2814 17.5384 19.3726 17.8905 19.5367C18.2426 19.7007 18.5593 19.9345 18.822 20.2243C19.0848 20.5141 19.2885 20.8543 19.4212 21.225C19.5539 21.5956 19.6131 21.9895 19.5952 22.3837C19.5987 24.0927 18.3487 25.4989 16.7708 25.4989ZM32.8978 11.402C30.8261 10.4652 28.6454 9.80198 26.4087 9.42847C26.103 9.96986 25.8259 10.5276 25.5788 11.0992C23.2005 10.749 20.7851 10.749 18.4067 11.0992C18.1547 10.5281 17.8743 9.97046 17.5666 9.42847C15.3293 9.80279 13.1484 10.4684 11.0775 11.409C6.97921 17.4167 5.86241 23.2748 6.4191 29.0493C8.83313 30.8106 11.521 32.145 14.3699 32.9964C15.0139 32.1426 15.5852 31.2343 16.0775 30.2815C15.1502 29.9373 14.2533 29.5134 13.3965 29.0145C13.6185 28.8544 13.8405 28.6664 14.0522 28.5098C16.5478 29.6693 19.2586 30.2693 22.0013 30.2693C24.744 30.2693 27.4548 29.6693 29.9504 28.5098C30.1655 28.6838 30.3841 28.8578 30.6061 29.0145C29.7492 29.5157 28.851 29.9396 27.9217 30.2815C28.4139 31.2355 28.9852 32.145 29.6293 32.9999C32.4808 32.1479 35.171 30.8123 37.5869 29.0493C38.2222 22.3559 36.4531 16.55 32.8978 11.402Z"
      fill="currentColor"
    />
  </svg>
)

export const TwitterIcon = (props: SVGProps) => (
  <svg {...props} role="img" viewBox="0 0 44 44" xmlns="http://www.w3.org/2000/svg">
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M21.7923 16.074C20.5417 9.18217 28.4609 5.08251 33.1294 10.0503C33.1294 10.0503 35.2141 9.43965 37.0472 8.21838C36.5107 9.80217 35.4837 11.1529 34.1294 12.0558C34.1294 12.0558 36.6307 11.7074 37.7143 11.0093C37.7143 11.0093 36.6307 13.3645 34.6306 14.2374C35.713 30.0313 19.4592 39.6302 6.28571 31.7758C6.28571 31.7758 12.7884 32.1242 15.5414 28.81C15.5414 28.81 11.5389 29.1643 9.53874 24.0997C9.53874 24.0997 11.0388 24.7103 12.54 23.9249C12.54 23.9249 7.87157 23.1394 7.45393 17.2953C7.45393 17.2953 8.95516 18.5165 10.6223 18.0795C10.6223 18.0795 5.45382 14.8527 8.37161 9.00736C8.37161 9.00736 13.8731 16.3303 21.7934 16.074H21.7923Z"
      fill="currentColor"
    />
  </svg>
)

export const DiscordIconMenu = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M17.0325 6.95914C16.0748 6.51107 15.0508 6.18542 13.9802 6C13.8487 6.23771 13.6951 6.55743 13.5892 6.81177C12.4512 6.64062 11.3236 6.64062 10.2065 6.81177C10.1006 6.55743 9.94354 6.23771 9.81089 6C8.73916 6.18542 7.71399 6.51226 6.7563 6.96152C4.82465 9.8805 4.30101 12.727 4.56283 15.5331C5.844 16.4898 7.08562 17.071 8.30628 17.4513C8.60767 17.0365 8.87646 16.5956 9.10803 16.1309C8.66701 15.9633 8.24461 15.7565 7.84549 15.5164C7.95137 15.438 8.05494 15.356 8.15501 15.2716C10.5894 16.4102 13.2343 16.4102 15.6396 15.2716C15.7408 15.356 15.8444 15.438 15.9491 15.5164C15.5488 15.7577 15.1253 15.9645 14.6842 16.1321C14.9158 16.5956 15.1834 17.0377 15.486 17.4525C16.7078 17.0722 17.9506 16.491 19.2318 15.5331C19.539 12.2801 18.707 9.45977 17.0325 6.95914ZM9.43967 13.8074C8.70891 13.8074 8.10962 13.1251 8.10962 12.2944C8.10962 11.4636 8.69611 10.7802 9.43967 10.7802C10.1833 10.7802 10.7825 11.4624 10.7697 12.2944C10.7709 13.1251 10.1833 13.8074 9.43967 13.8074ZM14.3549 13.8074C13.6242 13.8074 13.0249 13.1251 13.0249 12.2944C13.0249 11.4636 13.6113 10.7802 14.3549 10.7802C15.0985 10.7802 15.6978 11.4624 15.685 12.2944C15.685 13.1251 15.0985 13.8074 14.3549 13.8074Z"
      fill="currentColor"
    />
  </svg>
)

export const TwitterIconMenu = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M18.955 7.34849C18.4393 7.57725 17.885 7.73179 17.3033 7.80132C17.897 7.44542 18.3531 6.88184 18.5678 6.21026C18.012 6.53989 17.3966 6.77916 16.7415 6.90812C16.217 6.34918 15.4695 6 14.6424 6C13.0542 6 11.7665 7.28758 11.7665 8.87576C11.7665 9.10117 11.7919 9.32068 11.841 9.53116C9.45084 9.41124 7.33181 8.26628 5.91338 6.52636C5.66583 6.95109 5.52397 7.44507 5.52397 7.97213C5.52397 8.96989 6.03168 9.85012 6.80335 10.3658C6.33194 10.3509 5.8885 10.2215 5.50077 10.0061C5.50044 10.0181 5.50044 10.0302 5.50044 10.0423C5.50044 11.4356 6.49176 12.5979 7.80738 12.8623C7.56607 12.928 7.312 12.9631 7.04973 12.9631C6.86442 12.9631 6.6843 12.9451 6.50866 12.9116C6.87465 14.054 7.93669 14.8855 9.19518 14.9086C8.21094 15.68 6.97095 16.1398 5.62352 16.1398C5.39139 16.1398 5.16248 16.1261 4.9375 16.0996C6.21022 16.9156 7.72187 17.3917 9.34594 17.3917C14.6357 17.3917 17.5284 13.0096 17.5284 9.20917C17.5284 9.08449 17.5256 8.9605 17.52 8.83708C18.0819 8.43169 18.5694 7.92524 18.955 7.34849H18.955Z"
      fill="currentColor"
    />
  </svg>
)

export const GithubIconMenu = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M12 5.25C8.27062 5.25 5.25 8.34703 5.25 12.1707C5.25 15.2332 7.18219 17.8198 9.86531 18.7368C10.2028 18.7973 10.3294 18.5897 10.3294 18.408C10.3294 18.2437 10.3209 17.6987 10.3209 17.1191C8.625 17.4391 8.18625 16.6952 8.05125 16.3059C7.97531 16.1069 7.64625 15.4927 7.35938 15.3283C7.12312 15.1986 6.78562 14.8785 7.35094 14.8698C7.8825 14.8612 8.26219 15.3716 8.38875 15.5792C8.99625 16.626 9.96656 16.3318 10.3547 16.1502C10.4138 15.7003 10.5909 15.3975 10.785 15.2245C9.28312 15.0515 7.71375 14.4546 7.71375 11.8074C7.71375 11.0548 7.97531 10.4319 8.40563 9.94745C8.33812 9.77443 8.10187 9.06505 8.47312 8.11345C8.47312 8.11345 9.03844 7.93178 10.3294 8.82283C10.8694 8.66711 11.4431 8.58925 12.0169 8.58925C12.5906 8.58925 13.1644 8.66711 13.7044 8.82283C14.9953 7.92313 15.5606 8.11345 15.5606 8.11345C15.9319 9.06505 15.6956 9.77443 15.6281 9.94745C16.0584 10.4319 16.32 11.0461 16.32 11.8074C16.32 14.4632 14.7422 15.0515 13.2403 15.2245C13.485 15.4408 13.6959 15.856 13.6959 16.5048C13.6959 17.4305 13.6875 18.1745 13.6875 18.408C13.6875 18.5897 13.8141 18.806 14.1516 18.7368C16.8178 17.8198 18.75 15.2245 18.75 12.1707C18.75 8.34703 15.7294 5.25 12 5.25Z"
      fill="currentColor"
    />
  </svg>
)

export const CheckMarkIcon = (props: SVGProps) => (
  <svg {...props} fill="none" xmlns="http://www.w3.org/2000/svg">
    <path
      d="M16.6693 5.33789L7.5026 14.3175L3.33594 10.2358"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const GovernanceIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M18.75 16.5V6.00002C18.7512 5.7042 18.6939 5.41106 18.5813 5.13752C18.4686 4.86397 18.3029 4.61544 18.0938 4.40626C17.8846 4.19708 17.636 4.0314 17.3625 3.91876C17.089 3.80613 16.7958 3.74878 16.5 3.75002H3.75"
      stroke="currentColor"
      strokeWidth="2"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path d="M9.75 8.75H15.75" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    <path d="M9.75 12.75H15.75" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    <path
      d="M2.07187 7.49977C1.829 7.22785 1.65585 6.90097 1.56733 6.54728C1.4788 6.19359 1.47756 5.82369 1.56369 5.46941C1.64983 5.11513 1.82077 4.7871 2.0618 4.51353C2.30283 4.23997 2.60673 4.02908 2.94734 3.89902C3.28795 3.76895 3.65507 3.72361 4.01709 3.76689C4.37911 3.81017 4.72519 3.94077 5.02554 4.14747C5.32589 4.35416 5.57151 4.63074 5.74126 4.95342C5.91101 5.27609 5.9998 5.63518 6 5.99977V17.9998C6.00019 18.3644 6.08899 18.7235 6.25874 19.0461C6.42849 19.3688 6.67411 19.6454 6.97446 19.8521C7.27481 20.0588 7.62089 20.1894 7.98291 20.2327C8.34493 20.2759 8.71205 20.2306 9.05266 20.1005C9.39327 19.9705 9.69717 19.7596 9.9382 19.486C10.1792 19.2125 10.3502 18.8844 10.4363 18.5301C10.5224 18.1759 10.5212 17.806 10.4327 17.4523C10.3441 17.0986 10.171 16.7717 9.92813 16.4998H20.4281C20.7177 16.8236 20.9074 17.2243 20.9743 17.6536C21.0411 18.0828 20.9823 18.5223 20.8048 18.9188C20.6274 19.3154 20.3389 19.6521 19.9743 19.8882C19.6096 20.1244 19.1844 20.25 18.75 20.2498H8.25"
      stroke="currentColor"
      strokeWidth="2"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const ChevronLeftIcon = (props: SVGProps) => (
  <svg width="8" height="16" viewBox="0 0 8 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path d="M7 1L1 7L7 13" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
  </svg>
)

export const ChevronRightIcon = styled(ChevronLeftIcon)`
  transform: rotate(180deg);
`

export const TrendingArrow = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M17.5 5.8335H18.25C18.25 5.41928 17.9142 5.0835 17.5 5.0835V5.8335ZM11.0227 12.4307L10.4876 12.9562C10.6286 13.0998 10.8214 13.1807 11.0227 13.1807C11.224 13.1807 11.4169 13.0998 11.5579 12.9562L11.0227 12.4307ZM7.61364 8.9585L8.14881 8.43305C8.00778 8.28941 7.81493 8.2085 7.61364 8.2085C7.41234 8.2085 7.21949 8.28941 7.07846 8.43305L7.61364 8.9585ZM1.96483 13.6414C1.67463 13.937 1.67899 14.4118 1.97456 14.702C2.27013 14.9922 2.74498 14.9878 3.03517 14.6923L1.96483 13.6414ZM13.4091 5.0835C12.9949 5.0835 12.6591 5.41928 12.6591 5.8335C12.6591 6.24771 12.9949 6.5835 13.4091 6.5835V5.0835ZM16.75 10.0002C16.75 10.4144 17.0858 10.7502 17.5 10.7502C17.9142 10.7502 18.25 10.4144 18.25 10.0002H16.75ZM16.9648 5.30805L10.4876 11.9053L11.5579 12.9562L18.0352 6.35894L16.9648 5.30805ZM11.5579 11.9053L8.14881 8.43305L7.07846 9.48394L10.4876 12.9562L11.5579 11.9053ZM7.07846 8.43305L1.96483 13.6414L3.03517 14.6923L8.14881 9.48394L7.07846 8.43305ZM13.4091 6.5835H17.5V5.0835H13.4091V6.5835ZM16.75 5.8335V10.0002H18.25V5.8335H16.75Z"
      fill={vars.color.gray300}
    />
  </svg>
)
export const ActivityListingIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M14.7254 9.93992L9.94537 14.7199C9.82153 14.8439 9.67448 14.9422 9.51262 15.0093C9.35076 15.0764 9.17725 15.111 9.00203 15.111C8.82681 15.111 8.65331 15.0764 8.49145 15.0093C8.32958 14.9422 8.18253 14.8439 8.0587 14.7199L2.33203 8.99992V2.33325H8.9987L14.7254 8.05992C14.9737 8.30974 15.1131 8.64767 15.1131 8.99992C15.1131 9.35217 14.9737 9.6901 14.7254 9.93992V9.93992Z"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path d="M5.66797 5.66675H5.6763" stroke="currentColor" strokeLinecap="round" strokeLinejoin="round" />
  </svg>
)

export const ActivitySaleIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M12.6667 4.00012H3.33333C2.59695 4.00012 2 4.54733 2 5.22234V13.7779C2 14.4529 2.59695 15.0001 3.33333 15.0001H12.6667C13.403 15.0001 14 14.4529 14 13.7779V5.22234C14 4.54733 13.403 4.00012 12.6667 4.00012Z"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M10 6.66675C10 7.19718 9.78929 7.70589 9.41421 8.08096C9.03914 8.45603 8.53043 8.66675 8 8.66675C7.46957 8.66675 6.96086 8.45603 6.58579 8.08096C6.21071 7.70589 6 7.19718 6 6.66675"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M6 4.00012L6 3.50012C6 2.96969 6.21071 2.46098 6.58579 2.08591C6.96086 1.71083 7.46957 1.50012 8 1.50012C8.53043 1.50012 9.03914 1.71083 9.41421 2.08591C9.78929 2.46098 10 2.96969 10 3.50012L10 4.00012"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const ActivityTransferIcon = (props: SVGProps) => (
  <svg fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path d="M14.6654 1.33337L7.33203 8.66671" stroke="currentColor" strokeLinecap="round" strokeLinejoin="round" />
    <path
      d="M14.6654 1.33337L9.9987 14.6667L7.33203 8.66671L1.33203 6.00004L14.6654 1.33337Z"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const ActivityExternalLinkIcon = () => (
  <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
    <mask id="path-1-outside-1_3799_46574" maskUnits="userSpaceOnUse" x="2" y="2" width="15" height="15" fill="black">
      <rect fill="white" x="2" y="2" width="15" height="15" />
      <path
        fillRule="evenodd"
        clipRule="evenodd"
        d="M11.0078 3.5C11.0078 3.22386 11.2317 3 11.5078 3H15.5094L15.5127 3.00001C15.6416 2.99918 15.7709 3.04794 15.8693 3.14631C15.9877 3.26455 16.0344 3.42734 16.0094 3.58074V7.50474C16.0094 7.78088 15.7855 8.00474 15.5094 8.00474C15.2332 8.00474 15.0094 7.78088 15.0094 7.50474V4.71431L8.53338 11.1954C8.3382 11.3908 8.02161 11.3909 7.82627 11.1957C7.63093 11.0005 7.63081 10.6839 7.82599 10.4886L14.3095 4H11.5078C11.2317 4 11.0078 3.77614 11.0078 3.5ZM4.24437 6.24684C4.40078 6.09031 4.61283 6.00244 4.83386 6.00244H8.83544C9.11158 6.00244 9.33544 5.77858 9.33544 5.50244C9.33544 5.2263 9.11158 5.00244 8.83544 5.00244H4.83386C4.34736 5.00244 3.88087 5.19586 3.53699 5.54001C3.19311 5.88416 3 6.35083 3 6.83735V14.1794C3 14.6659 3.19311 15.1326 3.53699 15.4767C3.88087 15.8209 4.34736 16.0143 4.83386 16.0143H12.1701C12.6566 16.0143 13.1231 15.8209 13.467 15.4767C13.8108 15.1326 14.0039 14.6659 14.0039 14.1794V10.1746C14.0039 9.8985 13.7801 9.67464 13.5039 9.67464C13.2278 9.67464 13.0039 9.8985 13.0039 10.1746V14.1794C13.0039 14.4009 12.916 14.6133 12.7596 14.7699C12.6032 14.9264 12.3911 15.0143 12.1701 15.0143H4.83386C4.61283 15.0143 4.40078 14.9264 4.24437 14.7699C4.08795 14.6133 4 14.4009 4 14.1794V6.83735C4 6.61579 4.08795 6.40339 4.24437 6.24684Z"
      />
    </mask>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M11.0078 3.5C11.0078 3.22386 11.2317 3 11.5078 3H15.5094L15.5127 3.00001C15.6416 2.99918 15.7709 3.04794 15.8693 3.14631C15.9877 3.26455 16.0344 3.42734 16.0094 3.58074V7.50474C16.0094 7.78088 15.7855 8.00474 15.5094 8.00474C15.2332 8.00474 15.0094 7.78088 15.0094 7.50474V4.71431L8.53338 11.1954C8.3382 11.3908 8.02161 11.3909 7.82627 11.1957C7.63093 11.0005 7.63081 10.6839 7.82599 10.4886L14.3095 4H11.5078C11.2317 4 11.0078 3.77614 11.0078 3.5ZM4.24437 6.24684C4.40078 6.09031 4.61283 6.00244 4.83386 6.00244H8.83544C9.11158 6.00244 9.33544 5.77858 9.33544 5.50244C9.33544 5.2263 9.11158 5.00244 8.83544 5.00244H4.83386C4.34736 5.00244 3.88087 5.19586 3.53699 5.54001C3.19311 5.88416 3 6.35083 3 6.83735V14.1794C3 14.6659 3.19311 15.1326 3.53699 15.4767C3.88087 15.8209 4.34736 16.0143 4.83386 16.0143H12.1701C12.6566 16.0143 13.1231 15.8209 13.467 15.4767C13.8108 15.1326 14.0039 14.6659 14.0039 14.1794V10.1746C14.0039 9.8985 13.7801 9.67464 13.5039 9.67464C13.2278 9.67464 13.0039 9.8985 13.0039 10.1746V14.1794C13.0039 14.4009 12.916 14.6133 12.7596 14.7699C12.6032 14.9264 12.3911 15.0143 12.1701 15.0143H4.83386C4.61283 15.0143 4.40078 14.9264 4.24437 14.7699C4.08795 14.6133 4 14.4009 4 14.1794V6.83735C4 6.61579 4.08795 6.40339 4.24437 6.24684Z"
      fill="currentColor"
    />
    <path
      d="M15.5094 3L15.51 2.8H15.5094V3ZM15.5127 3.00001L15.512 3.20002L15.514 3.20001L15.5127 3.00001ZM15.8693 3.14631L16.0107 3.00483V3.00483L15.8693 3.14631ZM16.0094 3.58074L15.812 3.54861C15.8103 3.55923 15.8094 3.56998 15.8094 3.58074H16.0094ZM15.0094 4.71431H15.2094C15.2094 4.6334 15.1606 4.56047 15.0859 4.52952C15.0111 4.49857 14.9251 4.51571 14.8679 4.57294L15.0094 4.71431ZM8.53338 11.1954L8.67486 11.3368V11.3368L8.53338 11.1954ZM7.82627 11.1957L7.68491 11.3372L7.82627 11.1957ZM7.82599 10.4886L7.96747 10.63L7.82599 10.4886ZM14.3095 4L14.451 4.14137C14.5081 4.08415 14.5252 3.99814 14.4942 3.92343C14.4633 3.84871 14.3904 3.8 14.3095 3.8V4ZM4.24437 6.24684L4.38585 6.38821H4.38585L4.24437 6.24684ZM3.53699 5.54001L3.39551 5.39865H3.39551L3.53699 5.54001ZM3.53699 15.4767L3.67846 15.3354L3.67846 15.3354L3.53699 15.4767ZM13.467 15.4767L13.6084 15.6181L13.467 15.4767ZM12.7596 14.7699L12.6181 14.6285L12.6181 14.6285L12.7596 14.7699ZM4.24437 14.7699L4.10289 14.9113H4.10289L4.24437 14.7699ZM11.5078 2.8C11.1212 2.8 10.8078 3.1134 10.8078 3.5H11.2078C11.2078 3.33431 11.3421 3.2 11.5078 3.2V2.8ZM15.5094 2.8H11.5078V3.2H15.5094V2.8ZM15.5133 2.80001L15.51 2.8L15.5087 3.2L15.512 3.20001L15.5133 2.80001ZM16.0107 3.00483C15.8729 2.86715 15.6916 2.79885 15.5114 2.80001L15.514 3.20001C15.5917 3.1995 15.6689 3.22873 15.728 3.28778L16.0107 3.00483ZM16.2068 3.61287C16.2417 3.39869 16.1765 3.17048 16.0107 3.00483L15.728 3.28778C15.7989 3.35862 15.8271 3.456 15.812 3.54861L16.2068 3.61287ZM16.2094 7.50474V3.58074H15.8094V7.50474H16.2094ZM15.5094 8.20474C15.896 8.20474 16.2094 7.89134 16.2094 7.50474H15.8094C15.8094 7.67042 15.6751 7.80474 15.5094 7.80474V8.20474ZM14.8094 7.50474C14.8094 7.89134 15.1228 8.20474 15.5094 8.20474V7.80474C15.3437 7.80474 15.2094 7.67042 15.2094 7.50474H14.8094ZM14.8094 4.71431V7.50474H15.2094V4.71431H14.8094ZM8.67486 11.3368L15.1509 4.85568L14.8679 4.57294L8.3919 11.0541L8.67486 11.3368ZM7.68491 11.3372C7.95838 11.6105 8.4016 11.6103 8.67486 11.3368L8.3919 11.0541C8.27479 11.1713 8.08484 11.1713 7.96764 11.0542L7.68491 11.3372ZM7.68452 10.3472C7.41126 10.6207 7.41143 11.0639 7.68491 11.3372L7.96764 11.0542C7.85044 10.9371 7.85036 10.7472 7.96747 10.63L7.68452 10.3472ZM14.168 3.85863L7.68452 10.3472L7.96747 10.63L14.451 4.14137L14.168 3.85863ZM11.5078 4.2H14.3095V3.8H11.5078V4.2ZM10.8078 3.5C10.8078 3.8866 11.1212 4.2 11.5078 4.2V3.8C11.3421 3.8 11.2078 3.66569 11.2078 3.5H10.8078ZM4.83386 5.80244C4.55974 5.80244 4.2968 5.91142 4.10289 6.10548L4.38585 6.38821C4.50476 6.2692 4.66593 6.20244 4.83386 6.20244V5.80244ZM8.83544 5.80244H4.83386V6.20244H8.83544V5.80244ZM9.13544 5.50244C9.13544 5.66813 9.00112 5.80244 8.83544 5.80244V6.20244C9.22204 6.20244 9.53544 5.88904 9.53544 5.50244H9.13544ZM8.83544 5.20244C9.00112 5.20244 9.13544 5.33676 9.13544 5.50244H9.53544C9.53544 5.11584 9.22204 4.80244 8.83544 4.80244V5.20244ZM4.83386 5.20244H8.83544V4.80244H4.83386V5.20244ZM3.67846 5.68138C3.98485 5.37475 4.40046 5.20244 4.83386 5.20244V4.80244C4.29427 4.80244 3.77689 5.01697 3.39551 5.39865L3.67846 5.68138ZM3.2 6.83735C3.2 6.40382 3.37208 5.988 3.67846 5.68138L3.39551 5.39865C3.01415 5.78031 2.8 6.29784 2.8 6.83735H3.2ZM3.2 14.1794V6.83735H2.8V14.1794H3.2ZM3.67846 15.3354C3.37208 15.0287 3.2 14.6129 3.2 14.1794H2.8C2.8 14.7189 3.01415 15.2364 3.39551 15.6181L3.67846 15.3354ZM4.83386 15.8143C4.40046 15.8143 3.98485 15.642 3.67846 15.3354L3.39551 15.6181C3.77689 15.9998 4.29427 16.2143 4.83386 16.2143V15.8143ZM12.1701 15.8143H4.83386V16.2143H12.1701V15.8143ZM13.3255 15.3354C13.0191 15.642 12.6035 15.8143 12.1701 15.8143V16.2143C12.7097 16.2143 13.2271 15.9998 13.6084 15.6181L13.3255 15.3354ZM13.8039 14.1794C13.8039 14.6129 13.6319 15.0287 13.3255 15.3354L13.6084 15.6181C13.9898 15.2364 14.2039 14.7189 14.2039 14.1794H13.8039ZM13.8039 10.1746V14.1794H14.2039V10.1746H13.8039ZM13.5039 9.87464C13.6696 9.87464 13.8039 10.009 13.8039 10.1746H14.2039C14.2039 9.78804 13.8905 9.47464 13.5039 9.47464V9.87464ZM13.2039 10.1746C13.2039 10.009 13.3383 9.87464 13.5039 9.87464V9.47464C13.1173 9.47464 12.8039 9.78804 12.8039 10.1746H13.2039ZM13.2039 14.1794V10.1746H12.8039V14.1794H13.2039ZM12.9011 14.9113C13.095 14.7172 13.2039 14.4539 13.2039 14.1794H12.8039C12.8039 14.3479 12.737 14.5095 12.6181 14.6285L12.9011 14.9113ZM12.1701 15.2143C12.4442 15.2143 12.7071 15.1053 12.9011 14.9113L12.6181 14.6285C12.4992 14.7475 12.338 14.8143 12.1701 14.8143V15.2143ZM4.83386 15.2143H12.1701V14.8143H4.83386V15.2143ZM4.10289 14.9113C4.2968 15.1053 4.55974 15.2143 4.83386 15.2143V14.8143C4.66593 14.8143 4.50476 14.7475 4.38585 14.6285L4.10289 14.9113ZM3.8 14.1794C3.8 14.4539 3.90898 14.7172 4.10289 14.9113L4.38585 14.6285C4.26691 14.5095 4.2 14.3479 4.2 14.1794H3.8ZM3.8 6.83735V14.1794H4.2V6.83735H3.8ZM4.10289 6.10548C3.90898 6.29954 3.8 6.5628 3.8 6.83735H4.2C4.2 6.66879 4.26691 6.50724 4.38585 6.38821L4.10289 6.10548Z"
      fill="currentColor"
      mask="url(#path-1-outside-1_3799_46574)"
    />
  </svg>
)

export const LargeTagIcon = (props: SVGProps) => (
  <svg width="96" height="96" viewBox="0 0 96 96" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M55.6215 80.6815L55.621 80.6819C55.0823 81.2212 54.4427 81.649 53.7386 81.9409C53.0345 82.2328 52.2797 82.383 51.5175 82.383C50.7553 82.383 50.0006 82.2328 49.2965 81.9409C48.5923 81.649 47.9527 81.2212 47.414 80.6819L47.413 80.681L17.7 51.0025V17.7H51.0029L80.714 47.411C80.7141 47.4111 80.7142 47.4112 80.7143 47.4113C81.7943 48.498 82.4006 49.9679 82.4006 51.5C82.4006 53.032 81.7944 54.5017 80.7146 55.5884C80.7144 55.5886 80.7142 55.5888 80.714 55.589L55.6215 80.6815Z"
      stroke="currentColor"
      strokeWidth="2.4"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <circle cx="34" cy="34" r="3" fill="currentColor" />
  </svg>
)

export const CircularCloseIcon = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <circle cx="8" cy="8" r="8" fill="#293249" />
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M4.64645 4.64645C4.84171 4.45118 5.15829 4.45118 5.35355 4.64645L8 7.29289L10.6464 4.64645C10.8417 4.45118 11.1583 4.45118 11.3536 4.64645C11.5488 4.84171 11.5488 5.15829 11.3536 5.35355L8.70711 8L11.3536 10.6464C11.5488 10.8417 11.5488 11.1583 11.3536 11.3536C11.1583 11.5488 10.8417 11.5488 10.6464 11.3536L8 8.70711L5.35355 11.3536C5.15829 11.5488 4.84171 11.5488 4.64645 11.3536C4.45118 11.1583 4.45118 10.8417 4.64645 10.6464L7.29289 8L4.64645 5.35355C4.45118 5.15829 4.45118 4.84171 4.64645 4.64645Z"
      fill="white"
    />
  </svg>
)

export const SquareArrowDownIcon = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M4.66797 4.66663L11.3346 11.3333M11.3346 11.3333V4.66663M11.3346 11.3333H4.66797"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const SquareArrowUpIcon = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M4.66797 11.3334L11.3346 4.66671M11.3346 4.66671V11.3334M11.3346 4.66671H4.66797"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const CloseTimerIcon = (props: SVGProps) => (
  <svg width="11" height="12" viewBox="0 0 11 12" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M8.2435 3.95892C8.53639 3.66603 8.53639 3.19115 8.2435 2.89826C7.95061 2.60537 7.47573 2.60537 7.18284 2.89826L8.2435 3.95892ZM2.03998 8.04112C1.74709 8.33401 1.74709 8.80888 2.03998 9.10178C2.33288 9.39467 2.80775 9.39467 3.10064 9.10178L2.03998 8.04112ZM3.10064 2.89826C2.80775 2.60537 2.33288 2.60537 2.03998 2.89826C1.74709 3.19115 1.74709 3.66603 2.03998 3.95892L3.10064 2.89826ZM7.18284 9.10178C7.47573 9.39467 7.95061 9.39467 8.2435 9.10178C8.53639 8.80888 8.53639 8.33401 8.2435 8.04112L7.18284 9.10178ZM7.18284 2.89826L2.03998 8.04112L3.10064 9.10178L8.2435 3.95892L7.18284 2.89826ZM2.03998 3.95892L7.18284 9.10178L8.2435 8.04112L3.10064 2.89826L2.03998 3.95892Z"
      fill="currentColor"
    />
  </svg>
)

export const ChevronDownBagIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M5 7.5L10 12.5L15 7.5"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const ChevronUpBagIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M5 12.5L10 7.5L15 12.5"
      stroke="currentColor"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const BagCloseIcon = (props: SVGProps) => (
  <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M18.7071 6.70711C19.0976 6.31658 19.0976 5.68342 18.7071 5.29289C18.3166 4.90237 17.6834 4.90237 17.2929 5.29289L18.7071 6.70711ZM5.29289 17.2929C4.90237 17.6834 4.90237 18.3166 5.29289 18.7071C5.68342 19.0976 6.31658 19.0976 6.70711 18.7071L5.29289 17.2929ZM6.70711 5.29289C6.31658 4.90237 5.68342 4.90237 5.29289 5.29289C4.90237 5.68342 4.90237 6.31658 5.29289 6.70711L6.70711 5.29289ZM17.2929 18.7071C17.6834 19.0976 18.3166 19.0976 18.7071 18.7071C19.0976 18.3166 19.0976 17.6834 18.7071 17.2929L17.2929 18.7071ZM17.2929 5.29289L5.29289 17.2929L6.70711 18.7071L18.7071 6.70711L17.2929 5.29289ZM5.29289 6.70711L17.2929 18.7071L18.7071 17.2929L6.70711 5.29289L5.29289 6.70711Z"
      fill="currentColor"
    />
  </svg>
)

export const CancelListingIcon = (props: SVGProps) => (
  <svg width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      d="M12.6667 6L13.3933 6.72667C13.6417 6.97648 13.7811 7.31442 13.7811 7.66667C13.7811 8.01891 13.6417 8.35685 13.3933 8.60667L8.61333 13.3867C8.4895 13.5106 8.34245 13.609 8.18059 13.6761C8.01872 13.7432 7.84522 13.7777 7.67 13.7777C7.49478 13.7777 7.32128 13.7432 7.15941 13.6761C6.99755 13.609 6.8505 13.5106 6.72667 13.3867L6 12.6667M8.8 2.13333L7.66667 1H1V7.66667L2.13333 8.8M13.7811 1L1.10498 13.6761M4.33333 4.33333H4.33889"
      stroke="currentColor"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const ListingModalWindowActive = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <circle cx="8" cy="8" r="8" fill={props.fill ? props.fill : themeVars.colors.accentAction} fillOpacity="0.24" />
    <circle cx="8" cy="8" r="5" fill={props.fill ? props.fill : themeVars.colors.accentAction} />
  </svg>
)

export const ListingModalWindowClosed = (props: SVGProps) => (
  <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <circle cx="8" cy="8" r="7" stroke="#333D59" strokeWidth="2" />
  </svg>
)

export const OpenSeaMarketplaceIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M24 16.06V14.806C24 14.69 23.888 14.606 23.778 14.638L17.856 16.35C17.824 16.358 17.796 16.376 17.774 16.4C17.1396 17.1008 16.6005 17.571 16.4578 17.6955L16.448 17.704C16.08 18.016 15.624 18.186 15.144 18.186H14.093C13.4894 18.186 13 17.6966 13 17.093C13 16.4894 13.4894 16 14.093 16H14.704C14.748 16 14.79 15.984 14.822 15.956L15.042 15.754C15.136 15.668 15.248 15.564 15.382 15.43C15.3933 15.4187 15.4047 15.4073 15.4163 15.3958C15.4868 15.3256 15.5621 15.2505 15.636 15.168C15.724 15.082 15.81 14.986 15.89 14.892C16.024 14.748 16.152 14.598 16.286 14.44C16.382 14.336 16.47 14.218 16.556 14.1C16.652 13.988 16.746 13.862 16.834 13.742C16.8666 13.6941 16.9013 13.6457 16.9367 13.5963C16.9708 13.5486 17.0057 13.5 17.04 13.45C17.104 13.354 17.168 13.252 17.222 13.156C17.39 12.896 17.532 12.618 17.652 12.34C17.707 12.2211 17.751 12.096 17.7937 11.9743C17.7992 11.9588 17.8046 11.9434 17.81 11.928C17.858 11.786 17.898 11.652 17.928 11.51C18 11.176 18.016 10.844 17.984 10.512C17.9764 10.4136 17.9688 10.317 17.9477 10.2255C17.9454 10.2152 17.944 10.2046 17.944 10.194C17.936 10.126 17.92 10.05 17.898 9.98001C17.826 9.65601 17.714 9.332 17.572 9.014C17.524 8.89599 17.468 8.77599 17.414 8.66598C17.286 8.42802 17.152 8.19001 17 7.96C16.9695 7.91136 16.9357 7.86209 16.902 7.81289C16.8762 7.77511 16.8503 7.73737 16.826 7.70002C16.7297 7.5514 16.6213 7.40815 16.5163 7.26916C16.4926 7.2379 16.4692 7.20686 16.446 7.17602C16.384 7.09458 16.3161 7.01314 16.2477 6.93116C16.2103 6.88629 16.1728 6.84127 16.136 6.79599C16.032 6.66998 15.93 6.54998 15.826 6.43201C15.454 6.01201 15.064 5.63198 14.716 5.30802C14.652 5.24399 14.582 5.18001 14.51 5.11798C14.24 4.87201 13.994 4.65801 13.788 4.49198C13.726 4.44425 13.6703 4.39722 13.6185 4.35345C13.5835 4.32387 13.5503 4.29579 13.518 4.26998C13.4545 4.22272 13.3996 4.18086 13.3537 4.14585C13.3258 4.12459 13.3012 4.10585 13.28 4.08998C13.264 4.078 13.246 4.06999 13.228 4.06398C13.0932 4.02615 13 3.90323 13 3.76322V2.11201C13 1.80401 12.876 1.52802 12.678 1.326C12.48 1.12398 12.204 1 11.9 1C11.292 1 10.8 1.498 10.8 2.11201V3.2656C10.8 3.32504 10.7432 3.36806 10.686 3.35198L10.376 3.26399L10.102 3.18821C10.1004 3.18775 10.0987 3.18716 10.097 3.18657C10.0934 3.18529 10.0898 3.18401 10.086 3.18401C10.082 3.18401 10.078 3.18348 10.0742 3.18244L7.93999 2.60399C7.84603 2.578 7.766 2.68 7.81402 2.766L8.15602 3.39801C8.17546 3.44665 8.2001 3.4953 8.22543 3.54529C8.24175 3.57751 8.25835 3.61028 8.27403 3.64398C8.33001 3.75602 8.38603 3.87399 8.44002 3.992C8.48799 4.09599 8.53601 4.198 8.59203 4.30999C8.61561 4.36275 8.63965 4.41614 8.66403 4.4703C8.75336 4.6687 8.8473 4.87732 8.94 5.10202L8.94079 5.10389C9.02051 5.29326 9.10024 5.48265 9.17001 5.68C9.36199 6.178 9.54402 6.70999 9.70201 7.25601C9.7413 7.37805 9.7727 7.49617 9.80452 7.61587C9.81806 7.66682 9.83168 7.71806 9.84601 7.77001L9.86799 7.866C9.93201 8.12001 9.98799 8.372 10.028 8.62601C10.06 8.8 10.09 8.96598 10.106 9.134L10.106 9.13407C10.13 9.32404 10.154 9.51401 10.162 9.70398C10.178 9.87801 10.186 10.06 10.186 10.234C10.186 10.678 10.146 11.106 10.052 11.51C10.0462 11.5316 10.0403 11.5534 10.0344 11.5755C10.008 11.6739 9.98068 11.776 9.94802 11.874C9.91838 11.9792 9.87997 12.0844 9.84008 12.1937C9.82613 12.2319 9.812 12.2706 9.798 12.31C9.7957 12.3162 9.7934 12.3224 9.7911 12.3286C9.76138 12.4087 9.73114 12.4902 9.694 12.57C9.49601 13.046 9.24999 13.52 8.99602 13.964C8.624 14.622 8.25002 15.2 7.988 15.572C7.97207 15.5959 7.95652 15.6186 7.94154 15.6405C7.92269 15.6681 7.90474 15.6944 7.88803 15.72C7.80601 15.836 7.89002 16 8.032 16H9.707C10.3106 16 10.8 16.4894 10.8 17.093C10.8 17.6966 10.3106 18.186 9.707 18.186H8.00003C7.24802 18.186 6.55203 17.76 6.21599 17.078C6.042 16.736 5.974 16.36 6.01402 15.992C6.02401 15.882 5.94199 15.778 5.82999 15.778H0.17403C0.0779956 15.778 0 15.856 0 15.952V16.068C0 19.676 2.914 22.6 6.51002 22.6H16.656C18.5579 22.6 19.6378 20.8669 20.6993 19.1634C20.9951 18.6886 21.2896 18.216 21.6 17.784C22.158 17.008 23.5 16.392 23.892 16.224C23.956 16.196 24 16.132 24 16.06ZM1.51195 13.202L1.42794 13.334C1.35397 13.448 1.43594 13.6 1.57593 13.6H6.78395C6.84196 13.6 6.89594 13.572 6.92796 13.524C6.99596 13.4201 7.05994 13.312 7.11795 13.202C7.56797 12.446 7.96794 11.628 8.11394 11.024C8.45594 9.55604 7.72595 7.19805 6.87994 5.30201C6.82396 5.17604 6.64993 5.16401 6.57596 5.28004L1.51195 13.202Z"
      fill="white"
    />
  </svg>
)

export const CollectionSelectedAssetIcon = (props: SVGProps) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" fill="none" {...props}>
    <path
      d="M17.5 9.16667V6.52778C17.5 5.68401 16.7538 5 15.8333 5H4.16667C3.24619 5 2.5 5.68401 2.5 6.52778V17.2222C2.5 18.066 3.24619 18.75 4.16667 18.75H11.25"
      stroke="#F5F6FC"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M12.5 14.6667L14.4048 16.6667L19.1667 11.6667"
      stroke="#F5F6FC"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
    <path
      d="M6.66671 5L6.66671 4.16666C6.66671 3.28261 7.0179 2.43476 7.64302 1.80964C8.26814 1.18452 9.11599 0.833333 10 0.833333C10.8841 0.833333 11.7319 1.18452 12.3571 1.80964C12.9822 2.43476 13.3334 3.28261 13.3334 4.16667L13.3334 5"
      stroke="#F5F6FC"
      strokeWidth="1.5"
      strokeLinecap="round"
      strokeLinejoin="round"
    />
  </svg>
)

export const Nft20Icon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M20.1797 2C18.6608 1.99964 17.2055 2.60684 16.1403 3.68523V2H15.3199C13.8003 1.99901 12.3441 2.60626 11.2784 3.68523V2H10.4603C7.32315 2.00245 4.78057 4.53488 4.77811 7.65966V16.3545H4.51732C4.29849 16.3542 4.08024 16.3773 3.86643 16.4234C3.47831 16.5022 3.10944 16.6561 2.78085 16.8763C2.54716 16.3376 2.0146 15.9887 1.42549 15.9881C1.32123 15.9869 1.21723 15.998 1.11563 16.0214C0.327205 16.2173 -0.15257 17.0126 0.0440797 17.7979C0.202958 18.4325 0.764814 18.8854 1.42105 18.9079H1.46342C1.39473 19.2972 1.40305 19.6962 1.48796 20.0824C1.80032 21.5009 3.05723 22.515 4.51518 22.5248C4.95221 22.5247 5.38438 22.4332 5.78359 22.2562C6.32672 22.7363 7.02799 23.001 7.75417 23H18.0085C19.6557 23.0049 20.9951 21.6788 21 20.0381V20.0314V2H20.1797ZM20.1797 20.0314C20.176 21.2182 19.211 22.1793 18.0195 22.183H7.75417C7.02124 22.1819 6.33891 21.8107 5.94184 21.1971C5.66604 21.4217 5.34054 21.5777 4.99223 21.6523C4.83388 21.6855 4.67254 21.7026 4.51073 21.7034C3.43635 21.7023 2.50819 20.9554 2.28156 19.9093C2.21779 19.6077 2.21779 19.2962 2.28156 18.9946C2.29731 18.9073 2.31892 18.8213 2.34617 18.737C2.35062 18.6819 2.34607 18.6264 2.33278 18.5727C2.27026 18.273 2.00466 18.0586 1.69742 18.0598C1.65102 18.06 1.60483 18.0645 1.55921 18.0731C1.51396 18.0838 1.4675 18.089 1.42099 18.0886C1.06149 18.0888 0.769941 17.7986 0.769836 17.4405C0.769732 17.1328 0.986836 16.8673 1.28947 16.8052C1.33483 16.7953 1.38129 16.7909 1.42769 16.792C1.73362 16.7923 1.99802 17.0049 2.06305 17.3027C2.12556 17.6023 2.39116 17.8167 2.6984 17.8155C2.74496 17.8159 2.79131 17.8106 2.83662 17.8C2.89113 17.7895 2.94381 17.7708 2.99267 17.7445C3.0589 17.6862 3.12811 17.6314 3.19999 17.5801C3.45231 17.401 3.73925 17.2762 4.04262 17.2138C4.19925 17.1794 4.35923 17.1623 4.51968 17.1627C4.89739 17.162 5.26913 17.2559 5.60086 17.4358V7.66633C5.59589 4.99306 7.76766 2.82204 10.4516 2.81709H10.4605V7.65503C10.4617 4.98388 12.6355 2.81834 15.3179 2.81709H15.3201V7.65504C15.3214 4.98389 17.4952 2.81834 20.1776 2.81709H20.1798L20.1797 20.0314ZM18.5612 15.3443V19.3409C18.5576 20.0184 18.0064 20.5663 17.3263 20.5687H16.9406V19.7628C16.9171 19.317 16.5353 18.9746 16.0878 18.998C15.6731 19.0197 15.3417 19.3497 15.3199 19.7628V20.571H10.4603V19.7628C10.4579 19.3181 10.0953 18.959 9.64886 18.959C9.2032 18.9578 8.84092 19.3167 8.83966 19.7606V19.7628V20.571H8.45405C7.7725 20.5697 7.22033 20.0197 7.21907 19.3409V15.3443C7.22033 14.6654 7.7725 14.1154 8.45405 14.1142H8.83966V14.9202C8.83966 15.3657 9.20157 15.7272 9.64886 15.7285C10.0965 15.7272 10.4591 15.3661 10.4603 14.9202V14.1142H15.3199V14.9202C15.2965 15.366 15.6402 15.7463 16.0878 15.7697C16.5353 15.7931 16.9171 15.4507 16.9406 15.0049C16.9421 14.9767 16.9421 14.9484 16.9406 14.9202V14.1142H17.3263C18.0073 14.1166 18.5588 14.6659 18.5612 15.3443ZM12.1764 12.4616C12.1764 12.4616 11.7194 11.9677 11.6458 11.7557C11.5454 11.4663 11.6201 10.5311 11.6201 10.5311C11.6188 10.276 11.5133 10.1986 11.2572 10.1998H10.3578C10.1029 10.1998 9.97363 10.3285 9.97363 10.5824V11.6915C9.97363 12.0851 10.8363 12.4616 12.1764 12.4616ZM15.4274 10.531C15.4286 10.276 15.5341 10.1986 15.7902 10.1997H16.6897C16.9445 10.1997 17.0738 10.3285 17.0738 10.5823V11.6915C17.0738 12.0851 16.2111 12.4615 14.871 12.4615C14.871 12.4615 15.328 11.9676 15.4016 11.7556C15.502 11.4663 15.4274 10.531 15.4274 10.531ZM8.21559 4.38903C8.63871 4.05164 9.10875 3.7772 9.61102 3.57419C9.03525 3.5955 8.47512 3.7662 7.98603 4.06931C7.48842 4.37684 7.07127 4.79765 6.76889 5.29714C6.46039 5.79429 6.27902 6.35923 6.24057 6.94241C6.22832 7.08576 6.22832 7.22983 6.24057 7.37318C6.24427 7.49333 6.25569 7.60731 6.26714 7.72155L6.27399 7.7906L6.2902 7.91729L6.2902 7.9173C6.31983 8.15043 6.34945 8.38349 6.39662 8.61659C6.44525 8.36635 6.47544 8.11619 6.50563 7.86601L6.51474 7.7906C6.51977 7.74621 6.5241 7.70159 6.52841 7.65719L6.52841 7.65718C6.53747 7.56388 6.54643 7.47154 6.56157 7.38427C6.56575 7.35291 6.56947 7.32154 6.57315 7.29039C6.57729 7.25541 6.5814 7.22073 6.5861 7.18669C6.59119 7.14987 6.59921 7.11302 6.60722 7.07617L6.60722 7.07616L6.60722 7.07615C6.61322 7.0486 6.61921 7.02105 6.62398 6.99353C6.72766 6.48387 6.92008 5.99615 7.19242 5.55252C7.46524 5.11003 7.8112 4.71673 8.21559 4.38903ZM14.4706 3.57419C13.9924 3.77173 13.5427 4.03153 13.1331 4.34683C12.7415 4.65248 12.402 5.0191 12.1278 5.43257C11.8553 5.8463 11.6486 6.29936 11.5148 6.77588C11.4959 6.8404 11.4827 6.90553 11.4695 6.97063L11.4695 6.97067C11.4576 7.02879 11.4458 7.08688 11.43 7.14448C11.4093 7.22036 11.3997 7.2988 11.39 7.37768C11.3842 7.42576 11.3783 7.47401 11.3698 7.52194C11.3386 7.78393 11.3052 8.04373 11.2517 8.30797C11.2001 8.06325 11.1695 7.82238 11.1387 7.57998L11.1387 7.57994L11.1313 7.52194C11.1313 7.44632 11.1256 7.36927 11.1198 7.29161L11.1198 7.29159L11.1198 7.29156C11.1154 7.23241 11.1109 7.17291 11.109 7.11342C11.1045 6.97576 11.1224 6.84033 11.1402 6.70491C11.2114 6.14982 11.4088 5.61817 11.7176 5.15067C12.0234 4.68604 12.4342 4.29899 12.9169 4.02053C13.3875 3.74072 13.9228 3.58701 14.4706 3.57419ZM17.995 4.34683C18.4047 4.03168 18.8545 3.77188 19.3326 3.57419C18.7861 3.59217 18.2534 3.74968 17.7855 4.03158C17.3018 4.30566 16.8895 4.68891 16.5817 5.15067C16.2722 5.61802 16.074 6.14961 16.0021 6.70491C15.9843 6.84033 15.9687 6.97576 15.9731 7.11342C15.975 7.17161 15.9789 7.22981 15.9827 7.28768C15.988 7.36667 15.9932 7.44504 15.9932 7.52194C15.9985 7.56064 16.0036 7.59929 16.0088 7.63791C16.0386 7.8608 16.0683 8.08276 16.1158 8.30797C16.1649 8.0657 16.1952 7.82716 16.2256 7.58724C16.2284 7.56549 16.2312 7.54372 16.2339 7.52194C16.2406 7.48004 16.246 7.4379 16.2514 7.39584C16.2624 7.31085 16.2733 7.22618 16.2941 7.14448C16.3071 7.09389 16.3173 7.04293 16.3276 6.9919C16.3422 6.91973 16.3567 6.84743 16.3789 6.77588C16.5127 6.29936 16.7194 5.8463 16.9919 5.43257C17.266 5.01968 17.6047 4.65316 17.995 4.34683Z"
      fill="white"
    />
  </svg>
)

export const NftXIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M12 0L2.80713 9.19286L10.5399 16.6017C9.50042 17.3018 8.96753 18.0136 8.48384 18.6597L8.48313 18.6607L8.48293 18.6609C7.9305 19.4002 7.45337 20.0387 6.10491 20.7007C5.15896 21.1652 4.48456 21.2978 3.84753 21.3305C4.34999 21.9968 5.19599 22.5819 6.27942 23.0318C6.53176 22.9339 6.79713 22.818 7.07742 22.6803C8.90264 21.7852 9.61871 20.8277 10.2506 19.9829L10.2514 19.9817C10.8039 19.2423 11.281 18.6039 12.6296 17.9417C13.9782 17.2795 14.7753 17.2918 15.698 17.306H15.698L15.6982 17.306C16.3164 17.3158 16.9831 17.3259 17.7963 17.1518C16.8046 16.619 15.5296 16.2159 14.0884 15.9998L21.1928 9.19286L12 0ZM7.37363 17.0155L7.37899 17.0083L7.37901 17.0083C7.57949 16.7399 7.76995 16.4849 7.99763 16.237C5.23472 16.9028 3.33203 18.2972 3.33203 19.9111C3.33203 19.9712 3.33529 20.0311 3.34043 20.0904C3.61146 19.9875 3.8974 19.8644 4.20117 19.7151C6.0262 18.8185 6.74209 17.8606 7.37363 17.0155ZM18.6096 18.0667C17.5547 18.0496 16.3591 18.0312 14.5339 18.928C12.7087 19.8247 11.9928 20.7825 11.3613 21.6278L11.3609 21.6284C10.8085 22.3673 10.3313 23.0057 8.98288 23.6678C8.92185 23.6978 8.86288 23.7261 8.80322 23.7535C9.74427 23.9195 10.6982 24.002 11.6537 24C11.8645 24 12.0731 23.9962 12.2796 23.9885C12.6172 23.6313 12.8796 23.2809 13.1282 22.9481C13.6805 22.2088 14.1578 21.5702 15.5064 20.9081C16.855 20.2461 17.6523 20.2584 18.5748 20.2726H18.575C18.9998 20.2793 19.4476 20.2862 19.95 20.2337C19.967 20.1272 19.9755 20.0195 19.9756 19.9116C19.9756 19.2494 19.6548 18.6242 19.0862 18.071C18.9683 18.0715 18.8504 18.0699 18.7306 18.0683H18.7304L18.6096 18.0667ZM19.5883 21.1462C18.8621 22.2725 17.1703 23.1878 14.9902 23.6572C15.5194 23.0755 16.2343 22.4722 17.4111 21.8937C18.2554 21.4786 18.9648 21.2601 19.5883 21.1462Z"
      fill="white"
    />
  </svg>
)

export const X2y2Icon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M21.528 4.70405C19.6893 2.90732 17.174 1.8 14.4 1.8C8.7667 1.8 4.2 6.3667 4.2 12C4.2 17.6333 8.7667 22.2 14.4 22.2C17.1739 22.2 19.6893 21.0927 21.528 19.296C19.3347 22.156 15.8827 24 12 24C5.37258 24 0 18.6274 0 12C0 5.37258 5.37258 0 12 0C15.8827 0 19.3347 1.84402 21.528 4.70405ZM6.77759 17.8368C8.24862 19.2742 10.2609 20.16 12.48 20.16C16.9867 20.16 20.64 16.5067 20.64 12C20.64 7.49337 16.9867 3.84002 12.48 3.84002C10.2609 3.84002 8.24862 4.72587 6.77759 6.16324C8.53226 3.87524 11.2939 2.40002 14.4 2.40002C19.702 2.40002 24 6.69809 24 12C24 17.302 19.702 21.6 14.4 21.6C11.2939 21.6 8.53226 20.1248 6.77759 17.8368ZM12 19.2C15.9765 19.2 19.2001 15.9764 19.2001 12C19.2001 8.02353 15.9765 4.79999 12 4.79999C8.0236 4.79999 4.80005 8.02353 4.80005 12C4.80005 15.9764 8.0236 19.2 12 19.2ZM12 16.8C14.651 16.8 16.8 14.651 16.8 12C16.8 9.34902 14.651 7.19999 12 7.19999C9.34908 7.19999 7.20005 9.34902 7.20005 12C7.20005 14.651 9.34908 16.8 12 16.8Z"
      fill="white"
    />
  </svg>
)

export const SudoSwapIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M12 24C18.6274 24 24 18.6274 24 12C24 5.37258 18.6274 0 12 0C5.37258 0 0 5.37258 0 12C0 18.6274 5.37258 24 12 24ZM4.37501 7.35706V7.33475V7.3125H5.05176H5.72851L5.92208 7.82812L6.11564 8.34375L6.51939 9.4375L6.92308 10.5312L7.18845 11.2647L7.45383 11.9982L7.10851 12.9522L6.76326 13.9062L6.24608 15.2969L5.72895 16.6875H5.05201H4.37501V16.6622V16.6369L5.12889 14.5841L5.88276 12.5312L5.97514 12.2656L6.06745 12L5.97501 11.7344L5.88258 11.4688L5.12883 9.41288L4.37501 7.35706ZM18.0691 7.84375L18.168 7.57812L18.267 7.3125H18.9527H19.6384L19.6096 7.39062L19.5808 7.46875L18.7554 9.73438L17.93 12L18.7554 14.2656L19.5808 16.5312L19.6096 16.6094L19.6384 16.6875H18.9527H18.267L18.168 16.4219L18.0691 16.1562L17.5731 14.8125L17.0773 13.4688L16.8114 12.7344L16.5455 12L16.8114 11.2656L17.0773 10.5312L17.5731 9.1875L18.0691 7.84375ZM10.779 9.492L10.9989 9.43669L11.2188 9.38131L12.2188 9.39819L13.2188 9.41506L13.5938 9.5595L13.9688 9.70387L14.2512 9.993L14.5336 10.2821L14.6106 10.5389L14.6875 10.7957V10.9291V11.0625H14.0999H13.5123L13.4463 10.8734L13.3804 10.6842L13.0496 10.5302L12.7188 10.3763L12.0303 10.3756L11.3418 10.375L11.124 10.4914L10.9063 10.6079L10.7871 10.8109L10.668 11.0138L10.672 11.2816L10.676 11.5492L10.7441 11.6765L10.8123 11.8038L11.0043 11.9406L11.1965 12.0774L12.2701 12.2621L13.3438 12.4468L13.7313 12.6314L14.1188 12.8159L14.3195 13.0166L14.5203 13.2174L14.6481 13.4993L14.7759 13.7812L14.776 14.3125L14.7761 14.8438L14.6353 15.1984L14.4946 15.553L14.2499 15.8052L14.0051 16.0573L13.5495 16.1999L13.0938 16.3424L12.0625 16.3431L11.0313 16.3438L10.6875 16.2258L10.3438 16.1079L10.09 15.9612L9.83614 15.8145L9.70239 15.6721L9.56864 15.5297L9.44551 15.2493L9.32239 14.9688L9.27926 14.7344L9.23608 14.5H9.88564H10.5351L10.627 14.7647L10.7188 15.0293L11 15.1866L11.2813 15.3438H12.0625H12.8438L13.0542 15.255L13.2646 15.1663L13.4448 14.9175L13.625 14.6687L13.6247 14.35L13.6244 14.0312L13.4841 13.865L13.3438 13.6987L12.875 13.5398L12.4063 13.3808L11.8241 13.2834L11.242 13.1861L10.8554 13.0576L10.4688 12.9291L10.2395 12.7795L10.0101 12.6299L9.85439 12.4243L9.69864 12.2188L9.59164 11.8483L9.4847 11.4777L9.51633 11.1295L9.54795 10.7812L9.62876 10.5118L9.70958 10.2423L9.95214 9.99238L10.1946 9.7425L10.4868 9.61725L10.779 9.492Z"
      fill="white"
    />
  </svg>
)

export const LooksRareIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M7.13513 3.08105L0 10.2211L12 22.2162L24 10.2211L16.8649 3.08105H7.13513ZM6.48649 7.62158C9.51816 4.5766 14.4818 4.57658 17.5135 7.62156L20.1081 10.2162L17.5135 12.8108C14.4818 15.8558 9.51816 15.8558 6.48649 12.8108L3.89189 10.2162L6.48649 7.62158ZM8.27026 10.2162C8.27026 12.2769 9.94096 13.946 12 13.946C14.059 13.946 15.7297 12.2769 15.7297 10.2162C15.7297 8.15553 14.059 6.48651 12 6.48651C9.94096 6.48651 8.27026 8.15553 8.27026 10.2162ZM12 11.8379C11.1048 11.8379 10.3784 11.1122 10.3784 10.2162C10.3784 9.32028 11.1048 8.59462 12 8.59462C12.8952 8.59462 13.6216 9.32028 13.6216 10.2162C13.6216 11.1122 12.8952 11.8379 12 11.8379Z"
      fill="white"
    />
  </svg>
)

export const LarvaLabsMarketplaceIcon = (props: SVGProps) => (
  <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" {...props}>
    <path
      fillRule="evenodd"
      clipRule="evenodd"
      d="M9 5H12V17H9V5ZM13 5H16V13H13V5ZM13 14V17H22V14H13ZM2 17V14H8V17H2Z"
      fill="white"
    />
  </svg>
)
