package caddy

import (
	"bytes"
	"fmt"
	log "github.com/sirupsen/logrus"
	"github.com/xueqianlu/caddyproxy/types"
	"github.com/xueqianlu/caddyproxy/utils"
	"io/ioutil"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"text/template"
	"time"
)

type CaddyAPI struct {
	Url          string // URL of the Caddy API
	RootDir      string // Root directory of the Caddy to store website files and .caddy files.
	MasterDomain map[string]bool
	client       *http.Client
}

func NewCaddyAPI(url string, root string, masterList string) *CaddyAPI {
	master := make(map[string]bool)
	list := strings.Split(masterList, ",")
	for _, domain := range list {
		master[strings.ToLower(domain)] = true
	}

	return &CaddyAPI{Url: url, RootDir: root, client: &http.Client{}, MasterDomain: master}
}

func (c *CaddyAPI) buildPayloadForCreateWebsite(domain string, root string) (string, error) {
	// build the payload for creating a new website.
	buffer := bytes.NewBufferString("")
	cfg := StaticWebsiteCreatePayload{
		DomainName:       domain,
		RootDir:          root,
		WebsiteCaddyFile: filepath.Join(c.RootDir, fmt.Sprintf("%s.caddy", domain)),
	}
	tmpl, err := template.New("test").Parse(newWebsitePayloadTempl)
	if err != nil {
		log.WithError(err).Error("failed to parse website caddyfile template")
		return "", err
	}
	err = tmpl.Execute(buffer, cfg)
	if err != nil {
		log.WithError(err).Error("failed to build website caddyfile")
		return "", err
	}
	return buffer.String(), nil
}

func (c *CaddyAPI) newWebsiteCaddyFile(domain string, root string, master string) error {
	buffer := bytes.NewBufferString("")
	caddypath := filepath.Join(c.RootDir, fmt.Sprintf("%s.caddy", domain))
	// build CaddyFileTemplate.
	cfg := NewWebsiteCaddyFile{
		DomainName:   fmt.Sprintf("https://%s", domain),
		RootDir:      root,
		MasterDomain: master,
	}
	tmpl, err := template.New("test").Parse(staticWebsiteCaddyFileTempl)
	if err != nil {
		log.WithError(err).Error("failed to parse website caddyfile template")
		return err
	}
	err = tmpl.Execute(buffer, cfg)
	if err != nil {
		log.WithError(err).Error("failed to build website caddyfile")
		return err
	}

	if err = utils.CreateFile(caddypath, buffer.String()); err != nil {
		log.WithError(err).Error("failed to create website caddyfile")
		return err
	}
	return nil
}

func (c *CaddyAPI) newForwardCaddyFile(from string, to string, master string) error {
	buffer := bytes.NewBufferString("")
	caddypath := filepath.Join(c.RootDir, fmt.Sprintf("%s.caddy", from))
	// build CaddyFileTemplate.
	cfg := ForwardWebsiteCaddyFile{
		DomainName:   fmt.Sprintf("https://%s", from),
		Target:       to,
		MasterDomain: master,
	}
	tmpl, err := template.New("test").Parse(forwardWebsiteCaddyFileTempl)
	if err != nil {
		log.WithError(err).Error("failed to parse forward caddyfile template")
		return err
	}
	err = tmpl.Execute(buffer, cfg)
	if err != nil {
		log.WithError(err).Error("failed to build forward caddyfile")
		return err
	}

	if err = utils.CreateFile(caddypath, buffer.String()); err != nil {
		log.WithError(err).Error("failed to create forward caddyfile")
		return err
	}
	return nil
}

func (c *CaddyAPI) buildPayloadForForwardWebsite(domain string, target string) (string, error) {
	// build the payload for creating a new website.
	buffer := bytes.NewBufferString("")
	cfg := ForwardWebsitePayload{
		DomainName: domain,
		Target:     target,
	}
	tmpl, err := template.New("test").Parse(forwardWebsitePayloadTempl)
	if err != nil {
		log.WithError(err).Error("failed to parse forward payload template")
		return "", err
	}
	err = tmpl.Execute(buffer, cfg)
	if err != nil {
		log.WithError(err).Error("failed to build forward payload")
		return "", err
	}
	return buffer.String(), nil
}

func pathExist(path string) bool {
	_, err := os.Stat(path)
	if os.IsNotExist(err) {
		return false
	}
	return true
}

func extractMasterDomain(domain string) (string, error) {
	parts := strings.Split(domain, ".")
	if len(parts) < 2 {
		return domain, nil
	}

	return strings.ToLower(strings.Join(parts[len(parts)-2:], ".")), nil
}

func (c *CaddyAPI) CreateWebsite(domain string, resource string) error {
	// check domain is in masterDomain list.
	masterDomain, err := extractMasterDomain(domain)
	if err != nil || !c.MasterDomain[masterDomain] {
		return fmt.Errorf("domain %s is not support on this service", domain)
	}

	// unzip resource to the website root directory.
	websiteRoot := filepath.Join(c.RootDir, "websites", domain)
	rollback := false
	success := false
	backupname := fmt.Sprintf("%s.%s.bak", websiteRoot, time.Now().Format("2006-01-02-15-04-05"))
	defer func() {
		if !success && rollback {
			os.Remove(websiteRoot)
			if err := os.Rename(backupname, websiteRoot); err != nil {
				log.WithFields(log.Fields{
					"backup":  backupname,
					"website": websiteRoot,
				}).WithError(err).Error("failed to rollback website")
			}
		}
	}()
	// if websiteRoot exist, remove it first. and add rollback flag.
	if pathExist(websiteRoot) {
		// backup old website.
		if err := os.Rename(websiteRoot, backupname); err != nil {
			log.WithFields(log.Fields{
				"backup":  backupname,
				"website": websiteRoot,
			}).WithError(err).Error("failed to backup website")
			return err
		} else {
			rollback = true
		}
	}
	if err := utils.Unzip(resource, websiteRoot); err != nil {
		return err
	}
	// create caddyfile.
	if err := c.newWebsiteCaddyFile(domain, websiteRoot, masterDomain); err != nil {
		return err
	}

	// build api payload.
	payload, err := c.buildPayloadForCreateWebsite(domain, websiteRoot)
	if err != nil {
		return err
	}

	// send put request to the caddy api.
	path := fmt.Sprintf("%s/config/apps/http/servers/srv0/routes/%d", c.Url, 0)
	log.WithField("payload", payload).Debug("sending payload to caddy")
	if err := c.post(path, []byte(payload)); err != nil {
		return err
	}
	success = true
	return nil
}

func (c *CaddyAPI) ForwardWebsite(param types.ForwardWebsite) error {
	// check domain is in masterDomain list.
	masterDomain, err := extractMasterDomain(param.Domain)
	if err != nil || !c.MasterDomain[masterDomain] {
		return fmt.Errorf("domain %s is not support on this service", param.Domain)
	}
	// create caddyfile.
	if err := c.newForwardCaddyFile(param.Domain, param.Target, masterDomain); err != nil {
		return err
	}

	// build api payload.
	payload, err := c.buildPayloadForForwardWebsite(param.Domain, param.Target)
	if err != nil {
		return err
	}

	// send put request to the caddy api.
	path := fmt.Sprintf("%s/config/apps/http/servers/srv0/routes/%d", c.Url, 0)
	log.WithField("payload", payload).Debug("sending payload to caddy")
	if err := c.post(path, []byte(payload)); err != nil {
		return err
	}

	return nil
}

func (c *CaddyAPI) post(path string, data []byte) error {
	// send a get request to the Caddy API.
	req, err := http.NewRequest("POST", path, bytes.NewBuffer(data))
	if err != nil {
		log.WithError(err).Error("failed to create request")
		return err
	}

	req.Header.Set("Content-Type", "application/json")

	resp, err := c.client.Do(req)
	if err != nil {
		log.WithError(err).Error("failed to send request")
		return err
	}
	defer resp.Body.Close()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.WithError(err).Error("failed to read response body")
		return err
	}
	if resp.StatusCode != http.StatusOK {
		log.WithFields(log.Fields{
			"status":  resp.StatusCode,
			"message": string(body),
		}).Error("failed to post data")
		return fmt.Errorf("failed to post data")
	}
	return nil
}

func (c *CaddyAPI) put(path string, data []byte) error {
	// send a get request to the Caddy API.
	req, err := http.NewRequest("PUT", path, bytes.NewBuffer(data))
	if err != nil {
		log.WithError(err).Error("failed to create request")
		return err
	}

	req.Header.Set("Content-Type", "application/json")

	resp, err := c.client.Do(req)
	if err != nil {
		log.WithError(err).Error("failed to send request")
		return err
	}
	defer resp.Body.Close()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.WithError(err).Error("failed to read response body")
		return err
	}
	if resp.StatusCode != http.StatusOK {
		log.WithFields(log.Fields{
			"status":  resp.StatusCode,
			"message": string(body),
		}).Error("failed to put data")
		return fmt.Errorf("failed to put data")
	}
	return nil
}
