import { Box, Flex, Icon, Skeleton, Text, useColorModeValue } from '@chakra-ui/react';
import BigNumber from 'bignumber.js';
import { useRouter } from 'next/router';
import React from 'react';

import appConfig from 'configs/app/config';
import walletIcon from 'icons/wallet.svg';
import useApiQuery from 'lib/api/useApiQuery';
import getCurrencyValue from 'lib/getCurrencyValue';
import DataFetchAlert from 'ui/shared/DataFetchAlert';

import { getTokenBalanceTotal, calculateUsdValue } from '../utils/tokenUtils';

const TokenBalances = () => {
  const router = useRouter();

  const addressQuery = useApiQuery('address', {
    pathParams: { id: router.query.id?.toString() },
    queryOptions: { enabled: Boolean(router.query.id) },
  });

  const balancesQuery = useApiQuery('address_token_balances', {
    pathParams: { id: addressQuery.data?.hash },
    queryOptions: { enabled: Boolean(addressQuery.data) },
  });

  const bgColor = useColorModeValue('blackAlpha.50', 'whiteAlpha.50');

  if (addressQuery.isError || balancesQuery.isError) {
    return <DataFetchAlert/>;
  }

  if (addressQuery.isLoading || balancesQuery.isLoading) {
    const item = <Skeleton w={{ base: '100%', lg: '240px' }} h="82px" borderRadius="16px"/>;
    return (
      <Flex columnGap={ 3 } rowGap={ 3 } mt={{ base: '6px', lg: 0 }} flexDirection={{ base: 'column', lg: 'row' }}>
        { item }
        { item }
        { item }
      </Flex>
    );
  }

  const addressData = addressQuery.data;
  const { valueStr: nativeValue, usd: nativeUsd } = getCurrencyValue({
    value: addressData.coin_balance || '0',
    accuracy: 8,
    accuracyUsd: 2,
    exchangeRate: addressData.exchange_rate,
    decimals: String(appConfig.network.currency.decimals),
  });

  const tokenBalanceBn = getTokenBalanceTotal(balancesQuery.data.map(calculateUsdValue)).toFixed(2);

  const totalUsd = nativeUsd ? BigNumber(nativeUsd).toNumber() + BigNumber(tokenBalanceBn).toNumber() : undefined;

  const itemProps = {
    p: 5,
    bgColor,
    borderRadius: '16px',
    alignItems: 'center',
  };

  return (
    <Flex columnGap={ 3 } rowGap={ 3 } mt={{ base: '6px', lg: 0 }} flexDirection={{ base: 'column', lg: 'row' }}>
      <Flex { ...itemProps }>
        <Icon as={ walletIcon } boxSize="30px" mr={ 3 }/>
        <Box>
          <Text variant="secondary" fontSize="xs">Net Worth</Text>
          <Text fontWeight="500">{ totalUsd ? `$${ totalUsd } USD` : 'N/A' }</Text>
        </Box>
      </Flex>
      <Flex { ...itemProps }>
        <Icon as={ walletIcon } boxSize="30px" mr={ 3 }/>
        <Box>
          <Text variant="secondary" fontSize="xs">{ `${ appConfig.network.currency.symbol } Balance` }</Text>
          <Text fontWeight="500">{ nativeUsd && `$${ nativeUsd } USD | ` } { `${ nativeValue } ${ appConfig.network.currency.symbol }` }</Text>
        </Box>
      </Flex>
      <Flex { ...itemProps }>
        <Icon as={ walletIcon } boxSize="30px" mr={ 3 }/>
        <Box>
          <Text variant="secondary" fontSize="xs">Tokens</Text>
          <Text fontWeight="500">
            { `$${ tokenBalanceBn } USD ` }
            { Boolean(balancesQuery.data.length) && ` | ${ balancesQuery.data.length } ${ balancesQuery.data.length === 1 ? 'token' : 'tokens' }` }
          </Text>
        </Box>
      </Flex>
    </Flex>
  );
};

export default React.memo(TokenBalances);
