import { Alert, Grid, GridItem } from '@chakra-ui/react';
import React from 'react';

import type { Blob } from 'types/api/blobs';

import Skeleton from 'ui/shared/chakra/Skeleton';
import CopyToClipboard from 'ui/shared/CopyToClipboard';
import * as DetailedInfo from 'ui/shared/DetailedInfo/DetailedInfo';

import DetailedInfoSponsoredItem from 'ui/shared/DetailedInfo/DetailedInfoSponsoredItem';
import TxEntity from 'ui/shared/entities/tx/TxEntity';

import BlobData from './BlobData';

interface Props {
  data: Blob;
  isLoading?: boolean;
}

const BlobInfo = ({ data, isLoading }: Props) => {
  return (
    <Grid
      columnGap={ 8 }
      rowGap={ 3 }
      templateColumns={{ base: 'minmax(0, 1fr)', lg: '216px minmax(728px, auto)' }}
    >
      { !data.blob_data && (
        <GridItem colSpan={{ base: undefined, lg: 2 }} mb={ 3 }>
          <Skeleton isLoaded={ !isLoading }>
            <Alert status="warning">This blob is not yet indexed</Alert>
          </Skeleton>
        </GridItem>
      ) }

      { data.kzg_proof && (
        <>
          <DetailedInfo.ItemLabel
            hint="Zero knowledge proof. Allows for quick verification of commitment"
            isLoading={ isLoading }
          >
            Proof
          </DetailedInfo.ItemLabel>
          <DetailedInfo.ItemValue>
            <Skeleton isLoaded={ !isLoading } overflow="hidden" whiteSpace="pre-wrap" wordBreak="break-all" lineHeight={ 6 } my="-2px">
              { data.kzg_proof }
              <CopyToClipboard text={ data.kzg_proof } isLoading={ isLoading }/>
            </Skeleton>
          </DetailedInfo.ItemValue>
        </>
      ) }

      { data.kzg_commitment && (
        <>
          <DetailedInfo.ItemLabel
            hint="Commitment to the data in the blob"
            isLoading={ isLoading }
          >
            Commitment
          </DetailedInfo.ItemLabel>
          <DetailedInfo.ItemValue>
            <Skeleton isLoaded={ !isLoading } overflow="hidden" whiteSpace="pre-wrap" wordBreak="break-all" lineHeight={ 6 } my="-2px">
              { data.kzg_commitment }
              <CopyToClipboard text={ data.kzg_commitment } isLoading={ isLoading }/>
            </Skeleton>
          </DetailedInfo.ItemValue>
        </>
      ) }

      { data.blob_data && (
        <>
          <DetailedInfo.ItemLabel
            hint="Blob size in bytes"
            isLoading={ isLoading }
          >
            Size, bytes
          </DetailedInfo.ItemLabel>
          <DetailedInfo.ItemValue>
            <Skeleton isLoaded={ !isLoading } overflow="hidden" whiteSpace="pre-wrap" wordBreak="break-all">
              { (data.blob_data.replace('0x', '').length / 2).toLocaleString() }
            </Skeleton>
          </DetailedInfo.ItemValue>
        </>
      ) }

      { data.blob_data && <DetailedInfo.ItemDivider/> }

      { data.transaction_hashes[0] && (
        <>
          <DetailedInfo.ItemLabel
            hint="Hash of the transaction with this blob"
            isLoading={ isLoading }
          >
            Transaction hash
          </DetailedInfo.ItemLabel>
          <DetailedInfo.ItemValue>
            <TxEntity hash={ data.transaction_hashes[0].transaction_hash } isLoading={ isLoading } noIcon noCopy={ false }/>
          </DetailedInfo.ItemValue>
        </>
      ) }

      <DetailedInfoSponsoredItem isLoading={ isLoading }/>

      { data.blob_data && (
        <>
          <DetailedInfo.ItemDivider/>
          <BlobData data={ data.blob_data } hash={ data.hash } isLoading={ isLoading }/>
        </>
      ) }
    </Grid>
  );
};

export default React.memo(BlobInfo);
