// Copyright 2018 The go-ethereum Authors
// This file is part of the go-ethereum library.
//
// The go-ethereum library is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// The go-ethereum library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with the go-ethereum library. If not, see <http://www.gnu.org/licenses/>.

package localstore

import (
	"context"
	"errors"
	"io/ioutil"
	"testing"

	"github.com/ethersphere/bee/pkg/logging"
	statestore "github.com/ethersphere/bee/pkg/statestore/mock"

	"github.com/ethersphere/bee/pkg/shed"
	"github.com/ethersphere/bee/pkg/storage"
	"github.com/ethersphere/bee/pkg/tags"
	tagtesting "github.com/ethersphere/bee/pkg/tags/testing"
	"github.com/syndtr/goleveldb/leveldb"
)

// here we try to set a normal tag (that should be handled by pushsync)
// as a result we should expect the tag value to remain in the pull index
// and we expect that the tag should not be incremented by pull sync set
func TestModeSetSyncNormalTag(t *testing.T) {
	mockStatestore := statestore.NewStateStore()
	logger := logging.New(ioutil.Discard, 0)
	db := newTestDB(t, &Options{Tags: tags.NewTags(mockStatestore, logger)})

	tag, err := db.tags.Create(1)
	if err != nil {
		t.Fatal(err)
	}

	ch := generateTestRandomChunk().WithTagID(tag.Uid)
	_, err = db.Put(context.Background(), storage.ModePutUpload, ch)
	if err != nil {
		t.Fatal(err)
	}

	err = tag.Inc(tags.StateStored) // so we don't get an error on tag.Status later on
	if err != nil {
		t.Fatal(err)
	}

	item, err := db.pullIndex.Get(shed.Item{
		Address: ch.Address().Bytes(),
		BinID:   1,
	})
	if err != nil {
		t.Fatal(err)
	}

	if item.Tag != tag.Uid {
		t.Fatalf("unexpected tag id value got %d want %d", item.Tag, tag.Uid)
	}

	err = db.Set(context.Background(), storage.ModeSetSync, ch.Address())
	if err != nil {
		t.Fatal(err)
	}

	item, err = db.pullIndex.Get(shed.Item{
		Address: ch.Address().Bytes(),
		BinID:   1,
	})
	if err != nil {
		t.Fatal(err)
	}

	// expect the same tag Uid because when we set pull sync on a normal tag
	// the tag Uid should remain untouched in pull index
	if item.Tag != tag.Uid {
		t.Fatalf("unexpected tag id value got %d want %d", item.Tag, tag.Uid)
	}

	// 1 stored (because incremented manually in test), 1 sent, 1 synced, 1 total
	tagtesting.CheckTag(t, tag, 0, 1, 0, 1, 1, 1)
}

// TestModeSetRemove validates ModeSetRemove index values on the provided DB.
func TestModeSetRemove(t *testing.T) {
	for _, tc := range multiChunkTestCases {
		t.Run(tc.name, func(t *testing.T) {
			db := newTestDB(t, nil)

			chunks := generateTestRandomChunks(tc.count)

			_, err := db.Put(context.Background(), storage.ModePutUpload, chunks...)
			if err != nil {
				t.Fatal(err)
			}

			err = db.Set(context.Background(), storage.ModeSetRemove, chunkAddresses(chunks)...)
			if err != nil {
				t.Fatal(err)
			}

			t.Run("retrieve indexes", func(t *testing.T) {
				for _, ch := range chunks {
					wantErr := leveldb.ErrNotFound
					_, err := db.retrievalDataIndex.Get(addressToItem(ch.Address()))
					if !errors.Is(err, wantErr) {
						t.Errorf("got error %v, want %v", err, wantErr)
					}

					// access index should not be set
					_, err = db.retrievalAccessIndex.Get(addressToItem(ch.Address()))
					if !errors.Is(err, wantErr) {
						t.Errorf("got error %v, want %v", err, wantErr)
					}
				}

				t.Run("retrieve data index count", newItemsCountTest(db.retrievalDataIndex, 0))

				t.Run("retrieve access index count", newItemsCountTest(db.retrievalAccessIndex, 0))
			})

			for _, ch := range chunks {
				newPullIndexTest(db, ch, 0, leveldb.ErrNotFound)(t)
			}

			t.Run("pull index count", newItemsCountTest(db.pullIndex, 0))

			t.Run("gc index count", newItemsCountTest(db.gcIndex, 0))

			t.Run("gc size", newIndexGCSizeTest(db))
		})
	}
}
