// Copyright 2020 The Swarm Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package pipeline

import (
	"hash"

	"github.com/ethersphere/bee/pkg/swarm"
	"github.com/ethersphere/bmt"
	bmtlegacy "github.com/ethersphere/bmt/legacy"
	"golang.org/x/crypto/sha3"
)

type bmtWriter struct {
	b    bmt.Hash
	next chainWriter
}

// newBmtWriter returns a new bmtWriter. Partial writes are not supported.
// Note: branching factor is the BMT branching factor, not the merkle trie branching factor.
func newBmtWriter(branches int, next chainWriter) chainWriter {
	return &bmtWriter{
		b:    bmtlegacy.New(bmtlegacy.NewTreePool(hashFunc, branches, bmtlegacy.PoolSize)),
		next: next,
	}
}

// chainWrite writes data in chain. It assumes span has been prepended to the data.
// The span can be encrypted or unencrypted.
func (w *bmtWriter) chainWrite(p *pipeWriteArgs) error {
	w.b.Reset()
	err := w.b.SetSpanBytes(p.data[:swarm.SpanSize])
	if err != nil {
		return err
	}
	_, err = w.b.Write(p.data[swarm.SpanSize:])
	if err != nil {
		return err
	}
	p.ref = w.b.Sum(nil)
	return w.next.chainWrite(p)
}

// sum calls the next writer for the cryptographic sum
func (w *bmtWriter) sum() ([]byte, error) {
	return w.next.sum()
}

func hashFunc() hash.Hash {
	return sha3.NewLegacyKeccak256()
}
