package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"html/template"
	"log"
	"net/http"
	"sort"
	"sync"
	"time"

	"nodemonitor/types"
)

var (
	port = flag.String("port", "8080", "Server port")
)

type Server struct {
	nodes map[string]*types.NodeStatus
	mutex sync.RWMutex
}

func NewServer() *Server {
	return &Server{
		nodes: make(map[string]*types.NodeStatus),
	}
}

func (s *Server) handleReport(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		http.Error(w, "Method not allowed", http.StatusMethodNotAllowed)
		return
	}

	var status types.NodeStatus
	if err := json.NewDecoder(r.Body).Decode(&status); err != nil {
		http.Error(w, "Invalid JSON", http.StatusBadRequest)
		return
	}

	status.LastHeartbeat = time.Now()

	s.mutex.Lock()
	s.nodes[status.NodeID] = &status
	s.mutex.Unlock()

	log.Printf("Received report from node: %s", status.NodeID)
	w.WriteHeader(http.StatusOK)
}
func (s *Server) handleNodes(w http.ResponseWriter, r *http.Request) {
	s.mutex.RLock()
	nodes := make([]*types.NodeStatus, 0, len(s.nodes))
	for _, node := range s.nodes {
		nodes = append(nodes, node)
	}
	s.mutex.RUnlock()

	// Sort by node ID
	sort.Slice(nodes, func(i, j int) bool {
		return nodes[i].NodeID < nodes[j].NodeID
	})

	tmpl := `
<!DOCTYPE html>
<html>
<head>
    <title>Node Monitor</title>
    <style>
        table { border-collapse: collapse; width: 100%; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        .offline { background-color: #ffebee; }
        .online { background-color: #e8f5e8; }
        .disk-item { margin-bottom: 5px; }
    </style>
</head>
<body>
    <h1>Node Monitor Dashboard</h1>
    <p>Last updated: {{.UpdateTime}}</p>
    <table>
        <tr>
            <th>Node ID</th>
            <th>Status</th>
            <th>Height</th>
            <th>CPU Usage (%)</th>
            <th>Memory Usage (%)</th>
            <th>Memory (Free/Total)</th>
            <th>Disk Information</th>
            <th>Private IP</th>
            <th>Public IP</th>
            <th>Last Report</th>
        </tr>
        {{range .Nodes}}
        <tr class="{{.StatusClass}}">
            <td>{{.NodeID}}</td>
            <td>{{.Status}}</td>
            <td>{{.Height}}</td>
            <td>{{printf "%.2f" .CPUUsage}}</td>
            <td>{{printf "%.2f" .MemoryUsage}}</td>
            <td>{{.MemoryInfo}}</td>
            <td>{{.AllDisksInfo}}</td>
            <td>{{.PrivateIP}}</td>
            <td>{{.PublicIP}}</td>
            <td>{{.LastReport}}</td>
        </tr>
        {{end}}
    </table>
</body>
</html>
`

	type NodeView struct {
		*types.NodeStatus
		Status       string
		StatusClass  string
		MemoryInfo   string
		AllDisksInfo template.HTML
		LastReport   string
	}

	nodeViews := make([]NodeView, len(nodes))
	for i, node := range nodes {
		isOnline := time.Since(node.LastHeartbeat) < 30*time.Second
		status := "Online"
		statusClass := "online"
		if !isOnline {
			status = "Offline"
			statusClass = "offline"
		}

		// Format all disk information
		var allDisksInfo string
		if len(node.DiskInfos) > 0 {
			for _, diskInfo := range node.DiskInfos {
				mountPoint := diskInfo.Path
				usage := diskInfo.UsedPercent * 100
				allDisksInfo += fmt.Sprintf("<div class='disk-item'><strong>%s:</strong> %.2f%% (%.2f GB / %.2f GB)</div>",
					mountPoint,
					usage,
					float64(diskInfo.Free)/(1024*1024*1024),
					float64(diskInfo.Total)/(1024*1024*1024))
			}
		} else {
			allDisksInfo = "No disk information available"
		}

		nodeViews[i] = NodeView{
			NodeStatus:   node,
			Status:       status,
			StatusClass:  statusClass,
			MemoryInfo:   fmt.Sprintf("%.2f GB / %.2f GB", float64(node.MemoryFree)/(1024*1024*1024), float64(node.MemoryTotal)/(1024*1024*1024)),
			AllDisksInfo: template.HTML(allDisksInfo),
			LastReport:   node.Timestamp.Format("2006-01-02 15:04:05"),
		}
	}

	data := struct {
		Nodes      []NodeView
		UpdateTime string
	}{
		Nodes:      nodeViews,
		UpdateTime: time.Now().Format("2006-01-02 15:04:05"),
	}

	t, err := template.New("nodes").Parse(tmpl)
	if err != nil {
		http.Error(w, "Template error", http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "text/html")
	if err := t.Execute(w, data); err != nil {
		http.Error(w, "Template execution error", http.StatusInternalServerError)
		return
	}
}
func main() {
	flag.Parse()

	server := NewServer()

	http.HandleFunc("/api/report", server.handleReport)
	http.HandleFunc("/", server.handleNodes)

	log.Printf("Server starting on port %s", *port)
	log.Fatal(http.ListenAndServe(":"+*port, nil))
}
