package main

import (
	"bytes"
	"encoding/json"
	"flag"
	"fmt"
	"github.com/ethereum/go-ethereum/ethclient"
	"log"
	"net/http"
	"strings"
	"time"

	"nodemonitor/types"
)

var (
	serverURL = flag.String("server", "http://localhost:8080", "Server URL")
	interval  = flag.Duration("interval", 5*time.Second, "Report interval")
	localRpc  = flag.String("local-rpc", "", "Local Node RPC URL")
	diskPaths = flag.String("disks", "/", "Comma-separated list of disk paths to monitor (e.g., /,/data,/logs)")
	nameTag   = flag.String("name-tag", "", "Name tag")
)

type AgentConfig struct {
	ServerURL string
	Interval  time.Duration
	LocalRpc  string
	DiskPaths []string
	NameTag   string
}

func parseConfig() *AgentConfig {
	flag.Parse()
	paths := strings.Split(*diskPaths, ",")
	for i, path := range paths {
		paths[i] = strings.TrimSpace(path)
	}
	return &AgentConfig{
		ServerURL: *serverURL,
		Interval:  *interval,
		LocalRpc:  *localRpc,
		DiskPaths: paths,
		NameTag:   *nameTag,
	}
}

func main() {
	config := parseConfig()
	if config.NameTag == "" {
		log.Fatal("name-tag is required")
	}

	log.Printf("Starting agent with node ID: %s", config.NameTag)
	log.Printf("Reporting to server: %s", *serverURL)

	ticker := time.NewTicker(config.Interval)
	defer ticker.Stop()

	var nodeClient *ethclient.Client
	var err error
	if len(config.LocalRpc) > 0 {
		nodeClient, err = ethclient.Dial(config.LocalRpc)
		if err != nil {
			log.Fatalf("Failed to connect to local node RPC: %v", err)
		}
	}

	for {
		status := collectNodeStatus(config, nodeClient)
		if err := reportToServer(status); err != nil {
			log.Printf("Failed to report to server: %v", err)
		} else {
			log.Printf("Successfully reported status")
		}

		<-ticker.C
	}
}

func collectNodeStatus(conf *AgentConfig, nodeClient *ethclient.Client) types.NodeStatus {
	cpuUsage := getCPUUsage()
	memUsage, memTotal, memFree := getMemoryInfo()
	diskInfos := getDiskInfo(conf.DiskPaths)
	privateIP := getPrivateIP()
	publicIP := getPublicIP()
	nodeHeight := getNodeHeight(nodeClient)

	return types.NodeStatus{
		NodeID:      conf.NameTag,
		CPUUsage:    cpuUsage,
		MemoryUsage: memUsage,
		MemoryTotal: memTotal,
		MemoryFree:  memFree,
		DiskInfos:   diskInfos,
		PrivateIP:   privateIP,
		PublicIP:    publicIP,
		Height:      nodeHeight,
		Timestamp:   time.Now(),
	}
}

func reportToServer(status types.NodeStatus) error {
	data, err := json.Marshal(status)
	if err != nil {
		return fmt.Errorf("failed to marshal status: %v", err)
	}

	resp, err := http.Post(*serverURL+"/api/report", "application/json", bytes.NewBuffer(data))
	if err != nil {
		return fmt.Errorf("failed to post to server: %v", err)
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("server returned status: %d", resp.StatusCode)
	}

	return nil
}
