package main

import (
	"context"
	"fmt"
	"github.com/ethereum/go-ethereum/ethclient"
	"io"
	"net"
	"net/http"
	"nodemonitor/types"
	"os"
	"strconv"
	"strings"
	"time"

	"github.com/shirou/gopsutil/v3/cpu"
	"github.com/shirou/gopsutil/v3/disk"
	"github.com/shirou/gopsutil/v3/mem"
)

// getPrivateIP gets the local private IP address
func getPrivateIP() string {
	conn, err := net.Dial("udp", "8.8.8.8:80")
	if err != nil {
		return "unknown"
	}
	defer conn.Close()

	localAddr := conn.LocalAddr().(*net.UDPAddr)
	return localAddr.IP.String()
}

// getPublicIP gets the public IP address
func getPublicIP() string {
	resp, err := http.Get("https://api.ipify.org?format=text")
	if err != nil {
		return "unknown"
	}
	defer resp.Body.Close()

	ip, err := io.ReadAll(resp.Body)
	if err != nil {
		return "unknown"
	}

	return strings.TrimSpace(string(ip))
}

func getNodeHeight(nodeClient *ethclient.Client) string {
	if nodeClient == nil {
		return "N/A"
	}
	height, err := nodeClient.BlockNumber(context.Background())
	if err != nil {
		return fmt.Sprintf("fetch error:%s", err)
	}
	return strconv.FormatUint(height, 10)
}

// getCPUUsage gets CPU usage percentage
func getCPUUsage() float64 {
	percent, err := cpu.Percent(time.Second, false)
	if err != nil || len(percent) == 0 {
		return 0
	}
	return percent[0]
}

// getMemoryInfo gets memory usage information
func getMemoryInfo() (float64, uint64, uint64) {
	v, err := mem.VirtualMemory()
	if err != nil {
		return 0, 0, 0
	}
	return v.UsedPercent, v.Total, v.Available
}

// getDiskInfo gets disk usage information for multiple paths
func getDiskInfo(paths []string) []types.DiskInfo {
	var diskInfos []types.DiskInfo

	// Default paths if none provided
	if len(paths) == 0 {
		paths = []string{"/"}
	}

	for _, path := range paths {
		usage, err := disk.Usage(path)
		if err != nil {
			// Add error entry for failed disk
			diskInfos = append(diskInfos, types.DiskInfo{
				Path:        path,
				UsedPercent: 0,
				Total:       0,
				Free:        0,
			})
			continue
		}

		diskInfos = append(diskInfos, types.DiskInfo{
			Path:        path,
			UsedPercent: usage.UsedPercent,
			Total:       usage.Total,
			Free:        usage.Free,
		})
	}

	return diskInfos
}

// getSystemDiskInfo gets system disk (root) usage
func getSystemDiskInfo() types.DiskInfo {
	usage, err := disk.Usage("/")
	if err != nil {
		return types.DiskInfo{Path: "/", UsedPercent: 0, Total: 0, Free: 0}
	}

	return types.DiskInfo{
		Path:        "/",
		UsedPercent: usage.UsedPercent,
		Total:       usage.Total,
		Free:        usage.Free,
	}
}

// getNodeID generates a unique node identifier
func getNodeID() string {
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}
	return fmt.Sprintf("%s-%s", hostname, getPrivateIP())
}
