#!/bin/bash

set -e

# 配置变量
INSTALL_DIR="/opt/nodemonitor"
BINARY_DIR="."
SERVER_PORT="8080"
AGENT_SERVER="https://movanm.bitheart.org"
AGENT_NAME_TAG=""
AGENT_DISKS="/"
AGENT_RPC=""

# 检查参数
if [ "$#" -lt 1 ]; then
    echo "Usage: $0 <server|agent> [options...]"
    echo "  server: Deploy nmserver"
    echo "  agent: Deploy nmagent"
    echo ""
    echo "Examples:"
    echo "  $0 server"
    echo "  $0 agent --name-tag=node1 --server=http://192.168.1.100:8080"
    exit 1
fi

SERVICE_TYPE=$1
shift

# 解析参数
while [[ $# -gt 0 ]]; do
    case $1 in
        --name-tag=*)
            AGENT_NAME_TAG="${1#*=}"
            shift
            ;;
        --server=*)
            AGENT_SERVER="${1#*=}"
            shift
            ;;
        --disks=*)
            AGENT_DISKS="${1#*=}"
            shift
            ;;
        --rpc=*)
            AGENT_RPC="${1#*=}"
            shift
            ;;
        --port=*)
            SERVER_PORT="${1#*=}"
            shift
            ;;
        *)
            echo "Unknown parameter: $1"
            exit 1
            ;;
    esac
done

# 创建安装目录
echo "Creating installation directory..."
sudo mkdir -p $INSTALL_DIR

case $SERVICE_TYPE in
    "server")
        echo "Deploying nmserver..."

        # 复制二进制文件
        if [ ! -f "$BINARY_DIR/nmserver" ]; then
            echo "Error: nmserver binary not found in $BINARY_DIR"
            echo "You can download it with  \`wget https://nmdl.bitheart.org/nmserver\` or build from source."
            exit 1
        fi
        sudo cp $BINARY_DIR/nmserver $INSTALL_DIR/
        sudo chmod +x $INSTALL_DIR/nmserver

        # 创建服务文件
        cat << EOF | sudo tee /etc/systemd/system/nmserver.service > /dev/null
[Unit]
Description=Node Monitor Server
After=network.target
Wants=network.target

[Service]
Type=simple
User=root
Group=root
WorkingDirectory=$INSTALL_DIR
ExecStart=$INSTALL_DIR/nmserver -port=$SERVER_PORT
Restart=always
RestartSec=10
StandardOutput=journal
StandardError=journal
SyslogIdentifier=nmserver

[Install]
WantedBy=multi-user.target
EOF

        # 重载 systemd 并启动服务
        sudo systemctl daemon-reload
        sudo systemctl enable nmserver
        sudo systemctl start nmserver

        echo "nmserver service deployed and started successfully!"
        echo "Service status:"
        sudo systemctl status nmserver --no-pager
        ;;

    "agent")
        echo "Deploying nmagent..."

        if [ -z "$AGENT_NAME_TAG" ]; then
            echo "Error: --name-tag is required for agent deployment"
            exit 1
        fi

        # 复制二进制文件
        if [ ! -f "$BINARY_DIR/nmagent" ]; then
            echo "Error: nmagent binary not found in $BINARY_DIR"
            echo "You can download it with  \`wget https://nmdl.bitheart.org/nmagent\` or build from source."
            exit 1
        fi
        sudo cp $BINARY_DIR/nmagent $INSTALL_DIR/
        sudo chmod +x $INSTALL_DIR/nmagent

        # 创建服务文件
        cat << EOF | sudo tee /etc/systemd/system/nmagent.service > /dev/null
[Unit]
Description=Node Monitor Agent
After=network.target
Wants=network.target

[Service]
Type=simple
User=root
Group=root
WorkingDirectory=$INSTALL_DIR
ExecStart=$INSTALL_DIR/nmagent -server=$AGENT_SERVER -name-tag=$AGENT_NAME_TAG -disks="$AGENT_DISKS" -local-rpc=$AGENT_RPC
Restart=always
RestartSec=10
StandardOutput=journal
StandardError=journal
SyslogIdentifier=nmagent

[Install]
WantedBy=multi-user.target
EOF

        # 重载 systemd 并启动服务
        sudo systemctl daemon-reload
        sudo systemctl enable nmagent
        sudo systemctl start nmagent

        echo "nmagent service deployed and started successfully!"
        echo "Service status:"
        sudo systemctl status nmagent --no-pager
        ;;

    *)
        echo "Invalid service type: $SERVICE_TYPE"
        echo "Use 'server' or 'agent'"
        exit 1
        ;;
esac

echo ""
echo "Useful commands:"
echo "  View logs: sudo journalctl -u $SERVICE_TYPE -f"
echo "  Stop service: sudo systemctl stop $SERVICE_TYPE"
echo "  Start service: sudo systemctl start $SERVICE_TYPE"
echo "  Restart service: sudo systemctl restart $SERVICE_TYPE"
echo "  Disable autostart: sudo systemctl disable $SERVICE_TYPE"