package main

import (
	"bytes"
	"database/sql"
	"encoding/json"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"strings"
	"time"

	_ "github.com/go-sql-driver/mysql"
)

type TbAccountInfo struct {
	Id             int64     `json:"id"`
	BlockId        int64     `json:"block_id"`
	BlockHash      string    `json:"block_hash"`
	TxHash         string    `json:"tx_hash"`
	AccountAddress string    `json:"account_address"`
	AccountType    int       `json:"account_type"`
	MyNameTag      string    `json:"my_name_tag"`
	Balance        float64   `json:"balance"`
	Status         int       `json:"status"`
	IsDeleted      int8      `json:"is_deleted"`
	SyncTime       time.Time `json:"sync_time"`
	CreateTime     time.Time `json:"create_time"`
	UpdateTime     time.Time `json:"update_time"`
}

func (t *TbAccountInfo) TableName() string {
	return "tb_account_info"
}

// RPC请求结构体
// 这里只处理eth_getBalance的params
// 其他方法直接转发

type RPCRequest struct {
	Jsonrpc string        `json:"jsonrpc"`
	Method  string        `json:"method"`
	Params  []interface{} `json:"params"`
	Id      interface{}   `json:"id"`
}

type RPCResponse struct {
	Jsonrpc string      `json:"jsonrpc"`
	Id      interface{} `json:"id"`
	Result  interface{} `json:"result"`
	Error   interface{} `json:"error,omitempty"`
}

var (
	db         *sql.DB
	rpcBackend = os.Getenv("ETH_RPC_BACKEND") // 真实以太坊RPC地址，建议用环境变量配置
)

func main() {
	// 初始化数据库连接
	var err error
	dsn := os.Getenv("MYSQL_DSN") // 例如 "user:password@tcp(127.0.0.1:3306)/dbname"
	db, err = sql.Open("mysql", dsn)
	if err != nil {
		log.Fatalf("数据库连接失败: %v", err)
	}
	defer db.Close()

	http.HandleFunc("/", proxyHandler)
	log.Println("RPC代理服务启动，监听端口: 8545")
	log.Fatal(http.ListenAndServe(":8545", nil))
}

func proxyHandler(w http.ResponseWriter, r *http.Request) {
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		http.Error(w, "读取请求失败", http.StatusBadRequest)
		return
	}
	defer r.Body.Close()

	var req RPCRequest
	if err := json.Unmarshal(body, &req); err != nil {
		forwardToBackend(w, body)
		return
	}

	if req.Method == "eth_getBalance" && len(req.Params) > 0 {
		address, ok := req.Params[0].(string)
		if !ok {
			forwardToBackend(w, body)
			return
		}
		if !accountExists(strings.ToLower(address)) {
			resp := RPCResponse{
				Jsonrpc: req.Jsonrpc,
				Id:      req.Id,
				Result:  "0x0",
			}
			w.Header().Set("Content-Type", "application/json")
			json.NewEncoder(w).Encode(resp)
			return
		}
	}
	// 其他情况直接转发
	forwardToBackend(w, body)
}

func accountExists(address string) bool {
	var count int
	query := "SELECT COUNT(1) FROM tb_account_info WHERE account_address = ? AND is_deleted = 0"
	err := db.QueryRow(query, address).Scan(&count)
	if err != nil {
		if err == sql.ErrNoRows {
			return false
		}
		return true
	}
	return count > 0
}

func forwardToBackend(w http.ResponseWriter, body []byte) {
	resp, err := http.Post(rpcBackend, "application/json", bytes.NewReader(body))
	if err != nil {
		http.Error(w, "后端RPC请求失败", http.StatusBadGateway)
		return
	}
	defer resp.Body.Close()
	w.Header().Set("Content-Type", "application/json")
	io.Copy(w, resp.Body)
}
