package main

import (
	"bytes"
	"database/sql"
	"encoding/json"
	"io"
	"log"
	"net"
	"net/http"
	"os"
	"strconv"
	"strings"
	"time"

	"bufio"
	"fmt"
	"net/url"
	"path/filepath"
	"regexp"
	"sync"

	"golang.org/x/time/rate"

	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
	"github.com/fsnotify/fsnotify"
	_ "github.com/go-sql-driver/mysql"
)

type TbAccountInfo struct {
	Id             int64     `json:"id"`
	BlockId        int64     `json:"block_id"`
	BlockHash      string    `json:"block_hash"`
	TxHash         string    `json:"tx_hash"`
	AccountAddress string    `json:"account_address"`
	AccountType    int       `json:"account_type"`
	MyNameTag      string    `json:"my_name_tag"`
	Balance        float64   `json:"balance"`
	Status         int       `json:"status"`
	IsDeleted      int8      `json:"is_deleted"`
	SyncTime       time.Time `json:"sync_time"`
	CreateTime     time.Time `json:"create_time"`
	UpdateTime     time.Time `json:"update_time"`
}

func (t *TbAccountInfo) TableName() string {
	return "tb_account_info"
}

// RPC request structure
// Only handle params for eth_getBalance here
// Forward other methods directly

type RPCRequest struct {
	Jsonrpc string        `json:"jsonrpc"`
	Method  string        `json:"method"`
	Params  []interface{} `json:"params"`
	Id      interface{}   `json:"id"`
}

type RPCResponse struct {
	Jsonrpc string      `json:"jsonrpc"`
	Id      interface{} `json:"id"`
	Result  interface{} `json:"result"`
	Error   interface{} `json:"error,omitempty"`
}

var (
	db                *sql.DB
	rpcBackend        = os.Getenv("ETH_RPC_BACKEND") // Real Ethereum RPC address, recommend using environment variable
	ethClient         *ethclient.Client
	blacklistContract common.Address
	// whitelist related
	whitelistFile     string
	whitelist         map[string]struct{}
	whitelistPatterns []*regexp.Regexp
	whitelistMu       sync.RWMutex

	// IP Blacklist related
	ipBlacklistFile     string
	ipBlacklist         map[string]struct{}
	ipBlacklistPatterns []*regexp.Regexp
	ipBlacklistMu       sync.RWMutex

	// Rate limiting
	visitors   = make(map[string]*visitor)
	visitorsMu sync.Mutex

	// Rate limit settings
	normalLimit     rate.Limit
	normalBurst     int
	penaltyLimit    rate.Limit
	penaltyBurst    int
	penaltyDuration time.Duration
	penaltyTrigger  int

	// Ban list (rate limit = 0)
	banListFile string
	banList     map[string]struct{}
	banListMu   sync.RWMutex

	// Local Address Blacklist
	localBlacklistFile string
	localBlacklist     map[string]struct{}
	localBlacklistMu   sync.RWMutex
)

type visitor struct {
	limiter             *rate.Limiter
	lastSeen            time.Time
	punishedUntil       time.Time
	blockedCount        int
	lastBlockTime       time.Time
	isStaticBlacklisted bool
	isBanned            bool
}

func main() {
	// Initialize database connection
	var err error
	dsn := os.Getenv("MYSQL_DSN") // Example: "user:password@tcp(127.0.0.1:3306)/dbname"
	db, err = sql.Open("mysql", dsn)
	if err != nil {
		log.Fatalf("Database connection failed: %v", err)
	}
	defer func() {
		if err := db.Close(); err != nil {
			log.Printf("db close error: %v", err)
		}
	}()

	// Initialize eth client for blacklist checks if backend provided
	if rpcBackend != "" {
		ehtCli, err := ethclient.Dial(rpcBackend)
		if err != nil {
			log.Fatalf("failed to create eth client: %v", err)
		}
		ethClient = ehtCli
	} else {
		log.Printf("ETH_RPC_BACKEND not set, blacklist checks will be disabled")
	}

	// Setup blacklist contract address from env if provided
	if addr := os.Getenv("BLACKLIST_CONTRACT_ADDR"); addr != "" {
		blacklistContract = common.HexToAddress(addr)
	} else {
		// leave zero address; checks will be skipped
		log.Printf("BLACKLIST_CONTRACT_ADDR not set, blacklist checks will be disabled")
	}

	// Load whitelist file and start watcher
	whitelistFile = os.Getenv("WHITELIST_FILE")
	if whitelistFile != "" {
		wlExact, wlPatterns := loadWhitelist(whitelistFile)
		whitelistMu.Lock()
		whitelist = wlExact
		whitelistPatterns = wlPatterns
		whitelistMu.Unlock()
		log.Printf("loaded whitelist entries: %d exact, %d patterns", len(wlExact), len(wlPatterns))
		startWhitelistWatcher(whitelistFile)
	} else {
		whitelist = map[string]struct{}{}
		whitelistPatterns = []*regexp.Regexp{}
		log.Printf("WHITELIST_FILE not set, whitelist feature disabled")
	}

	// Load IP blacklist file and start watcher
	ipBlacklistFile = os.Getenv("IP_BLACKLIST_FILE")
	if ipBlacklistFile != "" {
		blExact, blPatterns := loadIPBlacklist(ipBlacklistFile)
		ipBlacklistMu.Lock()
		ipBlacklist = blExact
		ipBlacklistPatterns = blPatterns
		ipBlacklistMu.Unlock()
		log.Printf("loaded ip blacklist entries: %d exact, %d patterns", len(blExact), len(blPatterns))
		startIPBlacklistWatcher(ipBlacklistFile)
	} else {
		ipBlacklist = map[string]struct{}{}
		ipBlacklistPatterns = []*regexp.Regexp{}
		log.Printf("IP_BLACKLIST_FILE not set, ip blacklist feature disabled")
	}

	// Load Ban list file
	banListFile = os.Getenv("BAN_LIST_FILE")
	if banListFile == "" {
		banListFile = "banlist.txt" // default file
	}
	banList = loadBanList(banListFile)
	log.Printf("loaded ban list entries: %d", len(banList))

	// Load Local Address Blacklist
	localBlacklistFile = os.Getenv("LOCAL_BLACKLIST_FILE")
	if localBlacklistFile != "" {
		localBlacklist = loadLocalBlacklist(localBlacklistFile)
		log.Printf("loaded local address blacklist entries: %d", len(localBlacklist))
		startLocalBlacklistWatcher(localBlacklistFile)
	} else {
		localBlacklist = map[string]struct{}{}
		log.Printf("LOCAL_BLACKLIST_FILE not set, local address blacklist disabled")
	}

	// Start cache janitor for blacklist cache. Interval can be configured via env BLACKLIST_CACHE_CLEANUP_INTERVAL (e.g. "5m").
	cleanupInterval := 5 * time.Minute
	if s := os.Getenv("BLACKLIST_CACHE_CLEANUP_INTERVAL"); s != "" {
		if d, err := time.ParseDuration(s); err == nil {
			cleanupInterval = d
		} else {
			log.Printf("invalid BLACKLIST_CACHE_CLEANUP_INTERVAL '%s', using default %s", s, cleanupInterval)
		}
	}
	startBlacklistCacheJanitor(cleanupInterval)

	// Initialize rate limiter settings
	// Normal Mode: High capacity for normal usage
	nLimit := 1000.0
	if s := os.Getenv("RATE_LIMIT_NORMAL"); s != "" {
		if val, err := strconv.ParseFloat(s, 64); err == nil {
			nLimit = val
		}
	}
	normalLimit = rate.Every(15 * time.Minute / time.Duration(nLimit))

	normalBurst = 20 // default burst
	if s := os.Getenv("RATE_BURST_NORMAL"); s != "" {
		if val, err := strconv.Atoi(s); err == nil {
			normalBurst = val
		}
	}

	// Penalty Mode: Strict limit for abusive IPs
	pLimit := 100.0
	if s := os.Getenv("RATE_LIMIT_PENALTY"); s != "" {
		if val, err := strconv.ParseFloat(s, 64); err == nil {
			pLimit = val
		}
	}
	penaltyLimit = rate.Every(15 * time.Minute / time.Duration(pLimit))

	penaltyBurst = 5 // default burst
	if s := os.Getenv("RATE_BURST_PENALTY"); s != "" {
		if val, err := strconv.Atoi(s); err == nil {
			penaltyBurst = val
		}
	}

	penaltyDuration = 20 * time.Minute
	if s := os.Getenv("PENALTY_DURATION"); s != "" {
		if d, err := time.ParseDuration(s); err == nil {
			penaltyDuration = d
		}
	}

	penaltyTrigger = 5
	if s := os.Getenv("PENALTY_TRIGGER_COUNT"); s != "" {
		if val, err := strconv.Atoi(s); err == nil {
			penaltyTrigger = val
		}
	}

	startRateLimitCleaner()

	http.HandleFunc("/", proxyHandler)
	log.Println("RPC proxy service started, listening on port: 8545")
	log.Fatal(http.ListenAndServe(":8545", nil))
}

func isSendTxFromBlacklist(req RPCRequest, ip string) bool {
	if req.Method == "eth_sendRawTransaction" && len(req.Params) > 0 {
		if rawHex, ok := req.Params[0].(string); ok {
			if fromAddr, err := getSenderFromRawTx(rawHex); err == nil {
				// Check local blacklist first
				if isLocalBlacklisted(fromAddr) {
					log.Printf("sender %s is in local blacklist, banning IP %s\n", fromAddr, ip)
					return true
				} else {
					// check in contract blacklist.
					if ethClient != nil && blacklistContract != (common.Address{}) {
						inBlack, _ := CachedIsInBlacklist(ethClient, blacklistContract, common.HexToAddress(strings.ToLower(fromAddr)))
						if inBlack {
							log.Printf("sender %s is in contract blacklist, banning IP %s\n", fromAddr, ip)
							return inBlack
						}
					}
				}
			}
		}
	}
	return false
}

func isFromInBlacklist(addr string) bool {
	if isLocalBlacklisted(addr) {
		return true
	} else {
		// check in contract blacklist.
		if ethClient != nil && blacklistContract != (common.Address{}) {
			inBlack, _ := CachedIsInBlacklist(ethClient, blacklistContract, common.HexToAddress(strings.ToLower(addr)))
			if inBlack {
				return inBlack
			}
		}
	}
	return false
}

func proxyHandler(w http.ResponseWriter, r *http.Request) {
	// Add CORS support
	setCORSHeaders(w, r)

	// Handle preflight requests
	if r.Method == "OPTIONS" {
		w.WriteHeader(http.StatusOK)
		return
	}

	body, err := io.ReadAll(r.Body)
	if err != nil {
		http.Error(w, "Failed to read request", http.StatusBadRequest)
		return
	}
	defer func() {
		if err := r.Body.Close(); err != nil {
			log.Printf("request body close error: %v", err)
		}
	}()

	// Whitelist short-circuit: if Origin or Referer matches whitelist, forward immediately
	origin := r.Header.Get("Origin")
	referer := r.Header.Get("Referer") // typical header key
	ip := getClientIP(r)
	if isWhitelisted(origin) || isWhitelisted(referer) || isWhitelisted(ip) {
		log.Printf("whitelist matched (origin=%s referer=%s ip=%s), forwarding directly", origin, referer, ip)
		forwardToBackend(w, body)
		return
	}

	// Rate limiting for non-whitelisted requests
	// ip is already extracted above
	if !checkRateLimit(ip) {
		http.Error(w, "Too Many Requests", http.StatusTooManyRequests)
		log.Printf("rate limit exceeded for IP: %s", ip)
		return
	}

	var reqs []RPCRequest
	// Try to parse as batch request first
	if err := json.Unmarshal(body, &reqs); err == nil {
		// Handle batch request
		if len(reqs) > 5 {
			req := reqs[0]
			resp := RPCResponse{
				Jsonrpc: req.Jsonrpc,
				Id:      req.Id,
				Result:  "invalid batch request",
			}
			xForwardedFor := r.Header.Get("X-Forwarded-For")
			log.Printf("stop forward to rpc on batch request, request from X-Forwarded-For: %s, param[0]: %v", xForwardedFor, req.Params)
			w.Header().Set("Content-Type", "application/json")
			if err := json.NewEncoder(w).Encode(resp); err != nil {
				log.Printf("encode response error: %v", err)
			}
		} else {
			for _, req := range reqs {
				if isSendTxFromBlacklist(req, ip) {
					addToBanList(ip)
					errResp := RPCResponse{
						Jsonrpc: req.Jsonrpc,
						Id:      req.Id,
						Error: map[string]interface{}{
							"code":    -32000,
							"message": "sender is denied",
						},
					}
					w.Header().Set("Content-Type", "application/json")
					if err := json.NewEncoder(w).Encode(errResp); err != nil {
						log.Printf("encode error: %v", err)
					}
					return
				}
				if from, ok := extractFromAddress(req); ok {
					if isFromInBlacklist(from) {
						errResp := RPCResponse{
							Jsonrpc: req.Jsonrpc,
							Id:      req.Id,
							Error: map[string]interface{}{
								"code":    -32000,
								"message": "sender is denied",
							},
						}
						w.Header().Set("Content-Type", "application/json")
						if err := json.NewEncoder(w).Encode(errResp); err != nil {
							log.Printf("encode error: %v", err)
						}
						return
					}
				}
			}
			forwardToBackend(w, body)
		}
		return
	}

	var req RPCRequest
	if err := json.Unmarshal(body, &req); err != nil {
		forwardToBackend(w, body)
		return
	}
	log.Printf("%s request from %s", req.Method, ip)

	// Handle eth_getBalance (existing behavior)
	if req.Method == "eth_getBalance" && len(req.Params) > 0 {
		// get remote ip from header
		xForwardedFor := r.Header.Get("X-Forwarded-For")
		realIp := r.Header.Get("X-Real-IP")
		log.Printf("eth_getBalance request from %s, X-Real-IP: %s, X-Forwarded-For: %s, address: %v", r.RemoteAddr, realIp, xForwardedFor, req.Params[0])
		address, ok := req.Params[0].(string)
		if !ok {
			forwardToBackend(w, body)
			return
		}
		if !accountExists(strings.ToLower(address)) {
			resp := RPCResponse{
				Jsonrpc: req.Jsonrpc,
				Id:      req.Id,
				Result:  "0x0",
			}
			log.Printf("stop forward to rpc on eth_getBalance request from %s, X-Real-IP: %s, X-Forwarded-For: %s, address: %v", r.RemoteAddr, realIp, xForwardedFor, req.Params[0])
			w.Header().Set("Content-Type", "application/json")
			if err := json.NewEncoder(w).Encode(resp); err != nil {
				log.Printf("encode response error: %v", err)
			}
			return
		}
	}

	// forbid eth_sendRawTransaction if sender is blacklisted, add to ban list and return error.
	if isSendTxFromBlacklist(req, ip) {
		addToBanList(ip)
		errResp := RPCResponse{
			Jsonrpc: req.Jsonrpc,
			Id:      req.Id,
			Error: map[string]interface{}{
				"code":    -32000,
				"message": "sender is blacklisted",
			},
		}
		w.Header().Set("Content-Type", "application/json")
		if err := json.NewEncoder(w).Encode(errResp); err != nil {
			log.Printf("encode error: %v", err)
		}
		return
	}

	// New: check if request includes a `from` address (common for eth_sendTransaction, eth_call, eth_estimateGas)
	if from, ok := extractFromAddress(req); ok {
		if isFromInBlacklist(from) {
			// return JSON-RPC error response indicating sender is blacklisted
			errResp := RPCResponse{
				Jsonrpc: req.Jsonrpc,
				Id:      req.Id,
				Error: map[string]interface{}{
					"code":    -32000,
					"message": "sender is blacklisted",
				},
			}
			w.Header().Set("Content-Type", "application/json")
			if err := json.NewEncoder(w).Encode(errResp); err != nil {
				log.Printf("encode error: %v", err)
			}
			return
		}
	}

	// Forward other cases directly
	forwardToBackend(w, body)
}

// getSenderFromRawTx decodes a raw transaction hex (0x...) and returns the sender address as a hex string.
func getSenderFromRawTx(rawHex string) (string, error) {
	// strip 0x if present
	b, err := hexutil.Decode(rawHex)
	if err != nil {
		return "", err
	}
	var tx types.Transaction
	if err := tx.UnmarshalBinary(b); err != nil {
		return "", err
	}
	// determine signer
	var signer types.Signer
	if tx.ChainId() != nil {
		signer = types.LatestSignerForChainID(tx.ChainId())
	} else {
		signer = types.HomesteadSigner{}
	}
	from, err := types.Sender(signer, &tx)
	if err != nil {
		return "", err
	}
	return strings.ToLower(from.Hex()), nil
}

func setCORSHeaders(w http.ResponseWriter, r *http.Request) {
	origin := r.Header.Get("Origin")
	if origin != "" {
		w.Header().Set("Access-Control-Allow-Origin", origin)
	} else {
		w.Header().Set("Access-Control-Allow-Origin", "*")
	}

	w.Header().Set("Access-Control-Allow-Methods", "POST, GET, OPTIONS, PUT, DELETE")
	w.Header().Set("Access-Control-Allow-Headers", "Accept, Content-Type, Content-Length, Accept-Encoding, X-CSRF-Token, Authorization, X-Real-IP, X-Forwarded-For")
	w.Header().Set("Access-Control-Allow-Credentials", "true")
	w.Header().Set("Access-Control-Max-Age", "86400")
}

// extractFromAddress inspects JSON-RPC params and returns the `from` address if present.
func extractFromAddress(req RPCRequest) (string, bool) {
	// Typical shapes: params[0] is an object with field "from"
	if len(req.Params) == 0 {
		return "", false
	}

	// Check first param
	if obj, ok := req.Params[0].(map[string]interface{}); ok {
		if f, ok2 := obj["from"].(string); ok2 && f != "" {
			return strings.ToLower(f), true
		}
	}

	// Fallback: check all params for an object that includes "from"
	for _, p := range req.Params {
		if obj, ok := p.(map[string]interface{}); ok {
			if f, ok2 := obj["from"].(string); ok2 && f != "" {
				return strings.ToLower(f), true
			}
		}
	}
	return "", false
}

func accountExists(address string) bool {
	var count int
	query := "SELECT COUNT(1) FROM tb_account_info WHERE account_address = ? AND is_deleted = 0"
	err := db.QueryRow(query, address).Scan(&count)
	if err != nil {
		if err == sql.ErrNoRows {
			return false
		}
		log.Printf("Database query error: %v", err)
		// fail open (treat as exists to allow forwarding)
		return true
	}
	return count > 0
}

// Forward request body to backend RPC and copy response headers/body.
func forwardToBackend(w http.ResponseWriter, body []byte) {
	if rpcBackend == "" {
		http.Error(w, "Backend RPC not configured", http.StatusServiceUnavailable)
		return
	}
	resp, err := http.Post(rpcBackend, "application/json", bytes.NewReader(body))
	if err != nil {
		http.Error(w, "Backend RPC request failed", http.StatusBadGateway)
		return
	}
	defer resp.Body.Close()
	for k, vals := range resp.Header {
		for _, v := range vals {
			w.Header().Add(k, v)
		}
	}
	w.WriteHeader(resp.StatusCode)
	io.Copy(w, resp.Body)
}

// ===== Whitelist helper functions (dynamic reload) =====
func loadWhitelist(path string) (map[string]struct{}, []*regexp.Regexp) {
	result := make(map[string]struct{})
	patterns := make([]*regexp.Regexp, 0)
	f, err := os.Open(path)
	if err != nil {
		log.Printf("open whitelist file '%s' error: %v", path, err)
		return result, patterns
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)
	lineNum := 0
	for scanner.Scan() {
		lineNum++
		line := strings.TrimSpace(scanner.Text())
		if line == "" || strings.HasPrefix(line, "#") || strings.HasPrefix(line, "//") {
			continue
		}
		// Determine if wildcard pattern present
		if strings.Contains(line, "*") {
			if re := compilePattern(line); re != nil {
				patterns = append(patterns, re)
			} else {
				log.Printf("skip invalid pattern at line %d: %s", lineNum, line)
			}
			continue
		}
		result[line] = struct{}{}
	}
	if err := scanner.Err(); err != nil {
		log.Printf("scan whitelist file error: %v", err)
	}
	return result, patterns
}

func isWhitelisted(v string) bool {
	if v == "" {
		return false
	}
	val := strings.TrimSpace(v)
	if val == "" {
		return false
	}

	// Collect candidate forms: raw, origin base (scheme://host), host (strip port), host:port
	candidates := make([]string, 0, 4)
	candidates = append(candidates, val)
	if u, err := url.Parse(val); err == nil && u.Host != "" {
		host := u.Host
		// strip port for host-only
		if strings.Contains(host, ":") {
			parts := strings.Split(host, ":")
			hostNoPort := parts[0]
			candidates = append(candidates, hostNoPort)
		}
		candidates = append(candidates, host)
		base := fmt.Sprintf("%s://%s", u.Scheme, host)
		candidates = append(candidates, base)
	}

	whitelistMu.RLock()
	defer whitelistMu.RUnlock()
	for _, c := range candidates {
		if _, ok := whitelist[c]; ok {
			return true
		}
	}
	// pattern matching
	for _, re := range whitelistPatterns {
		for _, c := range candidates {
			if re.MatchString(c) {
				return true
			}
		}
	}
	return false
}

// compilePattern converts a whitelist line with '*' wildcards to a safe anchored regexp.
// Supported examples:
//
//	*.example.com    -> subdomains of example.com
//	example.com*     -> prefix match
//	*example.com     -> suffix match
//	*mid*            -> substring match
//	https://*.foo.bar -> scheme + subdomain
func compilePattern(p string) *regexp.Regexp {
	p = strings.TrimSpace(p)
	if p == "" {
		return nil
	}
	// Special host wildcard prefix '*.'
	if strings.HasPrefix(p, "*.") {
		// Allow one or more subdomain levels
		root := strings.TrimPrefix(p, "*.")
		escaped := regexp.QuoteMeta(root)
		pattern := fmt.Sprintf(`^(?:[^.]+\.)+%s$`, escaped)
		re, err := regexp.Compile(pattern)
		if err != nil {
			log.Printf("compile pattern error (%s): %v", p, err)
			return nil
		}
		return re
	}
	// General case: escape then replace '*' with '.*'
	esc := regexp.QuoteMeta(p)
	esc = strings.ReplaceAll(esc, `*`, `.*`)
	pattern := fmt.Sprintf("^%s$", esc)
	re, err := regexp.Compile(pattern)
	if err != nil {
		log.Printf("compile pattern error (%s): %v", p, err)
		return nil
	}
	return re
}

func startWhitelistWatcher(path string) {
	watcher, err := fsnotify.NewWatcher()
	if err != nil {
		log.Printf("create whitelist watcher error: %v", err)
		return
	}
	dir := filepath.Dir(path)
	if err := watcher.Add(dir); err != nil {
		log.Printf("add whitelist watch dir error: %v", err)
		watcher.Close()
		return
	}
	go func() {
		defer watcher.Close()
		for {
			select {
			case ev, ok := <-watcher.Events:
				if !ok {
					return
				}
				if ev.Name == path {
					if ev.Op&(fsnotify.Write|fsnotify.Create|fsnotify.Rename) != 0 {
						exact, pats := loadWhitelist(path)
						whitelistMu.Lock()
						whitelist = exact
						whitelistPatterns = pats
						whitelistMu.Unlock()
						log.Printf("whitelist reloaded (%d exact, %d patterns) due to event: %s", len(exact), len(pats), ev.Op.String())
					}
					if ev.Op&fsnotify.Remove != 0 {
						whitelistMu.Lock()
						whitelist = map[string]struct{}{}
						whitelistPatterns = []*regexp.Regexp{}
						whitelistMu.Unlock()
						log.Printf("whitelist file removed, cleared entries")
					}
				}
			case err, ok := <-watcher.Errors:
				if !ok {
					return
				}
				log.Printf("whitelist watcher error: %v", err)
			}
		}
	}()
}

// ===== IP Blacklist helper functions (dynamic reload) =====
func loadIPBlacklist(path string) (map[string]struct{}, []*regexp.Regexp) {
	result := make(map[string]struct{})
	patterns := make([]*regexp.Regexp, 0)
	f, err := os.Open(path)
	if err != nil {
		log.Printf("open ip blacklist file '%s' error: %v", path, err)
		return result, patterns
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)
	lineNum := 0
	for scanner.Scan() {
		lineNum++
		line := strings.TrimSpace(scanner.Text())
		if line == "" || strings.HasPrefix(line, "#") || strings.HasPrefix(line, "//") {
			continue
		}
		// Determine if wildcard pattern present
		if strings.Contains(line, "*") {
			if re := compilePattern(line); re != nil {
				patterns = append(patterns, re)
			} else {
				log.Printf("skip invalid pattern at line %d: %s", lineNum, line)
			}
			continue
		}
		result[line] = struct{}{}
	}
	if err := scanner.Err(); err != nil {
		log.Printf("scan ip blacklist file error: %v", err)
	}
	return result, patterns
}

func isIPBlacklisted(v string) bool {
	if v == "" {
		return false
	}
	val := strings.TrimSpace(v)
	if val == "" {
		return false
	}

	// Collect candidate forms: raw, origin base (scheme://host), host (strip port), host:port
	candidates := make([]string, 0, 4)
	candidates = append(candidates, val)
	if u, err := url.Parse(val); err == nil && u.Host != "" {
		host := u.Host
		// strip port for host-only
		if strings.Contains(host, ":") {
			parts := strings.Split(host, ":")
			hostNoPort := parts[0]
			candidates = append(candidates, hostNoPort)
		}
		candidates = append(candidates, host)
		base := fmt.Sprintf("%s://%s", u.Scheme, host)
		candidates = append(candidates, base)
	}

	ipBlacklistMu.RLock()
	defer ipBlacklistMu.RUnlock()
	for _, c := range candidates {
		if _, ok := ipBlacklist[c]; ok {
			return true
		}
	}
	// pattern matching
	for _, re := range ipBlacklistPatterns {
		for _, c := range candidates {
			if re.MatchString(c) {
				return true
			}
		}
	}
	return false
}

func startIPBlacklistWatcher(path string) {
	watcher, err := fsnotify.NewWatcher()
	if err != nil {
		log.Printf("create ip blacklist watcher error: %v", err)
		return
	}
	dir := filepath.Dir(path)
	if err := watcher.Add(dir); err != nil {
		log.Printf("add ip blacklist watch dir error: %v", err)
		watcher.Close()
		return
	}
	go func() {
		defer watcher.Close()
		for {
			select {
			case ev, ok := <-watcher.Events:
				if !ok {
					return
				}
				if ev.Name == path {
					if ev.Op&(fsnotify.Write|fsnotify.Create|fsnotify.Rename) != 0 {
						exact, pats := loadIPBlacklist(path)
						ipBlacklistMu.Lock()
						ipBlacklist = exact
						ipBlacklistPatterns = pats
						ipBlacklistMu.Unlock()
						log.Printf("ip blacklist reloaded (%d exact, %d patterns) due to event: %s", len(exact), len(pats), ev.Op.String())
					}
					if ev.Op&fsnotify.Remove != 0 {
						ipBlacklistMu.Lock()
						ipBlacklist = map[string]struct{}{}
						ipBlacklistPatterns = []*regexp.Regexp{}
						ipBlacklistMu.Unlock()
						log.Printf("ip blacklist file removed, cleared entries")
					}
				}
			case err, ok := <-watcher.Errors:
				if !ok {
					return
				}
				log.Printf("ip blacklist watcher error: %v", err)
			}
		}
	}()
}

// ===== Local Address Blacklist helper functions =====
func loadLocalBlacklist(path string) map[string]struct{} {
	result := make(map[string]struct{})
	f, err := os.Open(path)
	if err != nil {
		log.Printf("open local blacklist file '%s' error: %v", path, err)
		return result
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if line == "" || strings.HasPrefix(line, "#") || strings.HasPrefix(line, "//") {
			continue
		}
		// Assume addresses are hex strings, normalize to lowercase
		addr := strings.ToLower(line)
		result[addr] = struct{}{}
	}
	return result
}

func isLocalBlacklisted(addr string) bool {
	localBlacklistMu.RLock()
	defer localBlacklistMu.RUnlock()
	_, ok := localBlacklist[strings.ToLower(addr)]
	return ok
}

func startLocalBlacklistWatcher(path string) {
	watcher, err := fsnotify.NewWatcher()
	if err != nil {
		log.Printf("create local blacklist watcher error: %v", err)
		return
	}
	dir := filepath.Dir(path)
	if err := watcher.Add(dir); err != nil {
		log.Printf("add local blacklist watch dir error: %v", err)
		watcher.Close()
		return
	}
	go func() {
		defer watcher.Close()
		for {
			select {
			case ev, ok := <-watcher.Events:
				if !ok {
					return
				}
				if ev.Name == path {
					if ev.Op&(fsnotify.Write|fsnotify.Create|fsnotify.Rename) != 0 {
						bl := loadLocalBlacklist(path)
						localBlacklistMu.Lock()
						localBlacklist = bl
						localBlacklistMu.Unlock()
						log.Printf("local blacklist reloaded (%d entries) due to event: %s", len(bl), ev.Op.String())
					}
					if ev.Op&fsnotify.Remove != 0 {
						localBlacklistMu.Lock()
						localBlacklist = map[string]struct{}{}
						localBlacklistMu.Unlock()
						log.Printf("local blacklist file removed, cleared entries")
					}
				}
			case err, ok := <-watcher.Errors:
				if !ok {
					return
				}
				log.Printf("local blacklist watcher error: %v", err)
			}
		}
	}()
}

// ===== Ban List helper functions =====
func loadBanList(path string) map[string]struct{} {
	result := make(map[string]struct{})
	f, err := os.Open(path)
	if err != nil {
		if !os.IsNotExist(err) {
			log.Printf("open ban list file '%s' error: %v", path, err)
		}
		return result
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if line != "" {
			result[line] = struct{}{}
		}
	}
	return result
}

func addToBanList(ip string) {
	banListMu.Lock()
	defer banListMu.Unlock()

	if _, exists := banList[ip]; exists {
		return
	}
	banList[ip] = struct{}{}

	// Append to file
	f, err := os.OpenFile(banListFile, os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
	if err != nil {
		log.Printf("failed to open ban list file for appending: %v", err)
		return
	}
	defer f.Close()
	if _, err := f.WriteString(ip + "\n"); err != nil {
		log.Printf("failed to write to ban list file: %v", err)
	}
}

func isBanned(ip string) bool {
	banListMu.RLock()
	defer banListMu.RUnlock()
	_, ok := banList[ip]
	return ok
}

func checkRateLimit(ip string) bool {
	visitorsMu.Lock()
	defer visitorsMu.Unlock()

	v, exists := visitors[ip]
	if !exists {
		limiter := rate.NewLimiter(normalLimit, normalBurst)
		v = &visitor{limiter: limiter, lastSeen: time.Now()}
		visitors[ip] = v
	}
	v.lastSeen = time.Now()

	// Check if IP is banned (rate limit 0)
	if isBanned(ip) {
		if !v.isBanned {
			v.isBanned = true
			v.limiter.SetLimit(0)
			v.limiter.SetBurst(0)
			log.Printf("IP %s is banned, blocking all requests", ip)
		}
		return false
	}

	// Check if IP is in static blacklist
	if isIPBlacklisted(ip) {
		if !v.isStaticBlacklisted {
			// First time detected as blacklisted, force penalty mode
			v.isStaticBlacklisted = true
			v.limiter.SetLimit(penaltyLimit)
			v.limiter.SetBurst(penaltyBurst)
			// Set a long punishment duration to avoid frequent checks, or just rely on the flag
			v.punishedUntil = time.Now().Add(24 * time.Hour)
			log.Printf("IP %s is in static blacklist, enforcing penalty mode", ip)
		} else {
			// Refresh punishment duration to keep it in penalty mode
			if time.Now().After(v.punishedUntil) {
				v.punishedUntil = time.Now().Add(24 * time.Hour)
			}
		}
		return v.limiter.Allow()
	} else if v.isStaticBlacklisted {
		// Was blacklisted but now removed from static blacklist
		v.isStaticBlacklisted = false
		v.punishedUntil = time.Time{} // Clear punishment
		v.limiter.SetLimit(normalLimit)
		v.limiter.SetBurst(normalBurst)
		log.Printf("IP %s removed from static blacklist, restoring normal limits", ip)
	}

	// Check if penalty expired (for dynamic penalty)
	if !v.punishedUntil.IsZero() {
		if time.Now().After(v.punishedUntil) {
			// Penalty expired, restore normal settings
			v.limiter.SetLimit(normalLimit)
			v.limiter.SetBurst(normalBurst)
			v.punishedUntil = time.Time{}
			v.blockedCount = 0
			log.Printf("IP %s penalty expired, restoring normal limits", ip)
		}
	}

	allowed := v.limiter.Allow()

	if !allowed {
		v.blockedCount++
		v.lastBlockTime = time.Now()

		// Only trigger penalty if not already punished
		if v.punishedUntil.IsZero() && v.blockedCount >= penaltyTrigger {
			v.punishedUntil = time.Now().Add(penaltyDuration)
			v.limiter.SetLimit(penaltyLimit)
			v.limiter.SetBurst(penaltyBurst)
			log.Printf("IP %s entered penalty mode (high traffic), limited to %v rps until %v", ip, penaltyLimit, v.punishedUntil)
		}
	} else {
		// If allowed and not punished, maybe decay blockedCount?
		// If it's been a while since last block, reset counter to avoid long-term accumulation
		if v.punishedUntil.IsZero() && v.blockedCount > 0 && time.Since(v.lastBlockTime) > 1*time.Minute {
			v.blockedCount = 0
		}
	}

	return allowed
}

func startRateLimitCleaner() {
	go func() {
		for {
			time.Sleep(5 * time.Minute)
			visitorsMu.Lock()
			for ip, v := range visitors {
				// Keep punished visitors until penalty expires
				if !v.punishedUntil.IsZero() && time.Now().Before(v.punishedUntil) {
					continue
				}
				if time.Since(v.lastSeen) > 15*time.Minute {
					delete(visitors, ip)
				}
			}
			visitorsMu.Unlock()
		}
	}()
}

func getClientIP(r *http.Request) string {
	// Try X-Forwarded-For
	xff := r.Header.Get("X-Forwarded-For")
	if xff != "" {
		parts := strings.Split(xff, ",")
		return strings.TrimSpace(parts[0])
	}
	// Try X-Real-IP
	xrip := r.Header.Get("X-Real-IP")
	if xrip != "" {
		return strings.TrimSpace(xrip)
	}
	// Fallback to RemoteAddr
	host, _, err := net.SplitHostPort(r.RemoteAddr)
	if err == nil {
		return host
	}
	return r.RemoteAddr
}
