package main

import (
	"code.wuban.net.cn/luxueqian/rpcproxy/contracts/blacklist"
	"context"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"strings"
	"sync"
	"time"

	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/ethclient"
)

// ABI for the blacklist contract that exposes: function inBlacklist(address) view returns (bool)
const blacklistABI = `[{"constant":true,"inputs":[{"name":"addr","type":"address"}],"name":"inBlacklist","outputs":[{"name":"","type":"bool"}],"payable":false,"stateMutability":"view","type":"function"}]`

// cacheEntry holds the cached blacklist boolean and expiry time
type cacheEntry struct {
	inBlack bool
	expiry  time.Time
}

var (
	blacklistCache   = make(map[string]cacheEntry)
	blacklistCacheMu sync.RWMutex
)

// IsInBlacklist calls the contract's inBlacklist(address) view function and returns its boolean result.
// Inputs:
// - client: an initialized *ethclient.Client connected to an Ethereum node
// - contract: address of the blacklist contract
// - addr: address to check
// Output:
// - bool indicating whether the address is in the blacklist
// - error in case of RPC/ABI/decoding problems
func IsInBlacklist(client *ethclient.Client, contract common.Address, addr common.Address) (bool, error) {
	if client == nil {
		return false, fmt.Errorf("nil ethclient")
	}
	contraction, err := blacklist.NewBlacklist(contract, client)
	if err != nil {
		return false, fmt.Errorf("failed to bind contract: %w", err)
	}
	opts := &bind.CallOpts{
		From:        addr,
		BlockNumber: nil,
		Context:     context.Background(),
	}
	results, err := contraction.InBlackList(opts, []common.Address{addr})
	if err != nil {
		return false, fmt.Errorf("contract call failed: %w", err)
	}
	if len(results) > 0 && results[0] {
		return true, nil
	}
	return false, nil
}

// CachedIsInBlacklist checks an in-memory cache before calling the contract.
// Cache policy:
// - if address is in blacklist -> cache for 1 hour
// - if address is not in blacklist -> cache for 1 minute
// On contract errors, the error is returned and nothing is cached.
func CachedIsInBlacklist(client *ethclient.Client, contract common.Address, addr common.Address) (bool, error) {
	key := strings.ToLower(addr.Hex())
	now := time.Now()

	// fast-path: read lock
	blacklistCacheMu.RLock()
	entry, ok := blacklistCache[key]
	blacklistCacheMu.RUnlock()
	if ok {
		if now.Before(entry.expiry) {
			return entry.inBlack, nil
		}
		// entry expired: proactively remove it to release memory sooner
		blacklistCacheMu.Lock()
		// re-check in case another goroutine updated it
		if cur, still := blacklistCache[key]; still {
			if cur.expiry.Equal(entry.expiry) || cur.expiry.Before(now) {
				delete(blacklistCache, key)
			}
		}
		blacklistCacheMu.Unlock()
	}

	// Miss or expired -> query contract
	inBlack, err := IsInBlacklist(client, contract, addr)
	if err != nil {
		return false, err
	}

	// determine ttl
	var ttl time.Duration
	if inBlack {
		ttl = time.Hour
	} else {
		ttl = time.Minute
	}

	blacklistCacheMu.Lock()
	blacklistCache[key] = cacheEntry{inBlack: inBlack, expiry: now.Add(ttl)}
	blacklistCacheMu.Unlock()

	return inBlack, nil
}

// startBlacklistCacheJanitor starts a background goroutine that periodically
// scans and removes expired entries from the in-memory blacklist cache.
// Call this once at startup. The cleanupInterval controls how often the scan runs.
func startBlacklistCacheJanitor(cleanupInterval time.Duration) {
	if cleanupInterval <= 0 {
		cleanupInterval = time.Minute * 5
	}
	go func() {
		ticker := time.NewTicker(cleanupInterval)
		defer ticker.Stop()
		for range ticker.C {
			now := time.Now()
			blacklistCacheMu.Lock()
			for k, e := range blacklistCache {
				if now.After(e.expiry) {
					delete(blacklistCache, k)
				}
			}
			blacklistCacheMu.Unlock()
		}
	}()
}

// ClearBlacklistCache removes all entries from the cache (useful for tests/admin).
func ClearBlacklistCache() {
	blacklistCacheMu.Lock()
	blacklistCache = make(map[string]cacheEntry)
	blacklistCacheMu.Unlock()
}

// CacheStats returns the current number of entries and the nearest expiry time (zero if empty).
func CacheStats() (count int, nextExpiry time.Time) {
	blacklistCacheMu.RLock()
	defer blacklistCacheMu.RUnlock()
	count = len(blacklistCache)
	var earliest time.Time
	for _, e := range blacklistCache {
		if earliest.IsZero() || e.expiry.Before(earliest) {
			earliest = e.expiry
		}
	}
	return count, earliest
}
