package main

import (
	"bytes"
	"database/sql"
	"encoding/json"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"strings"
	"time"

	_ "github.com/go-sql-driver/mysql"
)

type TbAccountInfo struct {
	Id             int64     `json:"id"`
	BlockId        int64     `json:"block_id"`
	BlockHash      string    `json:"block_hash"`
	TxHash         string    `json:"tx_hash"`
	AccountAddress string    `json:"account_address"`
	AccountType    int       `json:"account_type"`
	MyNameTag      string    `json:"my_name_tag"`
	Balance        float64   `json:"balance"`
	Status         int       `json:"status"`
	IsDeleted      int8      `json:"is_deleted"`
	SyncTime       time.Time `json:"sync_time"`
	CreateTime     time.Time `json:"create_time"`
	UpdateTime     time.Time `json:"update_time"`
}

func (t *TbAccountInfo) TableName() string {
	return "tb_account_info"
}

// RPC request structure
// Only handle params for eth_getBalance here
// Forward other methods directly

type RPCRequest struct {
	Jsonrpc string        `json:"jsonrpc"`
	Method  string        `json:"method"`
	Params  []interface{} `json:"params"`
	Id      interface{}   `json:"id"`
}

type RPCResponse struct {
	Jsonrpc string      `json:"jsonrpc"`
	Id      interface{} `json:"id"`
	Result  interface{} `json:"result"`
	Error   interface{} `json:"error,omitempty"`
}

var (
	db         *sql.DB
	rpcBackend = os.Getenv("ETH_RPC_BACKEND") // Real Ethereum RPC address, recommend using environment variable
)

func main() {
	// Initialize database connection
	var err error
	dsn := os.Getenv("MYSQL_DSN") // Example: "user:password@tcp(127.0.0.1:3306)/dbname"
	db, err = sql.Open("mysql", dsn)
	if err != nil {
		log.Fatalf("Database connection failed: %v", err)
	}
	defer db.Close()

	http.HandleFunc("/", proxyHandler)
	log.Println("RPC proxy service started, listening on port: 8545")
	log.Fatal(http.ListenAndServe(":8545", nil))
}

func proxyHandler(w http.ResponseWriter, r *http.Request) {
	// Add CORS support
	setCORSHeaders(w, r)

	// Handle preflight requests
	if r.Method == "OPTIONS" {
		w.WriteHeader(http.StatusOK)
		return
	}

	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		http.Error(w, "Failed to read request", http.StatusBadRequest)
		return
	}
	defer r.Body.Close()

	var reqs []RPCRequest
	// Try to parse as batch request first
	if err := json.Unmarshal(body, &reqs); err == nil {
		// Handle batch request
		if len(reqs) > 1 {
			req := reqs[0]
			resp := RPCResponse{
				Jsonrpc: req.Jsonrpc,
				Id:      req.Id,
				Result:  "invalid batch request",
			}
			xForwardedFor := r.Header.Get("X-Forwarded-For")
			log.Printf("stop forward to rpc on batch request, request from X-Forwarded-For: %s, param[0]: %v", xForwardedFor, req.Params)
			w.Header().Set("Content-Type", "application/json")
			json.NewEncoder(w).Encode(resp)
		}
		return
	}

	var req RPCRequest
	if err := json.Unmarshal(body, &req); err != nil {
		forwardToBackend(w, body)
		return
	}

	if req.Method == "eth_getBalance" && len(req.Params) > 0 {
		// get remote ip from header
		xForwardedFor := r.Header.Get("X-Forwarded-For")
		realIp := r.Header.Get("X-Real-IP")
		log.Printf("eth_getBalance request from %s, X-Real-IP: %s, X-Forwarded-For: %s, address: %v", r.RemoteAddr, realIp, xForwardedFor, req.Params[0])
		address, ok := req.Params[0].(string)
		if !ok {
			forwardToBackend(w, body)
			return
		}
		if !accountExists(strings.ToLower(address)) {
			resp := RPCResponse{
				Jsonrpc: req.Jsonrpc,
				Id:      req.Id,
				Result:  "0x0",
			}
			log.Printf("stop forward to rpc on eth_getBalance request from %s, X-Real-IP: %s, X-Forwarded-For: %s, address: %v", r.RemoteAddr, realIp, xForwardedFor, req.Params[0])
			w.Header().Set("Content-Type", "application/json")
			json.NewEncoder(w).Encode(resp)
			return
		}
	}
	// Forward other cases directly
	forwardToBackend(w, body)
}

func setCORSHeaders(w http.ResponseWriter, r *http.Request) {
	origin := r.Header.Get("Origin")
	if origin != "" {
		w.Header().Set("Access-Control-Allow-Origin", origin)
	} else {
		w.Header().Set("Access-Control-Allow-Origin", "*")
	}

	w.Header().Set("Access-Control-Allow-Methods", "POST, GET, OPTIONS, PUT, DELETE")
	w.Header().Set("Access-Control-Allow-Headers", "Accept, Content-Type, Content-Length, Accept-Encoding, X-CSRF-Token, Authorization, X-Real-IP, X-Forwarded-For")
	w.Header().Set("Access-Control-Allow-Credentials", "true")
	w.Header().Set("Access-Control-Max-Age", "86400")
}

func accountExists(address string) bool {
	var count int
	query := "SELECT COUNT(1) FROM tb_account_info WHERE account_address = ? AND is_deleted = 0"
	err := db.QueryRow(query, address).Scan(&count)
	if err != nil {
		if err == sql.ErrNoRows {
			return false
		} else {
			log.Printf("Database query error: %v", err)
		}
		return true
	}
	return count > 0
}

func forwardToBackend(w http.ResponseWriter, body []byte) {
	resp, err := http.Post(rpcBackend, "application/json", bytes.NewReader(body))
	if err != nil {
		http.Error(w, "Backend RPC request failed", http.StatusBadGateway)
		return
	}
	defer resp.Body.Close()

	// Copy response headers, including possible CORS headers
	for key, values := range resp.Header {
		for _, value := range values {
			w.Header().Add(key, value)
		}
	}

	w.WriteHeader(resp.StatusCode)
	io.Copy(w, resp.Body)
}
