package tech.pegasys.teku.attacker2;

import com.googlecode.jsonrpc4j.JsonRpcHttpClient;

import tech.pegasys.teku.attacker.client.*;
import tech.pegasys.teku.infrastructure.async.SafeFuture;

public class AttackService {

    private static final String BLOCK_MODULE = "block";

    private static final String ATTEST_MODULE = "attest";
    
    /**
     * Get new parent root for block.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param parentRoot The parent root
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockGetNewParentRoot(long slot, String pubkey, String parentRoot) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_getNewParentRoot",
                new Object[] {slot,pubkey,parentRoot},
                AttackerResponse.class);
    }

    /**
     * Delay for receive block.
     *
     * @param slot The slot number
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> delayForReceiveBlock(long slot) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_delayForReceiveBlock",
                new Object[] {slot},
                AttackerResponse.class);
    }

    /**
     * Before block broadcast.
     *
     * @param slot The slot number
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockBeforeBroadcast(long slot) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_beforeBroadCast",
                new Object[] {slot},
                AttackerResponse.class);
    }

    /**
     * After block broadcast.
     *
     * @param slot The slot number
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockAfterBroadcast(long slot) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_afterBroadCast",
                new Object[] {slot},
                AttackerResponse.class);
    }

    /**
     * Before sign block.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param blockDataBase64 The block data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockBeforeSign(long slot, String pubkey, String blockDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_beforeSign",
                AttackerResponse.class,
                new Object[] {slot,pubkey,blockDataBase64},
                AttackerResponse.class);
    }

    /**
     * After sign block.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param signedBlockDataBase64 The signed block data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockAfterSign(long slot, String pubkey, String signedBlockDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_afterSign",
                new Object[] {slot,pubkey,signedBlockDataBase64},
                AttackerResponse.class);
    }

    /**
     * Before propose block.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param signedBlockDataBase64 The signed block data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockBeforePropose(long slot, String pubkey, String signedBlockDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_beforePropose",
                new Object[] {slot,pubkey,signedBlockDataBase64},
                AttackerResponse.class);
    }

    /**
     * After propose block.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param signedBlockDataBase64 The signed block data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> blockAfterPropose(long slot, String pubkey, String signedBlockDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                BLOCK_MODULE + "_afterPropose",
                new Object[] {slot,pubkey,signedBlockDataBase64},
                AttackerResponse.class);
    }

    /**
     * Before attestation broadcast.
     *
     * @param slot The slot number
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> attestBeforeBroadcast(long slot) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                ATTEST_MODULE + "_beforeBroadCast",
                new Object[] {slot},
                AttackerResponse.class);
    }

    /**
     * After attestation broadcast.
     *
     * @param slot The slot number
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> attestAfterBroadcast(long slot) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                ATTEST_MODULE + "_afterBroadCast",
                new Object[] {slot},
                AttackerResponse.class);
    }

    /**
     * Before signing attestation.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param attestDataBase64 The attestation data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> attestBeforeSign(
            long slot,
            String pubkey,
            String attestDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                ATTEST_MODULE + "_beforeSign",
                new Object[] {slot,pubkey,attestDataBase64},
                AttackerResponse.class);
    }

    /**
     * After signing attestation.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param signedAttestDataBase64 The signed attestation data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> attestAfterSign(
            long slot,
            String pubkey,
            String signedAttestDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                ATTEST_MODULE + "_afterSign",
                new Object[] {slot,pubkey,signedAttestDataBase64},
                AttackerResponse.class);
    }

    /**
     * Before proposing attestation.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param signedAttestDataBase64 The signed attestation data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> attestBeforePropose(
            long slot,
            String pubkey,
            String signedAttestDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                ATTEST_MODULE + "_beforePropose",
                new Object[] {slot,pubkey,signedAttestDataBase64},
                AttackerResponse.class);
    }

    /**
     * After proposing attestation.
     *
     * @param slot The slot number
     * @param pubkey The validator public key
     * @param signedAttestDataBase64 The signed attestation data in base64 encoding
     * @return A future that completes with the response
     */
    public SafeFuture<AttackerResponse> attestAfterPropose(
            long slot,
            String pubkey,
            String signedAttestDataBase64) {
        return JsonRpcClientUtil.getJsonRpcClient().invoke(
                ATTEST_MODULE + "_afterPropose",
                new Object[] {slot,pubkey,signedAttestDataBase64},
                AttackerResponse.class);
    }


}
