package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/config"
	. "code.wuban.net.cn/movabridge/bridge-backend/constant"
	"code.wuban.net.cn/movabridge/bridge-backend/contract/bridge"
	dbModel "code.wuban.net.cn/movabridge/bridge-backend/model/db"
	"context"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	log "github.com/sirupsen/logrus"
	"math/big"
	"strings"
)

type ChainInterface interface {
	Name() string
	GetChain() *config.ChainConfig
	IsSyncing() bool
	ParseTransferOut(log *types.Log) (*bridge.BridgeContractTransferOut, error)
	ParseTransferIn(log *types.Log) (*bridge.BridgeContractTransferIn, error)
	ParseTransferInExecution(log *types.Log) (*bridge.BridgeContractTransferInExecution, error)
	ParseTransferInRejection(log *types.Log) (*bridge.BridgeContractTransferInRejection, error)
	ParseTransferInConfirmation(log *types.Log) (*bridge.BridgeContractTransferInConfirmation, error)
	ParseTokenConfigChanged(log *types.Log) (*bridge.BridgeContractTokenOutConfigChanged, error)
	ParseSwapConfigChanged(log *types.Log) (*bridge.BridgeContractSwapConfigChanged, error)
	GetReceiveToken(token common.Address, toChainId int64) (string, error)
	GetOutConfig(token common.Address, toChainId int64) (outConfig bridge.BridgeOutConfig, err error)
}

var (
	bridgeAbi, _                = bridge.BridgeContractMetaData.GetAbi()
	TransferOutEvent            = bridgeAbi.Events[EVENT_TRANSFER_OUT]
	TransferInEvent             = bridgeAbi.Events[EVENT_TRANSFER_IN]
	TransferInExecutionEvent    = bridgeAbi.Events[EVENT_TRANSFER_IN_EXECUTION]
	TransferInRejectionEvent    = bridgeAbi.Events[EVENT_TRANSFER_IN_REJECTION]
	TransferInConfirmationEvent = bridgeAbi.Events[EVENT_TRANSFER_IN_CONFIRMATION]
	TokenConfigChangedEvent     = bridgeAbi.Events[EVENT_TOKENCONFIGCHANGED]
	SwapConfigChangedEvent      = bridgeAbi.Events[EVENT_SWAPCONFIGCHANGED]
)

func (s *Dao) HandleEvents(chain ChainInterface, logs []types.Log) error {

	cname := chain.Name()
	// sort logs by topic.
	var allTransferOut = make([]types.Log, 0)
	var allTransferIn = make([]types.Log, 0)
	var allTokenConfigChanged = make([]types.Log, 0)
	var allSwapConfigChanged = make([]types.Log, 0)
	var allExecuted = make([]types.Log, 0)
	var allRejected = make([]types.Log, 0)
	var allConfirmed = make([]types.Log, 0)

	for _, lg := range logs {
		switch lg.Topics[0].String() {
		case TransferOutEvent.ID.Hex():
			allTransferOut = append(allTransferOut, lg)
		case TransferInEvent.ID.Hex():
			allTransferIn = append(allTransferIn, lg)
		case TokenConfigChangedEvent.ID.Hex():
			allTokenConfigChanged = append(allTokenConfigChanged, lg)
		case SwapConfigChangedEvent.ID.Hex():
			allSwapConfigChanged = append(allSwapConfigChanged, lg)
			log.WithField("tx", lg.TxHash).Info("got swap config changed event")
		case TransferInExecutionEvent.ID.Hex():
			allExecuted = append(allExecuted, lg)
		case TransferInRejectionEvent.ID.Hex():
			allRejected = append(allRejected, lg)
		case TransferInConfirmationEvent.ID.Hex():
			allConfirmed = append(allConfirmed, lg)
		}
	}
	var allLogs []types.Log
	allLogs = append(allLogs, allTransferOut...)
	allLogs = append(allLogs, allTransferIn...)
	allLogs = append(allLogs, allTokenConfigChanged...)
	allLogs = append(allLogs, allRejected...)
	allLogs = append(allLogs, allConfirmed...)
	allLogs = append(allLogs, allExecuted...)
	allLogs = append(allLogs, allSwapConfigChanged...)

	// disable the code.
	if false {

		var txs []common.Hash
		var txexist = make(map[common.Hash]bool)
		for _, lg := range logs {
			if _, ok := txexist[lg.TxHash]; !ok {
				txs = append(txs, lg.TxHash)
				txexist[lg.TxHash] = true
			}
		}
		var allLogs []*types.Log
		client := s.chainGroup[chain.GetChain().ChainId].cli
		for _, tx := range txs {
			receipt, err := client.TransactionReceipt(context.Background(), tx)
			if err != nil {
				log.WithField("chain", cname).WithError(err).Error("get tx receipt failed")
				return err
			}
			allLogs = append(allLogs, receipt.Logs...)
		}
	}

	s.handleMux.Lock()
	defer s.handleMux.Unlock()
	// begin orm transaction
	//supportSession := s.SupportsTransactions()
	var ctx context.Context = context.Background()
	//var tx *Transaction
	//var err error
	//if supportSession {
	//	tx, err = s.BeginTx(context.Background())
	//	if err != nil {
	//		supportSession = false
	//	}
	//}
	//if tx != nil {
	//	ctx = tx.ctx
	//} else {
	//	ctx = context.Background()
	//}

	var ormTxErr error
	for _, tlog := range allLogs {
		txLog := &tlog
		if err := s.filterTransferOut(chain, txLog, ctx); err != nil {
			ormTxErr = err
			break
		}
		if err := s.filterTransferIn(chain, txLog, ctx); err != nil {
			ormTxErr = err
			break
		}
		if err := s.filterTokenConfigChanged(chain, txLog, ctx); err != nil {
			ormTxErr = err
			break
		}
		if err := s.filterSwapConfigChanged(chain, txLog, ctx); err != nil {
			ormTxErr = err
			break
		}
	}

	// commit or rollback.
	if ormTxErr != nil {
		//if tx != nil {
		//	if rbErr := tx.Rollback(); rbErr != nil {
		//		log.WithField("chain", cname).WithError(rbErr).Error("failed to rollback transaction")
		//	} else {
		//		log.WithField("chain", cname).WithError(ormTxErr).Error("error processing logs, transaction rolled back")
		//	}
		//} else {
		log.WithFields(log.Fields{
			"chain": cname,
			"error": ormTxErr.Error(),
		}).Error("failed to process logs")
		//}
	} else {
		//if tx != nil {
		//	if cmtErr := tx.Commit(); cmtErr != nil {
		//		log.WithField("chain", cname).WithError(cmtErr).Error("failed to commit transaction")
		//	}
		//}
	}
	return nil
}

// filterTransferOut 用户从当前链跨出事件.
func (s *Dao) filterTransferOut(chain ChainInterface, txLog *types.Log, ctx context.Context) error {
	if len(txLog.Topics) == 0 {
		return nil
	}

	if txLog.Topics[0].Hex() == TransferOutEvent.ID.Hex() {
		event, err := chain.ParseTransferOut(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferOut log")
			return err
		}
		log.WithFields(log.Fields{
			"chain":      chain.Name(),
			"from_chain": event.FromChainID.Int64(),
			"out_id":     event.OutId.Int64(),
		}).Info("process transfer out event")
		blocktime, _ := s.GetBlockTime(chain.GetChain(), int64(txLog.BlockNumber))
		dbEvent := &dbModel.BridgeEvent{
			FromChain:       event.FromChainID.Int64(),
			OutId:           event.OutId.Int64(),
			OutTimestamp:    int64(blocktime),
			FromChainTxHash: strings.ToLower(txLog.TxHash.String()),
			FromAddress:     strings.ToLower(event.Sender.String()),
			FromToken:       strings.ToLower(event.Token.String()),
			SendAmount:      event.Amount.Text(10),
			FeeAmount:       event.Fee.Text(10),
			ToChain:         event.ToChainID.Int64(),
			Receiver:        strings.ToLower(event.Receiver.String()),
			ToToken:         strings.ToLower(event.ReceiveToken.String()),
			ReceiveAmount:   new(big.Int).Sub(event.Amount, event.Fee).Text(10),
			ToChainStatus:   int(TransferChainNoProcess),
		}
		if err := s.FillOutTransferEventInfo(ctx, dbEvent); err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("fill out transfer event info")
			return err
		}
		return nil
	}
	return nil
}

// filterTransferIn 用户从目标链跨入事件及执行结束事件.
func (s *Dao) filterTransferIn(chain ChainInterface, txLog *types.Log, ctx context.Context) error {
	if len(txLog.Topics) == 0 {
		return nil
	}

	switch txLog.Topics[0].Hex() {
	case TransferInEvent.ID.Hex():
		event, err := chain.ParseTransferIn(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferIn log")
			return err
		}
		log.WithFields(log.Fields{
			"chain":      chain.Name(),
			"from_chain": event.FromChainID.Int64(),
			"out_id":     event.OutId.Int64(),
			"to_chain":   chain.GetChain().ChainId,
			"in_id":      event.InId.Int64(),
		}).Info("process transfer in event")
		blocktime, _ := s.GetBlockTime(chain.GetChain(), int64(txLog.BlockNumber))

		dbEvent := &dbModel.BridgeEvent{
			FromChain:           event.FromChainID.Int64(),
			OutId:               event.OutId.Int64(),
			InId:                event.InId.Int64(),
			Receiver:            strings.ToLower(event.Receiver.String()),
			ToToken:             strings.ToLower(event.Token.String()),
			ReceiveAmount:       event.Amount.Text(10),
			ToChainStatus:       int(TransferChainWaitConfirm),
			ToContract:          strings.ToLower(txLog.Address.String()),
			InTimestamp:         int64(blocktime),
			ToChain:             chain.GetChain().ChainId,
			ToChainTxHash:       strings.ToLower(txLog.TxHash.String()),
			ConfirmedValidators: make([]string, 0),
			RejectedValidators:  make([]string, 0),
		}

		if err := s.FillInTransferEventInfo(ctx, dbEvent); err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"error": err.Error(),
			}).Error("db fill in transfer event")
			return err
		}
		return nil

	case TransferInExecutionEvent.ID.Hex():
		event, err := chain.ParseTransferInExecution(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInExecution log")
			return err
		}
		log.WithFields(log.Fields{
			"chain":    chain.Name(),
			"to_chain": chain.GetChain().ChainId,
			"in_id":    event.InId.Int64(),
		}).Info("process transfer in execution event")
		dbevent, err := s.GetBridgeEventByInId(ctx, chain.GetChain().ChainId, event.InId.Int64())
		if err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"in_id": event.InId.Int64(),
				"error": err.Error(),
			}).Error("bridge event by in id not found")
			return err
		}
		dbevent.ToChainStatus = int(TransferChainExecuted)
		dbevent.ToChainTxHash = strings.ToLower(txLog.TxHash.String())
		if err := s.UpdateBridgeEventWithId(ctx, dbevent); err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"in_id": event.InId.Int64(),
				"error": err.Error(),
			}).Error("db update transfer in execution event")
			return err
		}
		//
		//if err := s.UpdateBridgeResult(ctx, chain.GetChain().ChainId, event.InId.Int64(), TransferChainExecuted, txLog.TxHash); err != nil {
		//	log.WithField("chain", chain.Name()).WithFields(log.Fields{
		//		"error": err.Error(),
		//	}).Error("db update transfer in execution event")
		//	return err
		//}
	case TransferInConfirmationEvent.ID.Hex():
		event, err := chain.ParseTransferInConfirmation(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInConfirmation log")
			return err
		}
		log.WithFields(log.Fields{
			"chain":    chain.Name(),
			"to_chain": chain.GetChain().ChainId,
			"in_id":    event.InId.Int64(),
		}).Info("process transfer in confirmation event")
		dbevent, err := s.GetBridgeEventByInId(ctx, chain.GetChain().ChainId, event.InId.Int64())
		if err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"in_id": event.InId.Int64(),
				"error": err.Error(),
			}).Error("bridge event by in id not found")
			return err
		}
		dbevent.ConfirmedValidators = append(dbevent.ConfirmedValidators, strings.ToLower(event.Validator.String()))
		if err := s.UpdateBridgeEventWithId(ctx, dbevent); err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"in_id": event.InId.Int64(),
				"error": err.Error(),
			}).Error("db update transfer in confirmation event")
			return err
		}

		//if err := s.AddValidatorOp(ctx, chain.GetChain().ChainId, event.InId.Int64(), strings.ToLower(event.Validator.String()), true); err != nil {
		//	log.WithField("chain", chain.Name()).WithError(err).Error("add validator op failed")
		//	return err
		//}
	case TransferInRejectionEvent.ID.Hex():
		event, err := chain.ParseTransferInRejection(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInExecution log")
			return err
		}
		log.WithFields(log.Fields{
			"chain":    chain.Name(),
			"to_chain": chain.GetChain().ChainId,
			"in_id":    event.InId.Int64(),
		}).Info("process transfer in rejection event")
		dbevent, err := s.GetBridgeEventByInId(ctx, chain.GetChain().ChainId, event.InId.Int64())
		if err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"in_id": event.InId.Int64(),
				"error": err.Error(),
			}).Error("bridge event by in id not found")
			return err
		}
		dbevent.ConfirmedValidators = append(dbevent.ConfirmedValidators, strings.ToLower(event.Validator.String()))
		dbevent.ToChainStatus = int(TransferChainRejected)
		dbevent.ToChainTxHash = strings.ToLower(txLog.TxHash.String())
		if err := s.UpdateBridgeEventWithId(ctx, dbevent); err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"in_id": event.InId.Int64(),
				"error": err.Error(),
			}).Error("db update transfer in rejection event")
			return err
		}
		//if err := s.AddValidatorOp(ctx, chain.GetChain().ChainId, event.InId.Int64(), strings.ToLower(event.Validator.String()), false); err != nil {
		//	log.WithField("chain", chain.Name()).WithError(err).Error("add validator op failed")
		//	return err
		//}
		//if err := s.UpdateBridgeResult(ctx, chain.GetChain().ChainId, event.InId.Int64(), TransferChainRejected, txLog.TxHash); err != nil {
		//	log.WithField("chain", chain.Name()).WithFields(log.Fields{
		//		"error": err.Error(),
		//	}).Error("db update transfer in execution event")
		//	return err
		//}
	}
	return nil
}

func (s *Dao) filterTokenConfigChanged(chain ChainInterface, txLog *types.Log, ctx context.Context) error {
	if len(txLog.Topics) == 0 {
		return nil
	}

	if txLog.Topics[0].Hex() == TokenConfigChangedEvent.ID.Hex() {
		configure, err := chain.ParseTokenConfigChanged(txLog)
		if err != nil {
			return err
		}
		info := &dbModel.BridgeTokenInfo{
			ChainId:   chain.GetChain().ChainId,
			Token:     strings.ToLower(configure.Token.String()),
			ToChainId: configure.ToChainID.Int64(),
			Enabled:   configure.Enabled,
			Contract:  strings.ToLower(txLog.Address.String()),
		}

		// get receive token from contract.
		info.ToToken, err = chain.GetReceiveToken(configure.Token, info.ToChainId)
		if err != nil {
			log.WithFields(log.Fields{
				"chain":     chain.Name(),
				"token":     configure.Token.Hex(),
				"toChainId": info.ToChainId,
			}).WithError(err).Error("get receive token config failed")
			return err
		}

		err = s.CreateOrUpdateBridgeTokenInfo(ctx, info)
		if err != nil {
			log.WithFields(log.Fields{
				"chain":     chain.Name(),
				"token":     configure.Token.Hex(),
				"toChainId": info.ToChainId,
				"enabled":   info.Enabled,
			}).WithError(err).Error("db create or update token config failed")
			return err
		}
	}
	return nil
}

func (s *Dao) filterSwapConfigChanged(chain ChainInterface, txLog *types.Log, ctx context.Context) error {
	if len(txLog.Topics) == 0 {
		return nil
	}

	if txLog.Topics[0].Hex() == SwapConfigChangedEvent.ID.Hex() {
		configure, err := chain.ParseSwapConfigChanged(txLog)
		if err != nil {
			log.WithError(err).Error("parse SwapConfigChanged log")
			return err
		}
		info := &dbModel.SwapTokenInfo{
			ChainId:      chain.GetChain().ChainId,
			Contract:     strings.ToLower(txLog.Address.String()),
			Token:        strings.ToLower(configure.FromToken.String()),
			SwapContract: strings.ToLower(configure.Swap.String()),
			ToToken:      strings.ToLower(configure.ToToken.String()),
			SwapPath:     make([]string, 0),
		}
		for _, addr := range configure.Path {
			info.SwapPath = append(info.SwapPath, strings.ToLower(addr.String()))
		}

		if tokenInfo, err := s.tokenRepo.RetriveTokenInfo(chain.GetChain().ChainId, configure.FromToken.Hex()); err == nil {
			info.TokenName = tokenInfo.Symbol
		}

		if configure.Swap == (common.Address{}) && len(configure.Path) == 0 {
			// delete swap config
			err = s.DeleteSwapToken(ctx, info)
			if err != nil {
				log.WithFields(log.Fields{
					"chain":   chain.Name(),
					"token":   configure.FromToken.Hex(),
					"toToken": configure.ToToken.Hex(),
					"swap":    configure.Swap.Hex(),
				}).WithError(err).Error("db delete swap token config failed")
				return err
			} else {
				log.WithFields(log.Fields{
					"chain":   chain.Name(),
					"token":   configure.FromToken.Hex(),
					"toToken": configure.ToToken.Hex(),
					"swap":    configure.Swap.Hex(),
				}).Info("delete swap token config success")
				return nil
			}
		}

		err = s.CreateSwapTokenInfo(ctx, info)
		if err != nil {
			log.WithFields(log.Fields{
				"chain":    chain.Name(),
				"token":    configure.FromToken.Hex(),
				"toToken":  configure.ToToken.Hex(),
				"swap":     configure.Swap.Hex(),
				"swapPath": info.SwapPath,
			}).WithError(err).Error("db create or update token config failed")
			return err
		}
	}
	return nil
}
