package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/constant"
	dbModel "code.wuban.net.cn/movabridge/bridge-backend/model/db"
	"context"
	"github.com/ethereum/go-ethereum/common"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"strings"
	"time"
)

var (
	ErrRecordNotFound = mongo.ErrNoDocuments
)

// Transaction represents a MongoDB session with transaction
type Transaction struct {
	session mongo.Session
	ctx     context.Context
}

func (d *Dao) SupportsTransactions() bool {
	// 检查是否支持事务
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	err := d.db.Client().UseSession(ctx, func(sc mongo.SessionContext) error {
		return sc.StartTransaction()
	})

	return err == nil
}

// BeginTx starts a new MongoDB transaction
func (d *Dao) BeginTx(ctx context.Context) (*Transaction, error) {
	client := d.db.Client()
	session, err := client.StartSession()
	if err != nil {
		return nil, err
	}

	err = session.StartTransaction()
	if err != nil {
		session.EndSession(ctx)
		return nil, err
	}

	return &Transaction{
		session: session,
		ctx:     mongo.NewSessionContext(ctx, session),
	}, nil
}

// Commit commits the transaction
func (tx *Transaction) Commit() error {
	defer tx.session.EndSession(tx.ctx)
	return tx.session.CommitTransaction(tx.ctx)
}

// Rollback aborts the transaction
func (tx *Transaction) Rollback() error {
	defer tx.session.EndSession(tx.ctx)
	return tx.session.AbortTransaction(tx.ctx)
}

func (d *Dao) FillInTransferEventInfo(ctx context.Context, inEvent *dbModel.BridgeEvent) error {
	collection := d.db.Collection(inEvent.TableName())

	filter := bson.M{"from_chain": inEvent.FromChain, "out_id": inEvent.OutId}
	update := bson.M{
		"$set": bson.M{
			"from_chain":       inEvent.FromChain,
			"out_id":           inEvent.OutId,
			"to_chain":         inEvent.ToChain,
			"in_id":            inEvent.InId,
			"receiver":         inEvent.Receiver,
			"to_token":         inEvent.ToToken,
			"receive_amount":   inEvent.ReceiveAmount,
			"to_chain_tx_hash": inEvent.ToChainTxHash,
			"to_chain_status":  inEvent.ToChainStatus,
		},
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) FillOutTransferEventInfo(ctx context.Context, outEvent *dbModel.BridgeEvent) error {
	collection := d.db.Collection(outEvent.TableName())

	filter := bson.M{"from_chain": outEvent.FromChain, "out_id": outEvent.OutId}
	update := bson.M{
		"$set": bson.M{
			"from_chain":         outEvent.FromChain,
			"out_id":             outEvent.OutId,
			"out_timestamp":      outEvent.OutTimestamp,
			"from_chain_tx_hash": outEvent.FromChainTxHash,
			"from_address":       outEvent.FromAddress,
			"from_token":         outEvent.FromToken,
			"send_amount":        outEvent.SendAmount,
			"fee_amount":         outEvent.FeeAmount,
			"to_chain":           outEvent.ToChain,
			"receiver":           outEvent.Receiver,
			"to_token":           outEvent.ToToken,
			"receive_amount":     outEvent.ReceiveAmount,
		},
	}

	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (s *Dao) GetBridgeEventByOutId(ctx context.Context, fromChainId int64, outId int64) (event *dbModel.BridgeEvent, err error) {
	collection := s.db.Collection("bridge_events")
	filter := bson.M{"from_chain": fromChainId, "out_id": outId}

	err = collection.FindOne(ctx, filter).Decode(&event)
	if err == mongo.ErrNoDocuments {
		return event, ErrRecordNotFound
	}
	return event, err
}

func (s *Dao) GetBridgeEventByInId(ctx context.Context, toChainId int64, inId int64) (event *dbModel.BridgeEvent, err error) {
	collection := s.db.Collection("bridge_events")
	filter := bson.M{"to_chain": toChainId, "in_id": inId}
	err = collection.FindOne(ctx, filter).Decode(&event)
	if err == mongo.ErrNoDocuments {
		return event, ErrRecordNotFound
	}
	return event, err
}

func (s *Dao) UpdateBridgeEventWithId(ctx context.Context, event *dbModel.BridgeEvent) error {
	collection := s.db.Collection("bridge_events")

	filter := bson.M{"_id": event.ID}
	update := bson.M{
		"$set": event,
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (s *Dao) AddValidatorOp(ctx context.Context, toChainId int64, inId int64, validator string, isConfirmed bool) error {
	collection := s.db.Collection("bridge_events")

	var update bson.M
	if isConfirmed {
		update = bson.M{
			"$addToSet": bson.M{
				"confirmed_validators": validator,
			},
		}
	} else {
		update = bson.M{
			"$addToSet": bson.M{
				"rejected_validators": validator,
			},
		}
	}

	filter := bson.M{"to_chain": toChainId, "in_id": inId}
	opts := options.Update().SetUpsert(true)
	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) UpdateBridgeResult(ctx context.Context, toChainId int64, inId int64, result constant.TransferStatus, txhash common.Hash) error {
	collection := d.db.Collection(new(dbModel.BridgeEvent).TableName())

	filter := bson.M{"to_chain": toChainId, "in_id": inId}
	update := bson.M{
		"$set": bson.M{
			"to_chain_tx_hash": txhash.Hex(),
			"to_chain_status":  int(result),
		},
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) CreateOrUpdateBridgeTokenInfo(ctx context.Context, info *dbModel.BridgeTokenInfo) error {
	collection := d.db.Collection(info.TableName())

	filter := bson.M{"chain_id": info.ChainId, "token": info.Token, "to_chain_id": info.ToChainId}
	update := bson.D{
		{"$set", info},
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) CreateOrUpdateTokenInfo(ctx context.Context, info *dbModel.TokenInfo) error {
	collection := d.db.Collection(info.TableName())
	info.Address = strings.ToLower(info.Address)

	filter := bson.M{"address": info.Address}
	update := bson.D{
		{"$set", info},
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) GetTokenInfo(address string) (info dbModel.TokenInfo, err error) {
	collection := d.db.Collection(new(dbModel.TokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	err = collection.FindOne(ctx, bson.M{"address": strings.ToLower(address)}).Decode(&info)
	if err == mongo.ErrNoDocuments {
		return info, ErrRecordNotFound
	}
	return info, err
}

func (d *Dao) CreateSwapTokenInfo(ctx context.Context, info *dbModel.SwapTokenInfo) error {
	collection := d.db.Collection(info.TableName())

	filter := bson.M{"chain_id": info.ChainId, "token": info.Token, "contract": info.Contract}
	update := bson.D{
		{"$set", info},
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) DeleteSwapToken(ctx context.Context, info *dbModel.SwapTokenInfo) error {
	collection := d.db.Collection(info.TableName())

	filter := bson.M{"chain_id": info.ChainId, "token": info.Token, "to_token": info.ToToken, "contract": info.Contract}
	_, err := collection.DeleteOne(ctx, filter)
	return err
}
