package dao

import (
	"context"
	"fmt"
	"net/url"
	"os"
	"testing"
	"time"

	dbModel "code.wuban.net.cn/movabridge/bridge-backend/model/db"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

func TestDeleteSwapToken(t *testing.T) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	uri := os.Getenv("MONGO_TEST_URI")
	if uri == "" {
		user := "root"
		pass := "XN2UARuys3zy4Oux"
		authSource := "admin"
		// Build credentialed URI. url.QueryEscape to safely encode special chars.
		uri = fmt.Sprintf("mongodb://%s:%s@localhost:27017/bridge?authSource=%s&authMechanism=SCRAM-SHA-256",
			url.QueryEscape(user), url.QueryEscape(pass), url.QueryEscape(authSource))

	}

	client, err := mongo.Connect(ctx, options.Client().ApplyURI(uri))
	if err != nil {
		// If credentials are required but missing, surface a clearer message.
		if os.Getenv("MONGO_TEST_USER") != "" || os.Getenv("MONGO_TEST_PASS") != "" {
			// Provide hint instead of generic failure.
			t.Fatalf("mongo connect error with credentials (URI=%s): %v", uri, err)
		}
		// Generic failure.
		t.Fatalf("mongo connect error: %v", err)
	}
	defer func() { _ = client.Disconnect(ctx) }()

	dbName := "movabridge_test"
	db := client.Database(dbName)
	d := &Dao{db: db}

	// prepare test data
	info := &dbModel.SwapTokenInfo{
		ChainId:  1,
		Token:    "test-token",
		Contract: "0xdeadbeef",
		ToToken:  "to-token",
	}

	// ensure clean state
	coll := db.Collection(info.TableName())
	_, _ = coll.DeleteMany(ctx, bson.M{
		"chain_id": info.ChainId,
		"token":    info.Token,
		"contract": info.Contract,
	})

	// insert
	if err := d.CreateSwapTokenInfo(ctx, info); err != nil {
		// If insert fails due to auth issues, abort early.
		t.Fatalf("CreateSwapTokenInfo failed: %v", err)
	}

	// verify inserted
	var found dbModel.SwapTokenInfo
	filter := bson.M{"chain_id": info.ChainId, "token": info.Token, "contract": info.Contract}
	if err := coll.FindOne(ctx, filter).Decode(&found); err != nil {
		t.Fatalf("expected document to exist after insert: %v", err)
	}

	// delete via method under test
	if err := d.DeleteSwapToken(ctx, info); err != nil {
		t.Fatalf("DeleteSwapToken failed: %v", err)
	}

	// verify deleted
	err = coll.FindOne(ctx, filter).Decode(&found)
	if err != mongo.ErrNoDocuments {
		t.Fatalf("expected document to be deleted, got: %v", err)
	}

	// cleanup
	_ = coll.Drop(ctx)
}
