package router

import (
	"context"
	"errors"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"strings"
)

// Pre-parse the router ABI once at init to avoid repeated JSON parsing cost.
var (
	parsedRouterABI abi.ABI
	parseRouterErr  error
)

func init() {
	parsedRouterABI, parseRouterErr = abi.JSON(strings.NewReader(UniswapV2RouterABI))
}

// GetAmountsOut queries the router for the output amount given an input amount and swap path.
// Returns the final output amount (last element of amounts). It validates inputs and avoids panics
// on unexpected empty results.
func GetAmountsOut(
	client *ethclient.Client,
	user common.Address,
	router common.Address,
	amountIn *big.Int,
	path []string,
) (*big.Int, error) {
	if parseRouterErr != nil {
		return nil, fmt.Errorf("router ABI init: %w", parseRouterErr)
	}
	if client == nil {
		return nil, errors.New("nil eth client")
	}
	if amountIn == nil {
		return nil, errors.New("nil amountIn")
	}
	if len(path) < 2 {
		return nil, fmt.Errorf("path length must be >= 2, got %d", len(path))
	}
	addrPath := make([]common.Address, 0, len(path))
	for _, addrStr := range path {
		if !common.IsHexAddress(addrStr) || common.HexToAddress(addrStr) == (common.Address{}) {
			return nil, fmt.Errorf("invalid address in path: %s", addrStr)
		}
		addrPath = append(addrPath, common.HexToAddress(addrStr))
	}

	contract := bind.NewBoundContract(router, parsedRouterABI, client, client, client)

	callOpts := &bind.CallOpts{
		Context: context.Background(),
		From:    user, // optional; included for completeness (some routers ignore).
	}

	var amounts = make([]interface{}, len(path))
	if err := contract.Call(callOpts, &amounts, "getAmountsOut", amountIn, addrPath); err != nil {
		return nil, fmt.Errorf("call getAmountsOut (amountIn=%s pathLen=%d): %w", amountIn.String(), len(path), err)
	}

	amountOut := *abi.ConvertType(amounts[len(path)-1], new(*big.Int)).(**big.Int)

	return amountOut, nil
}
