package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/constant"
	apiModel "code.wuban.net.cn/movabridge/bridge-backend/model/api"
	"context"
	log "github.com/sirupsen/logrus"
	"sort"
	"time"

	dbModel "code.wuban.net.cn/movabridge/bridge-backend/model/db"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// GetStorageHeight 获取上次缓存的高度
func (d *Dao) GetStorageHeight(key string) (value int64, err error) {
	collection := d.db.Collection(new(dbModel.Height).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	var storage dbModel.Height
	err = collection.FindOne(ctx, bson.M{"key": key}).Decode(&storage)
	if err == mongo.ErrNoDocuments {
		return 0, ErrRecordNotFound
	}
	return storage.IntValue, err
}

// SetStorageHeight 设置上次缓存的高度
func (d *Dao) SetStorageHeight(key string, intValue int64) (err error) {
	collection := d.db.Collection(new(dbModel.Height).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	filter := bson.M{"key": key}
	update := bson.M{
		"$set": bson.D{
			{"key", key},
			{"int_value", intValue},
		},
	}

	opts := options.Update().SetUpsert(true)
	_, err = collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) GetBridgeConfig() (config apiModel.BridgeConfig, err error) {
	collection := d.db.Collection(new(dbModel.BridgeTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{})
	if err != nil {
		return config, err
	}
	defer cursor.Close(ctx)

	var tokens []dbModel.BridgeTokenInfo
	err = cursor.All(ctx, &tokens)
	if err != nil {
		return config, err
	}

	config.Chains = make(map[string]apiModel.ChainConfig)
	// Convert database model to API model
	for _, info := range tokens {
		if !info.Enabled {
			continue
		}
		if info.TokenName == "" {
			tokenInfo, err := d.tokenRepo.RetriveTokenInfo(info.ChainId, info.Token)
			if err != nil {
				log.WithFields(log.Fields{
					"chain_id": info.ChainId,
					"token":    info.Token,
					"error":    err,
				}).Error("not found token info with tokenrepo, skip symbol info")
				continue
			} else {
				info.TokenName = tokenInfo.Symbol
			}
		}
		chainInfo, exist := d.chainList.Get(info.ChainId)
		if !exist {
			log.WithFields(log.Fields{
				"chain_id":  info.ChainId,
				"token":     info.Token,
				"tokenName": info.TokenName,
			}).Error("not found chain info with chainlist, skip bridge config")
			continue
		}

		var chainConfig apiModel.ChainConfig

		if _chainConfig, exist := config.Chains[chainInfo.Name]; !exist {
			chainConfig = apiModel.ChainConfig{
				ChainId:        info.ChainId,
				Chain:          chainInfo.Name,
				RpcUrl:         chainInfo.Rpc,
				ExplorerUrl:    chainInfo.Explorer,
				BridgeContract: info.Contract,
				SupportTokens:  make(map[string]apiModel.ToToken),
			}
		} else {
			chainConfig = _chainConfig
		}
		toTokenInfo, err := d.tokenRepo.RetriveTokenInfo(info.ToChainId, info.ToToken)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": info.ToChainId,
				"token":    info.ToToken,
				"error":    err,
			}).Error("not found token info with tokenrepo, skip symbol info")
		}
		chainConfig.SupportTokens[info.TokenName] = apiModel.ToToken{
			TokenContract: info.Token,
			ToChainId:     info.ToChainId,
			ToToken:       info.ToToken,
			ToTokenSymbol: toTokenInfo.Symbol,
		}
		config.Chains[chainInfo.Name] = chainConfig
	}

	return config, nil
}

func (d *Dao) GetHistoryInfo(user string) (history apiModel.History, err error) {
	collection := d.db.Collection(new(dbModel.BridgeEvent).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{"from_address": user})
	if err != nil {
		return history, err
	}
	defer cursor.Close(ctx)

	var events []dbModel.BridgeEvent
	err = cursor.All(ctx, &events)
	if err != nil {
		return history, err
	}

	pendingHistory := make([]*apiModel.HistoryInfo, 0, 1000)
	completedHistory := make([]*apiModel.HistoryInfo, 0, 1000)

	for _, event := range events {
		if event.FromChainTxHash == "" || event.ToChainTxHash == "" {
			continue
		}
		fromChain, _ := d.chainList.Get(event.FromChain)
		toChain, _ := d.chainList.Get(event.ToChain)
		tokenInfo, _ := d.tokenRepo.RetriveTokenInfo(event.FromChain, event.FromToken)

		record := &apiModel.HistoryInfo{
			FromChain:   fromChain.Chain,
			ToChain:     toChain.Chain,
			TxHash:      event.FromChainTxHash,
			CreateTime:  event.OutTimestamp,
			Amount:      event.SendAmount,
			Token:       event.FromToken,
			TokenSymbol: tokenInfo.Symbol,
			UrlSource:   fromChain.Explorer + "/tx/" + event.FromChainTxHash,

			Status:       constant.TransferStatus(event.ToChainStatus).String(),
			ConfirmCount: len(event.ConfirmedValidators),
			RejectCount:  len(event.RejectedValidators),
		}
		if event.ToChainStatus <= int(constant.TransferChainWaitConfirm) {
			pendingHistory = append(pendingHistory, record)
		} else {
			record.UrlTarget = toChain.Explorer + "/tx/" + event.ToChainTxHash
			completedHistory = append(completedHistory, record)
		}
	}
	// sort pending by CreateTime desc
	sort.Slice(pendingHistory, func(i, j int) bool {
		return pendingHistory[i].CreateTime > pendingHistory[j].CreateTime
	})
	sort.Slice(completedHistory, func(i, j int) bool {
		return completedHistory[i].CreateTime > completedHistory[j].CreateTime
	})

	// sort completed by CreateTime desc
	history.Pending = pendingHistory
	history.Finish = completedHistory

	return history, nil

}
