package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/config"
	"context"
	"errors"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"time"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
)

func (d *Dao) GetBlockHeight(chain *config.ChainConfig, behindBlock ...int) (height int64, err error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	if _, ok := d.chainGroup[chain.ChainId]; !ok {
		return 0, errors.New("chain client not support")
	}
	chaininfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return 0, errors.New("chain client not support")
	}
	n, err := chaininfo.cli.BlockNumber(ctx)
	if len(behindBlock) > 0 {
		n -= uint64(behindBlock[0])
		if n < 0 {
			n = 0
		}
	}
	return int64(n), err
}

func (d *Dao) GetLatestBockHash(chain *config.ChainConfig) (hash string, err error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	chainInfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return "", errors.New("chain client not support")
	}
	block, err := chainInfo.cli.BlockByNumber(ctx, nil)
	if err != nil {
		return
	}
	return block.Hash().Hex(), nil
}

func (d *Dao) GetBlockTime(chain *config.ChainConfig, height int64) (timestamp int64, err error) {
	chainInfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return 0, errors.New("chain client not support")
	}
	for i := 0; i < 2; i++ {
		ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
		defer cancel()

		block, err := chainInfo.cli.BlockByNumber(ctx, big.NewInt(int64(height)))
		if err == nil {
			return int64(block.Time()), nil
		}
	}
	return
}

func (d *Dao) GetLogs(chain *config.ChainConfig, beginHeight, endHeight int64, topics, addresses []string) (logs []types.Log, err error) {
	chainInfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return nil, errors.New("chain client not support")
	}
	for i := 0; i < 2; i++ {
		// 重试2次
		logs, err = d.getLogs(chainInfo.cli, beginHeight, endHeight, topics, addresses)
		if err == nil {
			return logs, nil
		}
	}
	return
}

func (d *Dao) getLogs(client *ethclient.Client, beginHeight, endHeight int64, topics []string, addresses []string) (logs []types.Log, err error) {
	addrs := make([]common.Address, 0)
	for _, addr := range addresses {
		addrs = append(addrs, common.HexToAddress(addr))
	}

	q := ethereum.FilterQuery{
		FromBlock: big.NewInt(int64(beginHeight)),
		ToBlock:   big.NewInt(int64(endHeight)),
		Topics:    [][]common.Hash{{}},
		Addresses: addrs,
	}
	for _, topic := range topics {
		q.Topics[0] = append(q.Topics[0], common.HexToHash(topic))
	}

	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	return client.FilterLogs(ctx, q)
}

func (d *Dao) CheckEventValid() bool {
	// Implement the logic to check if the event is valid.
	// This is a placeholder implementation.
	return true
}
