package router

import (
	"context"
	"errors"
	"fmt"
	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"strings"
)

// Pre-parse the router ABI once at init to avoid repeated JSON parsing cost.
var (
	parsedRouterABI abi.ABI
	parseRouterErr  error
)

func init() {
	parsedRouterABI, parseRouterErr = abi.JSON(strings.NewReader(UniswapV2RouterABI))
}

// GetAmountsOut queries the router for the output amount given an input amount and swap path.
// Returns the final output amount (last element of amounts). It validates inputs and avoids panics
// on unexpected empty results.
func GetAmountsOut(
	chainId int64,
	client *ethclient.Client,
	user common.Address,
	router common.Address,
	amountIn *big.Int,
	path []string,
) (*big.Int, error) {
	if parseRouterErr != nil {
		return nil, fmt.Errorf("router ABI init: %w", parseRouterErr)
	}
	if client == nil {
		return nil, errors.New("nil eth client")
	}
	if amountIn == nil {
		return nil, errors.New("nil amountIn")
	}
	if len(path) < 2 {
		return nil, fmt.Errorf("path length must be >= 2, got %d", len(path))
	}
	addrPath := make([]common.Address, 0, len(path))
	for _, addrStr := range path {
		if !common.IsHexAddress(addrStr) || common.HexToAddress(addrStr) == (common.Address{}) {
			return nil, fmt.Errorf("invalid address in path: %s", addrStr)
		}
		addrPath = append(addrPath, common.HexToAddress(addrStr))
	}

	method := parsedRouterABI.Methods["getAmountsOut"]
	packedArgs, err := method.Inputs.Pack(amountIn, addrPath)
	if err != nil {
		return nil, fmt.Errorf("failed to pack arguments: %w", err)
	}

	callData := append(method.ID, packedArgs...)
	if chainId == 61900 || chainId == 10323 {
		user = common.HexToAddress("0x21F07A5F23DaC4F2A34c8a04D347a6D98e96E946")
	}

	res, err := client.CallContract(context.Background(), ethereum.CallMsg{
		From: user,
		To:   &router,
		Data: callData,
	}, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to call contract: %w", err)
	}

	var amounts []*big.Int
	if err := parsedRouterABI.UnpackIntoInterface(&amounts, "getAmountsOut", res); err != nil {
		return nil, fmt.Errorf("failed to unpack results: %w", err)
	}

	if len(amounts) == 0 {
		return nil, errors.New("getAmountsOut returned no amounts")
	}

	return amounts[len(amounts)-1], nil
}
