package main

import (
	"code.wuban.net.cn/movabridge/bridge-backend/chain"
	"code.wuban.net.cn/movabridge/bridge-backend/chainlist"
	"code.wuban.net.cn/movabridge/bridge-backend/config"
	"code.wuban.net.cn/movabridge/bridge-backend/dao"
	"code.wuban.net.cn/movabridge/bridge-backend/server"
	"code.wuban.net.cn/movabridge/bridge-backend/tokenrepo"
	"os"
	"os/signal"
	"syscall"

	log "github.com/sirupsen/logrus"
	"github.com/spf13/cobra"
)

var (
	confPath string
)

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})
}

var rootCmd = &cobra.Command{
	Use:   "backend",
	Short: "Token bridge backend service",
	Long:  "A backend service for the token bridge system that syncs with multiple blockchain networks",
	Run: func(cmd *cobra.Command, args []string) {
		conf, err := config.New(confPath)
		if err != nil {
			panic(err)
		}
		if conf.Debug {
			log.SetLevel(log.DebugLevel)
		}

		chainRepo := chainlist.New(conf.ChainListFile)
		tokenRepo := tokenrepo.NewTokenRepo(chainRepo)

		d, err := dao.New(conf, chainRepo, tokenRepo)
		if err != nil {
			panic(err)
		}
		go func() {
			if err := server.StartRestApi(d, conf); err != nil {
				log.WithError(err).Error("start rest api")
			}
		}()

		syncers := make([]*chain.ChainSync, 0)

		for _, chainConfig := range conf.Chains {
			syncer := chain.NewChainSync(chainConfig, d)
			syncer.Start()
			syncers = append(syncers, syncer)
			d.AddSyncer(chainConfig.ChainId, syncer)
		}

		// Set up signal handling
		sigCh := make(chan os.Signal, 1)
		signal.Notify(sigCh, syscall.SIGINT, syscall.SIGTERM)

		// Wait for termination signal
		sig := <-sigCh
		log.WithField("signal", sig.String()).Info("received termination signal, shutting down")

		// Stop all chain sync instances
		for _, syncer := range syncers {
			syncer.Stop()
		}

		log.Info("graceful shutdown completed")
		os.Exit(0)
	},
}

func init() {
	rootCmd.Flags().StringVarP(&confPath, "config", "c", "config.toml", "config file path")
}

func main() {
	if err := rootCmd.Execute(); err != nil {
		log.Fatal(err)
	}
}
