package server

import (
	"code.wuban.net.cn/movabridge/bridge-backend/constant"
	apiModel "code.wuban.net.cn/movabridge/bridge-backend/model/api"
	"github.com/ethereum/go-ethereum/common"
	log "github.com/sirupsen/logrus"
	"strconv"
	"strings"

	"github.com/gin-gonic/gin"
)

func getParam(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	config, err := _querier.GetBridgeConfig()
	if err != nil {
		log.Errorf("get bridge config error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(config))
}

func getHistory(c *gin.Context) {
	address := c.DefaultQuery("address", "")
	if !common.IsHexAddress(address) {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}

	history, err := _querier.GetHistoryInfo(strings.ToLower(common.HexToAddress(address).Hex()))
	if err != nil {
		log.Errorf("get history error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(history))
}

func getTxStatus(c *gin.Context) {
	txHashStr := c.DefaultQuery("txhash", "")
	txHash := common.HexToHash(txHashStr)

	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}

	status, err := _querier.GetTxStatus(strings.ToLower(txHash.String()))
	if err != nil {
		log.Errorf("get history error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(status))
}

func bridgeRouters(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}

	config, err := _querier.GetBridgeConfig()
	if err != nil {
		log.Errorf("get bridge config error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(config))
}

func bridgeBalance(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	var queryParam apiModel.TokenBalanceQuery
	// Try JSON first, then fallback to query binding for backward compatibility
	if err := c.ShouldBindJSON(&queryParam); err != nil {
		log.Errorf("bind param error json:%v", err)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	if !common.IsHexAddress(queryParam.User) {
		log.Errorf("invalid user address: %v", queryParam.User)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	balances, err := _querier.GetTokenBalance(queryParam.ChainId, queryParam.User, queryParam.TokeList)
	if err != nil {
		log.Errorf("get bridge balance error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(balances))
}

func bridgeQuote(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	var queryParam apiModel.QuoteBridgeParam
	if err := c.ShouldBindJSON(&queryParam); err != nil {
		log.Errorf("bind param error json:%v", err)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	if !common.IsHexAddress(queryParam.User) {
		log.Errorf("invalid user address: %v", queryParam.User)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	if !common.IsHexAddress(queryParam.Receiver) {
		log.Errorf("invalid receiver address: %v", queryParam.Receiver)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	quote, err := _querier.QuoteBridge(queryParam)
	if err != nil {
		log.Errorf("get bridge quote error: %v", err)
		c.JSON(500, withError(err.Error()))
		return
	}
	c.JSON(200, withSuccess(quote))
}

func getAllSwapRouters(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	configs, err := _querier.GetAllChainSwapConfig()
	if err != nil {
		log.Errorf("get swap config error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(configs))
}

func swapBalance(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	var queryParam apiModel.TokenBalanceQuery
	if err := c.ShouldBindJSON(&queryParam); err != nil {
		log.Errorf("bind param error json:%v", err)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	if !common.IsHexAddress(queryParam.User) {
		log.Errorf("invalid user address: %v", queryParam.User)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	balances, err := _querier.GetTokenBalance(queryParam.ChainId, queryParam.User, queryParam.TokeList)
	if err != nil {
		log.Errorf("get swap balance error: %v", err)
		c.JSON(500, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(balances))
}

func swapQuote(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	var queryParam apiModel.QuoteSwapParam
	if err := c.ShouldBindJSON(&queryParam); err != nil {
		log.Errorf("bind param error json:%v", err)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	if !common.IsHexAddress(queryParam.User) {
		log.Errorf("invalid user address: %v", queryParam.User)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	if !common.IsHexAddress(queryParam.Receiver) {
		log.Errorf("invalid receiver address: %v", queryParam.Receiver)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	quote, err := _querier.QuoteSwap(queryParam)
	if err != nil {
		log.Errorf("get swap quote error: %v", err)
		c.JSON(500, withError(err.Error()))
		return
	}
	c.JSON(200, withSuccess(quote))
}

func getSwapRoutersByChainId(c *gin.Context) {
	if _querier == nil {
		log.Error("querier is nil")
		c.JSON(500, withError(constant.InternalError))
		return
	}
	chainIdStr := c.Param("chainid")
	if chainIdStr == "" {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	chainId, err := strconv.ParseInt(chainIdStr, 10, 64)
	if err != nil {
		log.Errorf("convert chainid(%s) error: %v", chainIdStr, err)
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	config, err := _querier.GetSwapConfig(chainId)
	if err != nil {
		log.Errorf("get swap config for chain %d error: %v", chainId, err)
		c.JSON(500, withError(err.Error()))
		return
	}
	c.JSON(200, withSuccess(config))
}
