package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/constant"
	dbModel "code.wuban.net.cn/movabridge/bridge-backend/model/db"
	"context"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

var (
	ErrRecordNotFound = mongo.ErrNoDocuments
)

// Transaction represents a MongoDB session with transaction
type Transaction struct {
	session mongo.Session
	ctx     context.Context
}

// BeginTx starts a new MongoDB transaction
func (d *Dao) BeginTx(ctx context.Context) (*Transaction, error) {
	client := d.db.Client()
	session, err := client.StartSession()
	if err != nil {
		return nil, err
	}

	err = session.StartTransaction()
	if err != nil {
		session.EndSession(ctx)
		return nil, err
	}

	return &Transaction{
		session: session,
		ctx:     mongo.NewSessionContext(ctx, session),
	}, nil
}

// Commit commits the transaction
func (tx *Transaction) Commit() error {
	defer tx.session.EndSession(tx.ctx)
	return tx.session.CommitTransaction(tx.ctx)
}

// Rollback aborts the transaction
func (tx *Transaction) Rollback() error {
	defer tx.session.EndSession(tx.ctx)
	return tx.session.AbortTransaction(tx.ctx)
}

func (d *Dao) FillInTransferEventInfo(tx *Transaction, inEvent *dbModel.BridgeEvent) error {
	collection := d.db.Collection(inEvent.TableName())

	filter := bson.M{"from_chain": inEvent.FromChain, "out_id": inEvent.OutId}
	update := bson.M{
		"$set": bson.M{
			"from_chain":      inEvent.FromChain,
			"out_id":          inEvent.OutId,
			"in_id":           inEvent.InId,
			"receiver":        inEvent.Receiver,
			"to_token":        inEvent.ToToken,
			"receive_amount":  inEvent.ReceiveAmount,
			"to_chain_status": inEvent.ToChainStatus,
		},
	}

	_, err := collection.UpdateOne(tx.ctx, filter, update)
	return err
}

func (d *Dao) FillOutTransferEventInfo(tx *Transaction, outEvent *dbModel.BridgeEvent) error {
	collection := d.db.Collection(outEvent.TableName())

	filter := bson.M{"from_chain": outEvent.FromChain, "out_id": outEvent.OutId}
	update := bson.M{
		"$set": bson.M{
			"from_chain":         outEvent.FromChain,
			"out_id":             outEvent.OutId,
			"out_timestamp":      outEvent.OutTimestamp,
			"from_chain_tx_hash": outEvent.FromChainTxHash,
			"from_address":       outEvent.FromAddress,
			"from_token":         outEvent.FromToken,
			"send_amount":        outEvent.SendAmount,
			"fee_amount":         outEvent.FeeAmount,
			"to_chain":           outEvent.ToChain,
			"receiver":           outEvent.Receiver,
			"to_token":           outEvent.ToToken,
			"receive_amount":     outEvent.ReceiveAmount,
		},
	}

	_, err := collection.UpdateOne(tx.ctx, filter, update)
	return err
}

func (d *Dao) UpdateBridgeResult(tx *Transaction, toChainId int64, inId int64, result constant.TransferStatus) error {
	collection := d.db.Collection(new(dbModel.BridgeEvent).TableName())

	filter := bson.M{"to_chain": toChainId, "in_id": inId}
	update := bson.M{
		"$set": bson.M{
			"to_chain_status": int(result),
		},
	}

	_, err := collection.UpdateOne(tx.ctx, filter, update)
	return err
}

func (d *Dao) CreateOrUpdateBridgeTokenInfo(tx *Transaction, info *dbModel.BridgeTokenInfo) error {
	collection := d.db.Collection(info.TableName())

	filter := bson.M{"chain_id": info.ChainId, "token": info.Token, "to_chain_id": info.ToChainId}
	update := bson.D{
		{"$set", info},
	}
	opts := options.Update().SetUpsert(true)

	_, err := collection.UpdateOne(tx.ctx, filter, update, opts)
	return err
}
