package chainlist

import (
	"code.wuban.net.cn/movabridge/bridge-backend/types"
	"encoding/json"
	"errors"
	log "github.com/sirupsen/logrus"
	"os"
	"sync"
	"time"
)

// get latest chain list by download and parse https://chainlist.org/rpcs.json.
type ChainRepo struct {
	repo map[int]types.ChainInfo
	lock sync.RWMutex
	quit chan struct{}
}

type ChainData struct {
	Name  string `json:"name"`
	Chain string `json:"chain"`
	Rpc   []struct {
		Url string `json:"url"`
	} `json:"rpc"`
	NativeCurrency struct {
		Name     string `json:"name"`
		Symbol   string `json:"symbol"`
		Decimals int    `json:"decimals"`
	} `json:"nativeCurrency"`
	InfoURL   string `json:"infoURL"`
	ShortName string `json:"shortName"`
	ChainId   int    `json:"chainId"`
	NetworkId int    `json:"networkId"`
	Explorers []struct {
		Name     string `json:"name"`
		Url      string `json:"url"`
		Standard string `json:"standard"`
	} `json:"explorers"`
}

func New(local string) *ChainRepo {
	cr := &ChainRepo{
		repo: make(map[int]types.ChainInfo),
		quit: make(chan struct{}),
	}
	if newRepo, err := cr.loadChainFromFile(local); err != nil {
		log.WithError(err).Errorf("load chainlist from file %s failed", local)
	} else {
		cr.repo = newRepo
		log.Infof("load chainlist from file %s, total %d chains", local, len(cr.repo))
	}
	return cr
}

func (cr *ChainRepo) loadChainFromFile(fil string) (map[int]types.ChainInfo, error) {
	if fil == "" {
		return nil, errors.New("file name is empty")
	}
	data, err := os.ReadFile(fil)
	if err != nil {
		return nil, err
	}
	return cr.parseChainDataToMap(data)
}

func (cr *ChainRepo) parseChainDataToMap(data []byte) (map[int]types.ChainInfo, error) {
	chainList := make([]ChainData, 0)
	err := json.Unmarshal(data, &chainList)
	if err != nil {
		return nil, err
	}
	newMap := make(map[int]types.ChainInfo)
	for _, cd := range chainList {
		ci := types.ChainInfo{
			ChainId:        int64(cd.ChainId),
			Chain:          cd.Chain,
			Name:           cd.Name,
			Explorer:       "",
			Rpc:            "",
			NativeCurrency: cd.NativeCurrency,
		}
		if len(cd.Rpc) > 0 {
			ci.Rpc = cd.Rpc[0].Url
		}
		if len(cd.Explorers) > 0 {
			ci.Explorer = cd.Explorers[0].Url
		}
		newMap[cd.ChainId] = ci
	}
	return newMap, nil
}

func (cr *ChainRepo) Get(chainId int64) (types.ChainInfo, bool) {
	cr.lock.RLock()
	defer cr.lock.RUnlock()

	ci, ok := cr.repo[int(chainId)]
	return ci, ok
}

func (cr *ChainRepo) Start() {
	go cr.loop()
}

func (cr *ChainRepo) Stop() {
	close(cr.quit)
}

func (cr *ChainRepo) loop() {
	tc := time.NewTicker(time.Minute * 5)
	defer tc.Stop()
	for {
		select {
		case <-cr.quit:
			return
		case <-tc.C:
			var data = []byte{}
			// todo: fetch data from https://chainlist.org/rpcs.json
			if len(data) == 0 {
				continue
			}
			if newRepo, err := cr.parseChainDataToMap(data); err != nil {
				log.WithError(err).Error("parse chainlist data failed")
			} else {
				cr.lock.Lock()
				cr.repo = newRepo
				cr.lock.Unlock()
				log.Infof("update chainlist from remote, total %d chains", len(cr.repo))
			}
		}
	}
}
