package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/chainlist"
	"code.wuban.net.cn/movabridge/bridge-backend/config"
	"code.wuban.net.cn/movabridge/bridge-backend/contract/bridge"
	"code.wuban.net.cn/movabridge/bridge-backend/tokenrepo"
	"context"
	"crypto/ecdsa"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/ethclient"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"sync"
)

type ChainInfo struct {
	conf *config.ChainConfig
	cli  *ethclient.Client
}

type Dao struct {
	c           *config.Config
	db          *mongo.Database
	chainGroup  map[int64]ChainInfo
	syncer      map[int64]ChainInterface
	wg          sync.WaitGroup
	handleMux   sync.Mutex
	validatorPk *ecdsa.PrivateKey
	chainList   *chainlist.ChainRepo
	tokenRepo   *tokenrepo.TokenRepo
}

func New(_c *config.Config, clist *chainlist.ChainRepo, tokenRepo *tokenrepo.TokenRepo) (dao *Dao, err error) {
	dao = &Dao{
		c:          _c,
		chainGroup: make(map[int64]ChainInfo),
		syncer:     make(map[int64]ChainInterface),
		chainList:  clist,
		tokenRepo:  tokenRepo,
	}

	// Connect to all configured chains
	for name, chainConfig := range _c.Chains {
		var client *ethclient.Client
		client, err = ethclient.Dial(chainConfig.RPC)
		if err != nil {
			return nil, fmt.Errorf("failed to connect to %s chain: %w", name, err)
		}

		// Get and store chain ID
		chainId, err := client.ChainID(context.Background())
		if err != nil {
			return nil, fmt.Errorf("failed to get %s chain ID: %w", name, err)
		}

		// Update the chain ID in the config
		chainConfig.ChainId = chainId.Int64()
		dao.chainGroup[chainId.Int64()] = ChainInfo{
			conf: chainConfig,
			cli:  client,
		}
		fmt.Printf("Connected to %s chain with ID %d\n", name, chainConfig.ChainId)
	}

	// MongoDB connection
	mongoURI := fmt.Sprintf("mongodb://%s:%s@%s:%d/%s?authSource=admin&authMechanism=SCRAM-SHA-256",
		_c.Mongo.User, _c.Mongo.Password, _c.Mongo.Host, _c.Mongo.Port, _c.Mongo.Database)

	clientOptions := options.Client().ApplyURI(mongoURI)

	mongoClient, err := mongo.Connect(context.Background(), clientOptions)
	if err != nil {
		return nil, fmt.Errorf("failed to connect to MongoDB: %w", err)
	}

	// Ping to verify connection
	err = mongoClient.Ping(context.Background(), nil)
	if err != nil {
		return nil, fmt.Errorf("failed to ping MongoDB: %w", err)
	}

	dao.db = mongoClient.Database(_c.Mongo.Database)

	return dao, nil
}

func (d *Dao) ChainClient(chainId int64) *ethclient.Client {
	chainInfo := d.chainGroup[chainId]
	if chainInfo.cli != nil {
		return chainInfo.cli
	}
	return nil
}

func (d *Dao) Quit() {
	if d.db != nil {
		d.db.Client().Disconnect(context.Background())
	}

	d.chainGroup = nil
	d.db = nil
	d.c = nil
}

func (d *Dao) AddSyncer(chainId int64, syncer ChainInterface) {
	d.handleMux.Lock()
	defer d.handleMux.Unlock()
	d.syncer[chainId] = syncer
}

func (d *Dao) GetSyncer(chainId int64) (syncer ChainInterface, ok bool) {
	d.handleMux.Lock()
	defer d.handleMux.Unlock()
	syncer, ok = d.syncer[chainId]
	return
}

func (d *Dao) GetOutConfig(chainId int64, token common.Address, toChainId int64) (bridge.BridgeOutConfig, error) {
	syncer, ok := d.GetSyncer(chainId)
	if !ok {
		return bridge.BridgeOutConfig{}, fmt.Errorf("chain %d syncer not found", chainId)
	}
	return syncer.GetOutConfig(token, toChainId)
}

func (d *Dao) GetChainConfig(chainId int64) (ChainInfo, error) {
	chainInfo, ok := d.chainGroup[chainId]
	if !ok {
		return ChainInfo{}, fmt.Errorf("chain %d config not found", chainId)
	}
	return chainInfo, nil
}

func (d *Dao) GetRequireConfirmations(chainId int64) (int64, error) {
	syncer, ok := d.GetSyncer(chainId)
	if !ok {
		return 3, fmt.Errorf("chain %d syncer not found", chainId)
	}
	return syncer.GetRequiredConfirmations()
}
