package dao

import (
	"code.wuban.net.cn/movabridge/bridge-backend/constant"
	"code.wuban.net.cn/movabridge/bridge-backend/contract/bridge"
	"code.wuban.net.cn/movabridge/bridge-backend/contract/router"
	apiModel "code.wuban.net.cn/movabridge/bridge-backend/model/api"
	"context"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	log "github.com/sirupsen/logrus"
	"math/big"
	"sort"
	"strings"
	"time"

	dbModel "code.wuban.net.cn/movabridge/bridge-backend/model/db"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// GetStorageHeight 获取上次缓存的高度
func (d *Dao) GetStorageHeight(key string) (value int64, err error) {
	collection := d.db.Collection(new(dbModel.Height).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	var storage dbModel.Height
	err = collection.FindOne(ctx, bson.M{"key": key}).Decode(&storage)
	if err == mongo.ErrNoDocuments {
		return 0, ErrRecordNotFound
	}
	return storage.IntValue, err
}

// SetStorageHeight 设置上次缓存的高度
func (d *Dao) SetStorageHeight(key string, intValue int64) (err error) {
	collection := d.db.Collection(new(dbModel.Height).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	filter := bson.M{"key": key}
	update := bson.M{
		"$set": bson.D{
			{"key", key},
			{"int_value", intValue},
		},
	}

	opts := options.Update().SetUpsert(true)
	_, err = collection.UpdateOne(ctx, filter, update, opts)
	return err
}

func (d *Dao) GetBridgeConfig() (config apiModel.BridgeConfig, err error) {
	collection := d.db.Collection(new(dbModel.BridgeTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{})
	if err != nil {
		return config, err
	}
	defer cursor.Close(ctx)

	var tokens []dbModel.BridgeTokenInfo
	err = cursor.All(ctx, &tokens)
	if err != nil {
		return config, err
	}

	config.Chains = make(map[string]apiModel.ChainConfig)
	// Convert database model to API model
	for _, info := range tokens {
		if !info.Enabled {
			continue
		}
		if info.TokenName == "" {
			tokenInfo, err := d.tokenRepo.RetriveTokenInfo(info.ChainId, info.Token)
			if err != nil {
				log.WithFields(log.Fields{
					"chain_id": info.ChainId,
					"token":    info.Token,
					"error":    err,
				}).Error("not found token info with tokenrepo, skip symbol info")
				continue
			} else {
				info.TokenName = tokenInfo.Symbol
			}
		}
		chainInfo, exist := d.chainList.Get(info.ChainId)
		if !exist {
			log.WithFields(log.Fields{
				"chain_id":  info.ChainId,
				"token":     info.Token,
				"tokenName": info.TokenName,
			}).Error("not found chain info with chainlist, skip bridge config")
			continue
		}

		var chainConfig apiModel.ChainConfig

		if _chainConfig, exist := config.Chains[chainInfo.Name]; !exist {
			chainConfig = apiModel.ChainConfig{
				ChainId:        info.ChainId,
				Chain:          chainInfo.Name,
				RpcUrl:         chainInfo.Rpc,
				ExplorerUrl:    chainInfo.Explorer,
				BridgeContract: info.Contract,
				SupportTokens:  make(map[string]apiModel.SupportBridgeTokenInfo),
			}
		} else {
			chainConfig = _chainConfig
		}
		toTokenInfo, err := d.tokenRepo.RetriveTokenInfo(info.ToChainId, info.ToToken)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": info.ToChainId,
				"token":    info.ToToken,
				"error":    err,
			}).Error("not found token info with tokenrepo, skip symbol info")
		}

		tokenInfo := apiModel.ToBridgeToken{
			TokenContract: info.Token,
			ToChainId:     info.ToChainId,
			ToToken:       info.ToToken,
			ToTokenSymbol: toTokenInfo.Symbol,
		}
		outConfig, err := d.GetOutConfig(info.ChainId, common.HexToAddress(info.Token), info.ToChainId)
		if err == nil {
			tokenInfo.Fee = outConfig.Fee.String()
			tokenInfo.MaxLimit = outConfig.Limit.String()
		}
		if m, exist := chainConfig.SupportTokens[info.TokenName]; exist {
			m.BridgeTokens = append(m.BridgeTokens, tokenInfo)
			chainConfig.SupportTokens[info.TokenName] = m
		} else {
			chainConfig.SupportTokens[info.TokenName] = apiModel.SupportBridgeTokenInfo{
				TokenContract: info.Token,
				TokenSymbol:   info.TokenName,
				BridgeTokens:  []apiModel.ToBridgeToken{tokenInfo},
			}
		}
		config.Chains[chainInfo.Name] = chainConfig
	}

	return config, nil
}

func (d *Dao) GetHistoryInfo(user string) (history apiModel.History, err error) {
	collection := d.db.Collection(new(dbModel.BridgeEvent).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{"from_address": user})
	if err != nil {
		return history, err
	}
	defer cursor.Close(ctx)

	var events []dbModel.BridgeEvent
	err = cursor.All(ctx, &events)
	if err != nil {
		return history, err
	}

	pendingHistory := make([]*apiModel.HistoryInfo, 0, 1000)
	completedHistory := make([]*apiModel.HistoryInfo, 0, 1000)

	for _, event := range events {
		if event.FromChainTxHash == "" {
			continue
		}
		fromChain, _ := d.chainList.Get(event.FromChain)
		toChain, _ := d.chainList.Get(event.ToChain)
		tokenInfo, _ := d.tokenRepo.RetriveTokenInfo(event.FromChain, event.FromToken)

		record := &apiModel.HistoryInfo{
			FromChain:    fromChain.Chain,
			ToChain:      toChain.Chain,
			TxHash:       event.FromChainTxHash,
			CreateTime:   event.OutTimestamp,
			Amount:       event.SendAmount,
			Token:        event.FromToken,
			TokenSymbol:  tokenInfo.Symbol,
			UrlSource:    fromChain.Explorer + "/tx/" + event.FromChainTxHash,
			Status:       constant.TransferStatus(event.ToChainStatus).String(),
			FeeAmount:    event.FeeAmount,
			ConfirmCount: len(event.ConfirmedValidators),
			RejectCount:  len(event.RejectedValidators),
		}
		if event.ToChainStatus <= int(constant.TransferChainWaitConfirm) {
			pendingHistory = append(pendingHistory, record)
		} else {
			if event.ToChainTxHash != "" {
				record.UrlTarget = toChain.Explorer + "/tx/" + event.ToChainTxHash
				completedHistory = append(completedHistory, record)
			}
		}
	}
	// sort pending by CreateTime desc
	sort.Slice(pendingHistory, func(i, j int) bool {
		return pendingHistory[i].CreateTime > pendingHistory[j].CreateTime
	})
	sort.Slice(completedHistory, func(i, j int) bool {
		return completedHistory[i].CreateTime > completedHistory[j].CreateTime
	})

	// sort completed by CreateTime desc
	history.Pending = pendingHistory
	history.Finish = completedHistory

	return history, nil
}

func (d *Dao) GetTxStatus(tx string) (result apiModel.TxStatusResult, err error) {
	collection := d.db.Collection(new(dbModel.BridgeEvent).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	result = apiModel.TxStatusResult{
		Status: constant.TransferChainNoProcess.FriendlyString(0, 0),
	}

	cursor, err := collection.Find(ctx, bson.M{"from_chain_tx_hash": tx})
	if err != nil {
		return result, nil
	}
	defer cursor.Close(ctx)

	var events []dbModel.BridgeEvent
	err = cursor.All(ctx, &events)
	if err != nil {
		return result, nil
	}

	if len(events) == 0 {
		return result, nil
	}
	event := events[0]
	required, _ := d.GetRequireConfirmations(event.FromChain)
	result.Status = constant.TransferStatus(event.ToChainStatus).FriendlyString(len(event.ConfirmedValidators), int(required))

	return result, nil
}

type tokenBalance struct {
	Name     string
	Decimals int
	Balance  *big.Int
	Contract string
}

func (d *Dao) GetBridgeTokenBalance(chainId int64, user string) (balances apiModel.TokenBalances, err error) {
	balances = apiModel.TokenBalances{
		Balances: make([]apiModel.TokenBalance, 0),
	}
	collection := d.db.Collection(new(dbModel.BridgeTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{})
	if err != nil {
		return balances, err
	}
	defer cursor.Close(ctx)

	var tokens []dbModel.BridgeTokenInfo
	err = cursor.All(ctx, &tokens)
	if err != nil {
		return balances, err
	}

	chainClient := d.ChainClient(chainId)
	if chainClient == nil {
		log.Error("not found chain client for chain id:", chainId)
		return balances, nil
	}

	var innerBalance = make([]tokenBalance, 0)
	for _, token := range tokens {
		if token.ChainId != chainId {
			continue
		}
		tokenInfo, balance, err := d.tokenRepo.RetriveTokenInfoAndBalance(chainClient, chainId, token.Token, user)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": token.ChainId,
				"token":    token.Token,
				"error":    err,
			}).Error("get token balance failed, skip this token")
			continue
		}
		innerBalance = append(innerBalance, tokenBalance{
			Name:     tokenInfo.Symbol,
			Decimals: int(tokenInfo.Decimals),
			Balance:  balance,
			Contract: token.Token,
		})
	}
	// sort balances by Balance desc
	sort.Slice(innerBalance, func(i, j int) bool {
		return innerBalance[i].Balance.Cmp(innerBalance[j].Balance) > 0
	})

	for _, b := range innerBalance {
		balances.Balances = append(balances.Balances, apiModel.TokenBalance{
			Name:     b.Name,
			Decimals: b.Decimals,
			Balance:  b.Balance.String(),
			Contract: b.Contract,
		})
	}

	return balances, nil
}

func (d *Dao) GetTokenBalance(chainId int64, user string, tokens []string) (balances apiModel.TokenBalances, err error) {
	balances = apiModel.TokenBalances{
		Balances: make([]apiModel.TokenBalance, 0),
	}

	chainClient := d.ChainClient(chainId)
	if chainClient == nil {
		log.Error("not found chain client for chain id:", chainId)
		return balances, nil
	}

	var innerBalance = make([]tokenBalance, 0)
	for _, token := range tokens {
		tokenInfo, balance, err := d.tokenRepo.RetriveTokenInfoAndBalance(chainClient, chainId, token, user)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": chainId,
				"token":    token,
				"error":    err,
			}).Error("get token balance failed, skip this token")
			continue
		}
		innerBalance = append(innerBalance, tokenBalance{
			Name:     tokenInfo.Symbol,
			Decimals: int(tokenInfo.Decimals),
			Balance:  balance,
			Contract: token,
		})
	}
	// sort balances by Balance desc
	sort.Slice(innerBalance, func(i, j int) bool {
		return innerBalance[i].Balance.Cmp(innerBalance[j].Balance) > 0
	})

	for _, b := range innerBalance {
		balances.Balances = append(balances.Balances, apiModel.TokenBalance{
			Name:     b.Name,
			Decimals: b.Decimals,
			Balance:  b.Balance.String(),
			Contract: b.Contract,
		})
	}

	return balances, nil
}

func (d *Dao) buildBridgePayload(token common.Address, amount *big.Int, toChainID *big.Int, receiver common.Address) ([]byte, error) {
	contractAbi, _ := bridge.BridgeContractMetaData.GetAbi()
	return contractAbi.Pack("outTransfer", token, amount, toChainID, receiver)
}

func (d *Dao) QuoteBridge(param apiModel.QuoteBridgeParam) (quote apiModel.QuoteResult, err error) {
	chainInfo, err := d.GetChainConfig(param.FromChainId)
	if err != nil {
		log.Error("not found chain config for chain id:", param.FromChainId)
		return quote, fmt.Errorf("not found chain config for chain id: %s", param.FromChainId)
	}
	outConfig, err := d.GetOutConfig(param.FromChainId, common.HexToAddress(param.FromToken), param.ToChainId)
	if err != nil {
		log.WithFields(log.Fields{
			"chain_id": param.FromChainId,
			"token":    param.FromToken,
			"to_chain": param.ToChainId,
			"error":    err,
		}).Error("get out config failed")
		return quote, fmt.Errorf("internal error")
	}
	_, balance, err := d.tokenRepo.RetriveTokenInfoAndBalance(chainInfo.cli, chainInfo.conf.ChainId, param.FromToken, param.User)
	if err != nil {
		log.WithFields(log.Fields{
			"chain_id": param.FromChainId,
			"token":    param.FromToken,
			"user":     param.User,
			"error":    err,
		}).Error("get user balance failed")
		return quote, fmt.Errorf("internal error")
	}
	// check balance and param.InputAmount
	// check param.InputAmount and outConfig.Limit
	// calculate out amount = input amount - fee
	inputAmount, ok := new(big.Int).SetString(param.InputAmount, 10)
	if !ok {
		log.WithFields(log.Fields{
			"input_amount": param.InputAmount,
		}).Error("invalid input amount")
		return quote, fmt.Errorf("invalid input amount")
	}
	if balance.Cmp(inputAmount) < 0 {
		log.WithFields(log.Fields{
			"chain_id":    param.FromChainId,
			"token":       param.FromToken,
			"user":        param.User,
			"balance":     balance.String(),
			"inputAmount": inputAmount.String(),
		}).Error("user balance insufficient")
		return quote, fmt.Errorf("user balance insufficient")
	}
	if inputAmount.Cmp(outConfig.Limit) > 0 {
		log.WithFields(log.Fields{
			"chain_id":    param.FromChainId,
			"token":       param.FromToken,
			"user":        param.User,
			"inputAmount": inputAmount.String(),
			"limit":       outConfig.Limit.String(),
		}).Error("input amount exceed limit")
		return quote, fmt.Errorf("input amount exceed limit")
	}
	fee := outConfig.Fee
	if inputAmount.Cmp(fee) <= 0 {
		log.WithFields(log.Fields{
			"chain_id":    param.FromChainId,
			"token":       param.FromToken,
			"user":        param.User,
			"inputAmount": inputAmount.String(),
			"fee":         fee.String(),
		}).Error("input amount less than fee")
		return quote, fmt.Errorf("input amount less than fee")
	}
	outAmount := new(big.Int).Sub(inputAmount, fee)
	quote = apiModel.QuoteResult{
		ToContract: chainInfo.conf.BridgeContract,
		OutAmount:  outAmount.String(),
	}
	// build payload
	payload, err := d.buildBridgePayload(common.HexToAddress(param.ToToken), inputAmount, big.NewInt(param.ToChainId), common.HexToAddress(param.Receiver))
	if err != nil {
		log.WithFields(log.Fields{
			"chain_id":    param.FromChainId,
			"token":       param.FromToken,
			"user":        param.User,
			"inputAmount": inputAmount.String(),
			"error":       err,
		}).Error("build bridge payload failed")
		return quote, fmt.Errorf("internal error")
	}
	quote.Payload = common.Bytes2Hex(payload)
	return quote, nil
}

func (d *Dao) GetAllChainSwapConfig() (map[int64]*apiModel.ChainSwapConfig, error) {
	var res = make(map[int64]*apiModel.ChainSwapConfig)
	collection := d.db.Collection(new(dbModel.BridgeTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{})
	if err != nil {
		return res, fmt.Errorf("internal error")
	}
	defer cursor.Close(ctx)

	var tokens []dbModel.BridgeTokenInfo
	err = cursor.All(ctx, &tokens)
	if err != nil {
		return res, err
	}
	for _, token := range tokens {
		if _, exist := res[token.ChainId]; !exist {
			chainConfig, err := d.GetSwapConfig(token.ChainId)
			if err != nil {
				continue
			} else {
				res[token.ChainId] = chainConfig
			}
		}
	}
	return res, nil

}

func (d *Dao) GetSwapConfig(chainId int64) (*apiModel.ChainSwapConfig, error) {
	chainInfo, err := d.GetChainConfig(chainId)
	if err != nil {
		log.Error("not found chain config for chain id:", chainId)
		return nil, fmt.Errorf("not found chain config for chain id: %s", chainId)
	}
	chainRepo, exist := d.chainList.Get(chainId)
	if !exist {
		log.Error("not found chain info in chainlist for chain id:", chainId)
		return nil, fmt.Errorf("not found chain info in chainlist for chain id: %s", chainId)
	}

	config := &apiModel.ChainSwapConfig{
		Chain:          chainRepo.Name,
		ChainId:        chainId,
		RpcUrl:         chainRepo.Rpc,
		ExplorerUrl:    chainRepo.Explorer,
		BridgeContract: chainInfo.conf.BridgeContract,
		SupportTokens:  make(map[string]apiModel.SupportSwapTokenInfo),
	}

	bridgeTokens, err := d.getSupportedBridgeChains(chainId, "")
	if err != nil {
		return config, err
	}
	for _, bridgeToken := range bridgeTokens {
		swapChains := []dbModel.BridgeTokenInfo{bridgeToken}
		toChains := d.getTargetChainSwapTokens(swapChains, apiModel.SwapPath{
			SwapFromToken:   bridgeToken.Token,
			BridgeFromToken: bridgeToken.Token,
		})
		tokenInfo, err := d.tokenRepo.RetriveTokenInfo(chainId, bridgeToken.Token)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": chainId,
				"token":    bridgeToken.Token,
			}).Warning("get token info failed")
		}
		if len(toChains) > 0 {
			config.SupportTokens[bridgeToken.Token] = apiModel.SupportSwapTokenInfo{
				TokenContract: bridgeToken.Token,
				TokenSymbol:   tokenInfo.Symbol,
				SwapTokens:    toChains,
			}
		}
	}

	swapTokens, err := d.getSupportedSwapTokens(chainId)
	if err != nil {
		return config, err
	}
	for _, swap := range swapTokens {
		// if swap.ToToken is support, then it can swap to all bridged tokens and target chain token swaps.
		swapChains, err := d.getSupportedBridgeChains(chainId, swap.ToToken)
		if err != nil {
			return config, err
		}

		tokenInfo, err := d.tokenRepo.RetriveTokenInfo(chainId, swap.Token)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": chainId,
				"token":    swap.Token,
			}).Warning("get token info failed")
		}

		toChains := d.getTargetChainSwapTokens(swapChains, apiModel.SwapPath{
			SwapFromToken:   swap.Token,
			BridgeFromToken: swap.ToToken,
		})
		if len(toChains) > 0 {
			config.SupportTokens[swap.Token] = apiModel.SupportSwapTokenInfo{
				TokenContract: swap.Token,
				TokenSymbol:   tokenInfo.Symbol,
				SwapTokens:    toChains,
			}
		}
	}

	return config, nil
}

func (d *Dao) GetSwapTokenBalance(chainId int64, user string) (balances apiModel.TokenBalances, err error) {
	balances = apiModel.TokenBalances{
		Balances: make([]apiModel.TokenBalance, 0),
	}
	chainSwapConfig, err := d.GetSwapConfig(chainId)
	if err != nil {
		return balances, err
	}
	chainInfo, err := d.GetChainConfig(chainId)
	if err != nil {
		log.Error("not found chain config for chain id:", chainId)
		return balances, nil
	}

	innerBalance := make([]tokenBalance, 0)

	for _, token := range chainSwapConfig.SupportTokens {
		tokenInfo, balance, err := d.tokenRepo.RetriveTokenInfoAndBalance(chainInfo.cli, chainId, token.TokenContract, user)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": chainId,
				"token":    token.TokenContract,
				"error":    err,
			}).Error("get token balance failed, skip this token")
			continue
		}
		innerBalance = append(innerBalance, tokenBalance{
			Name:     tokenInfo.Symbol,
			Decimals: int(tokenInfo.Decimals),
			Balance:  balance,
			Contract: token.TokenContract,
		})
	}
	// sort balances by Balance desc
	sort.Slice(innerBalance, func(i, j int) bool {
		return innerBalance[i].Balance.Cmp(innerBalance[j].Balance) > 0
	})

	for _, b := range innerBalance {
		balances.Balances = append(balances.Balances, apiModel.TokenBalance{
			Name:     b.Name,
			Decimals: b.Decimals,
			Balance:  b.Balance.String(),
			Contract: b.Contract,
		})
	}

	return balances, nil
}

func (d *Dao) QuoteSwap(param apiModel.QuoteSwapParam) (quote apiModel.QuoteResult, err error) {
	fromChainInfo, err := d.GetChainConfig(param.FromChainId)
	if err != nil {
		log.Error("not found chain config for chain id:", param.FromChainId)
		return quote, fmt.Errorf("not found chain for chain id")
	}
	toChainInfo, err := d.GetChainConfig(param.ToChainId)
	if err != nil {
		log.Error("not found chain config for chain id:", param.ToChainId)
		return quote, fmt.Errorf("not found chain for chain id")
	}

	_, balance, err := d.tokenRepo.RetriveTokenInfoAndBalance(fromChainInfo.cli, fromChainInfo.conf.ChainId, param.Path.SwapFromToken, param.User)
	if err != nil {
		log.WithFields(log.Fields{
			"chain_id": param.FromChainId,
			"token":    param.Path.SwapFromToken,
			"user":     param.User,
			"error":    err,
		}).Error("get user balance failed")
		return quote, fmt.Errorf("internal error")
	}
	// check balance and param.InputAmount
	inputAmount, ok := new(big.Int).SetString(param.InputAmount, 10)
	if !ok {
		log.WithFields(log.Fields{
			"input_amount": param.InputAmount,
		}).Error("invalid input amount")
		return quote, fmt.Errorf("invalid input amount")
	}
	if balance.Cmp(inputAmount) < 0 {
		log.WithFields(log.Fields{
			"chain_id":    param.FromChainId,
			"token":       param.Path.SwapFromToken,
			"user":        param.User,
			"balance":     balance.String(),
			"inputAmount": inputAmount.String(),
		}).Error("user balance insufficient")
		return quote, fmt.Errorf("user balance insufficient")
	}

	quote = apiModel.QuoteResult{
		ToContract: fromChainInfo.conf.BridgeContract,
		OutAmount:  inputAmount.String(),
	}

	// if need swap, then calculate out amount after swap.
	if param.Path.SwapFromToken != param.Path.BridgeFromToken {
		// swap from token -> bridge from token
		swapConfig, err := d.findSwapConfig(param.FromChainId, param.Path.SwapFromToken, param.Path.BridgeFromToken)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id":          param.FromChainId,
				"swap_from_token":   param.Path.SwapFromToken,
				"bridge_from_token": param.Path.BridgeFromToken,
				"error":             err,
			}).Error("not found swap config for swap from token to bridge from token")
			return quote, err
		}
		swapFromAmount, err := router.GetAmountsOut(param.FromChainId, fromChainInfo.cli, common.HexToAddress(param.User), common.HexToAddress(swapConfig.SwapContract), inputAmount, swapConfig.SwapPath)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id":          param.FromChainId,
				"swap_from_token":   param.Path.SwapFromToken,
				"bridge_from_token": param.Path.BridgeFromToken,
				"input_amount":      inputAmount.String(),
				"swap_contract":     swapConfig.SwapContract,
				"error":             err,
			}).Error("get swap amounts out failed")
			return quote, fmt.Errorf("internal error")
		}
		inputAmount = swapFromAmount
	}
	// then sub bridge fee.
	outConfig, err := d.GetOutConfig(param.FromChainId, common.HexToAddress(param.Path.BridgeFromToken), param.ToChainId)
	if err != nil {
		log.WithFields(log.Fields{
			"chain_id":          param.FromChainId,
			"bridge_from_token": param.Path.BridgeFromToken,
			"to_chain":          param.ToChainId,
			"error":             err,
		}).Error("get out config failed")
		return quote, fmt.Errorf("internal error")
	}
	if outConfig.ReceiveToken != common.HexToAddress(param.Path.BridgeToToken) {
		log.WithFields(log.Fields{
			"chain_id":          param.FromChainId,
			"bridge_from_token": param.Path.BridgeFromToken,
			"to_chain":          param.ToChainId,
			"bridge_to_token":   param.Path.BridgeToToken,
			"receive_token":     outConfig.ReceiveToken.Hex(),
		}).Error("bridge to token not match receive token")
		return quote, fmt.Errorf("bridge to token not match receive token")
	}
	fee := outConfig.Fee
	if inputAmount.Cmp(fee) <= 0 {
		log.WithFields(log.Fields{
			"chain_id":          param.FromChainId,
			"bridge_from_token": param.Path.BridgeFromToken,
			"to_chain":          param.ToChainId,
			"inputAmount":       inputAmount.String(),
			"fee":               fee.String(),
		}).Error("input amount less than fee")
		return quote, fmt.Errorf("input amount less than fee")
	}
	outAmount := new(big.Int).Sub(inputAmount, fee)
	quote.OutAmount = outAmount.String()
	// if target need swap, then calculate out amount after target swap.
	if param.Path.BridgeToToken != param.Path.SwapToToken {
		// swap from bridge to token -> swap to token
		swapConfig, err := d.findSwapConfig(param.ToChainId, param.Path.BridgeToToken, param.Path.SwapToToken)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id":        param.ToChainId,
				"bridge_to_token": param.Path.BridgeToToken,
				"swap_to_token":   param.Path.SwapToToken,
				"error":           err,
			}).Error("not found swap config for bridge to token to swap to token")
			return quote, fmt.Errorf("internal error")
		}
		swapToAmount, err := router.GetAmountsOut(param.ToChainId, toChainInfo.cli, common.HexToAddress(param.User), common.HexToAddress(swapConfig.SwapContract), outAmount, swapConfig.SwapPath)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id":        param.ToChainId,
				"bridge_to_token": param.Path.BridgeToToken,
				"swap_to_token":   param.Path.SwapToToken,
				"input_amount":    outAmount.String(),
				"swap_contract":   swapConfig.SwapContract,
				"error":           err,
			}).Error("get swap amounts out failed")
			return quote, fmt.Errorf("internal error")
		}
		quote.OutAmount = swapToAmount.String()
	}

	// build payload
	payload, err := d.buildSwapPayload(common.HexToAddress(param.Path.BridgeFromToken), inputAmount, big.NewInt(param.ToChainId), common.HexToAddress(param.Receiver), common.HexToAddress(param.Path.SwapFromToken), common.HexToAddress(param.Path.SwapToToken))
	if err != nil {
		log.WithFields(log.Fields{
			"chain_id":    param.FromChainId,
			"token":       param.Path.SwapFromToken,
			"user":        param.User,
			"inputAmount": inputAmount.String(),
			"error":       err,
		}).Error("build swap payload failed")
		return quote, fmt.Errorf("internal error")
	}
	quote.Payload = common.Bytes2Hex(payload)
	return quote, nil
}

func (d *Dao) getSupportedBridgeChains(fromChain int64, token string) (configs []dbModel.BridgeTokenInfo, err error) {
	collection := d.db.Collection(new(dbModel.BridgeTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	filter := bson.M{"chain_id": fromChain}
	if token != "" {
		filter["token"] = strings.ToLower(token)
	}

	cursor, err := collection.Find(ctx, filter)
	if err != nil {
		return configs, err
	}
	defer cursor.Close(ctx)
	err = cursor.All(ctx, &configs)
	return configs, err
}

func (d *Dao) justABridgePair(path apiModel.SwapPath) bool {
	return path.SwapFromToken == path.BridgeFromToken && path.BridgeToToken == path.SwapToToken
}

func (d *Dao) getTargetChainSwapTokens(targetChains []dbModel.BridgeTokenInfo, fromPath apiModel.SwapPath) map[int64]map[string]apiModel.ToSwapToken {
	toChains := make(map[int64]map[string]apiModel.ToSwapToken)
	for _, sc := range targetChains {
		if !sc.Enabled {
			continue
		}

		toTokenInfo, err := d.tokenRepo.RetriveTokenInfo(sc.ToChainId, sc.ToToken)
		if err != nil {
			log.WithFields(log.Fields{
				"chain_id": sc.ToChainId,
				"token":    sc.ToToken,
			}).Error("get token info failed")
			continue
		}
		path := apiModel.SwapPath{
			SwapFromToken:   fromPath.SwapFromToken,
			BridgeFromToken: fromPath.BridgeFromToken,
			BridgeToToken:   sc.ToToken,
			SwapToToken:     sc.ToToken,
		}

		if !d.justABridgePair(path) {
			tokens, exist := toChains[sc.ToChainId]
			if !exist {
				tokens = make(map[string]apiModel.ToSwapToken)
				toChains[sc.ToChainId] = tokens
			}
			tokens[sc.ToToken] = apiModel.ToSwapToken{
				ToChainId:     sc.ToChainId,
				ToToken:       sc.ToToken,
				ToTokenSymbol: toTokenInfo.Symbol,
				Path:          path,
			}
		}

		// get all supported swap tokens for toChainId
		targetChainSwapTokens, err := d.getSupportedSwapTokens(sc.ToChainId)
		if err != nil {
			continue
		}
		for _, targetSwap := range targetChainSwapTokens {
			if common.HexToAddress(targetSwap.Token) == common.HexToAddress(sc.ToToken) {
				tokenInfo, err := d.tokenRepo.RetriveTokenInfo(targetSwap.ChainId, targetSwap.ToToken)
				if err != nil {
					continue
				}
				path := apiModel.SwapPath{
					SwapFromToken:   fromPath.SwapFromToken,
					BridgeFromToken: fromPath.BridgeFromToken,
					BridgeToToken:   sc.ToToken,
					SwapToToken:     targetSwap.ToToken,
				}

				if !d.justABridgePair(path) {
					tokens, exist := toChains[sc.ToChainId]
					if !exist {
						tokens = make(map[string]apiModel.ToSwapToken)
						toChains[sc.ToChainId] = tokens
					}
					// add targetSwap.ToToken to toChains
					tokens[targetSwap.ToToken] = apiModel.ToSwapToken{
						ToChainId:     targetSwap.ChainId,
						ToToken:       targetSwap.ToToken,
						ToTokenSymbol: tokenInfo.Symbol,
						Path:          path,
					}
				}
			}
		}
		if tokens, exist := toChains[sc.ToChainId]; exist && len(tokens) > 0 {
			toChains[sc.ToChainId] = tokens
		}
	}
	return toChains
}

func (d *Dao) getSupportedSwapTokens(chainId int64) (tokens []dbModel.SwapTokenInfo, err error) {
	collection := d.db.Collection(new(dbModel.SwapTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	cursor, err := collection.Find(ctx, bson.M{"chain_id": chainId})
	if err != nil {
		return tokens, err
	}
	defer cursor.Close(ctx)
	err = cursor.All(ctx, &tokens)
	return tokens, err
}

func (d *Dao) buildSwapPayload(token common.Address, amount *big.Int, toChainID *big.Int, receiver common.Address, fromToken common.Address, toToken common.Address) ([]byte, error) {
	contractAbi, _ := bridge.BridgeContractMetaData.GetAbi()
	return contractAbi.Pack("outTransferSwap", token, amount, toChainID, receiver, fromToken, toToken)
}

func (d *Dao) findSwapConfig(chainId int64, from string, to string) (*dbModel.SwapTokenInfo, error) {
	collection := d.db.Collection(new(dbModel.SwapTokenInfo).TableName())
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	var swapConfig dbModel.SwapTokenInfo
	err := collection.FindOne(ctx, bson.M{"chain_id": chainId, "token": strings.ToLower(from), "to_token": strings.ToLower(to)}).Decode(&swapConfig)
	if err == mongo.ErrNoDocuments {
		return nil, ErrRecordNotFound
	}
	return &swapConfig, err
}
