const fs = require("fs");
const path = require("path");
const hre = require("hardhat");

const DEPLOY_FILE = path.join(__dirname, "deploy.json");

async function mintToken(tokenA, tokenB, user) {
    var amount =  hre.ethers.parseEther("1000000")
    var tx = await tokenA.mint(user, amount);
    await tx.wait();
    console.log("Minted TTA to user:", user, " amount:", hre.ethers.formatEther(amount));
    tx = await tokenB.mint(user, amount);
    await tx.wait();
    console.log("Minted TTB to user:", user, " amount:", hre.ethers.formatEther(amount));
}

async function treasuryApprove(tokenA, tokenB, bridgeAddr, treasury) {
    var tx = await tokenA.connect(treasury).approve(bridgeAddr, hre.ethers.MaxUint256);
    await tx.wait();
    console.log("Approved TTA for bridge:", bridgeAddr);
    tx = await tokenB.connect(treasury).approve(bridgeAddr, hre.ethers.MaxUint256);
    await tx.wait();
    console.log("Approved TTB for bridge:", bridgeAddr);
}


async function getTargetChain(chainId) {
    if (!fs.existsSync(DEPLOY_FILE)) {
        throw new Error("deploy.json file not found");
    }

    const deployData = JSON.parse(fs.readFileSync(DEPLOY_FILE, "utf8"));
    if (!deployData[chainId]) {
        throw new Error(`No deployment data found for chainId ${chainId}`);
    }

    // iterator deployData to find the key not equal to chainId, chainId is number, and the key is string,
    // we need convert targetChainId from string to number before return.
    for (const key in deployData) {
        if (key != chainId.toString()) {
            return Number(key);
        }
    }
    throw new Error(`No target chain found for chainId ${chainId}`);
}

async function getDeploy(chainId) {
    if (!fs.existsSync(DEPLOY_FILE)) {
        throw new Error("deploy.json file not found");
    }

    const deployData = JSON.parse(fs.readFileSync(DEPLOY_FILE, "utf8"));
    if (!deployData[chainId]) {
        throw new Error(`No deployment data found for chainId ${chainId}`);
    }

    const addrs = deployData[chainId];
    const bridge = await hre.ethers.getContractAt("Bridge", addrs.bridge);

    return {bridge}
}

async function setOutConfig(curTokenMap, targetTokenMap, targetChainId, bridge) {
    const outConfigA = {
        receiveToken: await targetTokenMap.tokenA.getAddress(),
        fee: hre.ethers.parseEther("0.05"),
        limit: hre.ethers.parseEther("1000"),
        isBurn: false,
        enabled: true,
    };
    const outConfigB = {
        receiveToken: await targetTokenMap.tokenB.getAddress(),
        fee: hre.ethers.parseEther("0.04"),
        limit: hre.ethers.parseEther("1000"),
        isBurn: false,
        enabled: true,
    }

    var tx = await bridge.changeOutConfig(
        await curTokenMap.tokenA.getAddress(),
        targetChainId, // Target chain ID
        outConfigA
    );
    await tx.wait();
    console.log("Set out config for tokenA to chain", targetChainId, "tx", tx.hash);

    tx = await bridge.changeOutConfig(
        await curTokenMap.tokenB.getAddress(),
        targetChainId, // Target chain ID
        outConfigB
    );
    await tx.wait();
    console.log("Set out config for tokenB to chain", targetChainId, "tx", tx.hash);
}

// Define the script
async function main() {
    const chainId = await hre.ethers.provider.getNetwork().then(network => network.chainId);

    const curChainDeploy = await getDeploy(chainId);

    await setConfig(curChainDeploy.bridge);
    console.log("Configuration set successfully for chain", chainId);
}

// Run the script
main().catch((error) => {
    console.error("Error:", error);
});