const fs = require("fs");
const path = require("path");
const hre = require("hardhat");

let INITIAL_VALIDATORS = process.env.BRIDGE_VALIDATORS
let TREASURY = process.env.TREASURY

const DEPLOY_FILE = path.join(__dirname, "deploy.json");

async function addValidators(contract, validators) {
    let validatorList = validators.split(',')
    for (let i = 0; i < validatorList.length; i++) {
        var tx = await contract.addValidator(validatorList[i])
        await tx.wait();
        console.log("Added validator:", validatorList[i]);
    }
}

async function setTreasury(contract, treasury) {
    var tx = await contract.setTreasury(treasury);
    await tx.wait();
    console.log("Set treasury to:", treasury);
}

async function mintToken(tokenA, tokenB, user) {
    var amount =  hre.ethers.parseEther("1000000")
    var tx = await tokenA.mint(user, amount);
    await tx.wait();
    console.log("Minted TTA to user:", user, " amount:", hre.ethers.formatEther(amount));
    tx = await tokenB.mint(user, amount);
    await tx.wait();
    console.log("Minted TTB to user:", user, " amount:", hre.ethers.formatEther(amount));
}

async function treasuryApprove(tokenA, tokenB, bridgeAddr, treasury) {
    var tx = await tokenA.connect(treasury).approve(bridgeAddr, hre.ethers.MaxUint256);
    await tx.wait();
    console.log("Approved TTA for bridge:", bridgeAddr);
    tx = await tokenB.connect(treasury).approve(bridgeAddr, hre.ethers.MaxUint256);
    await tx.wait();
    console.log("Approved TTB for bridge:", bridgeAddr);
}

async function setTokenTreasuryConfig(tokenA, tokenB, bridge) {
    var tx = await bridge.setTreasuryConfig(
        await tokenA.getAddress(),
        hre.ethers.parseEther('0.0001'),
        500, // 5%
    )
    await tx.wait();
    console.log("Set treasury config for TTA");

    tx = await bridge.setTreasuryConfig(
        await tokenB.getAddress(),
        hre.ethers.parseEther('0.0001'),
        500, // 5%
    )
    await tx.wait();
    console.log("Set treasury config for TTB");
}

async function setConfig(tokenA, tokenB, bridge) {
    if (!INITIAL_VALIDATORS || !TREASURY) {
        throw new Error("Environment variables BRIDGE_VALIDATORS or TREASURY are not set.");
    }

    [owner, treasury] = await hre.ethers.getSigners();

    await addValidators(bridge, INITIAL_VALIDATORS);
    await setTreasury(bridge, TREASURY);
    await setTokenTreasuryConfig(tokenA, tokenB, bridge);

    // if treasury is exist, do approve .
    var bridgeAddr = await bridge.getAddress();
    if (treasury.address) {
        // test environment, mint token and approve.
        await mintToken(tokenA, tokenB, TREASURY);
        await treasuryApprove(tokenA, tokenB, bridgeAddr, treasury);
    }
}


async function getDeploy(chainId) {
    if (!fs.existsSync(DEPLOY_FILE)) {
        throw new Error("deploy.json file not found");
    }

    const deployData = JSON.parse(fs.readFileSync(DEPLOY_FILE, "utf8"));
    if (!deployData[chainId]) {
        throw new Error(`No deployment data found for chainId ${chainId}`);
    }

    const addrs = deployData[chainId];
    const tokenA = await hre.ethers.getContractAt("TestToken", addrs.tokenA);
    const tokenB = await hre.ethers.getContractAt("TestToken", addrs.tokenB);
    const bridge = await hre.ethers.getContractAt("Bridge", addrs.bridge);

    return {tokenA, tokenB, bridge}
}

async function setOutConfig(curTokenMap, targetTokenMap, targetChainId, bridge) {
    const outConfigA = {
        receiveToken: await targetTokenMap.tokenA.getAddress(),
        fee: hre.ethers.parseEther("0.05"),
        limit: hre.ethers.parseEther("1000"),
        isBurn: false,
        enabled: true,
    };
    const outConfigB = {
        receiveToken: await targetTokenMap.tokenB.getAddress(),
        fee: hre.ethers.parseEther("0.04"),
        limit: hre.ethers.parseEther("1000"),
        isBurn: false,
        enabled: true,
    }

    var tx = await bridge.changeOutConfig(
        await curTokenMap.tokenA.getAddress(),
        targetChainId, // Target chain ID
        outConfigA
    );
    await tx.wait();
    console.log("Set out config for tokenA to chain", targetChainId, "tx", tx.hash);

    tx = await bridge.changeOutConfig(
        await curTokenMap.tokenB.getAddress(),
        targetChainId, // Target chain ID
        outConfigB
    );
    await tx.wait();
    console.log("Set out config for tokenB to chain", targetChainId, "tx", tx.hash);
}

// Define the script
async function main() {
    const chainId = await hre.ethers.provider.getNetwork().then(network => network.chainId);
    const targetChainId = chainId === 269n ? 10323 : 269;
    console.log("Current chain ID:", chainId, "Target chain ID:", targetChainId);

    const curChainDeploy = await getDeploy(chainId);
    const targetChainDeploy = await getDeploy(targetChainId);

    await setConfig(curChainDeploy.tokenA, curChainDeploy.tokenB, curChainDeploy.bridge);
    await setOutConfig(curChainDeploy, targetChainDeploy, targetChainId, curChainDeploy.bridge);
    console.log("Configuration set successfully for chain", chainId);
}

// Run the script
main().catch((error) => {
    console.error("Error:", error);
});