const fs = require("fs");
const hre = require("hardhat");

const DEPLOY_FILE = "deploy.json";
const DEPLOY_TOKENS = "tokenpairs.json"

async function getTokenPairs(chainId) {
    if (!fs.existsSync(DEPLOY_TOKENS)) {
        throw new Error("tokenpairs.json file not found");
    }

    const tokenData = JSON.parse(fs.readFileSync(DEPLOY_TOKENS, "utf8"));
    if (!tokenData[chainId]) {
        throw new Error(`No token pair data found for chainId ${chainId}`);
    }

    const tokenAddrs = tokenData[chainId];
    return tokenAddrs
}

async function getDeploy(chainId) {
    if (!fs.existsSync(DEPLOY_FILE)) {
        throw new Error("deploy.json file not found");
    }

    const deployData = JSON.parse(fs.readFileSync(DEPLOY_FILE, "utf8"));
    if (!deployData[chainId]) {
        throw new Error(`No deployment data found for chainId ${chainId}`);
    }

    const addrs = deployData[chainId];
    const bridge = await hre.ethers.getContractAt("Bridge", addrs.bridge);

    return {bridge}
}

async function setOutConfig(bridge, tokenPair) {
    const tokenContract = await hre.ethers.getContractAt("TestToken", tokenPair.token);
    var decimals = await tokenContract.decimals();

    const outConfig = {
        receiveToken: tokenPair.targetToken,
        fee: hre.ethers.parseUnits(tokenPair.fee, decimals),
        limit: hre.ethers.parseUnits(tokenPair.limit, decimals),
        isBurn: tokenPair.isBurn,
        enabled: true,
    };
    console.log("set out config for token ", tokenPair.token, "-", outConfig);

    var tx = await bridge.changeOutConfig(
        tokenPair.token,
        tokenPair.targetChain,
        outConfig
    );
    await tx.wait();
    console.log("set out config for token ", tokenPair.token, " to chain", tokenPair.targetChain, "tx", tx.hash);
}

// Define the script
async function main() {
    const chainId = await hre.ethers.provider.getNetwork().then(network => network.chainId);

    const curChainDeploy = await getDeploy(chainId);
    const tokenPairs = await getTokenPairs(chainId);
    for (var i = 0; i < tokenPairs.length; i++) {
        await setOutConfig(curChainDeploy.bridge, tokenPairs[i]);
    }
    console.log("all token pairs processed.");
}

// Run the script
main().catch((error) => {
    console.error("Error:", error);
});