const fs = require("fs");
const path = require("path");
const { ethers, upgrades } = require("hardhat");

const DEPLOY_FILE = "deploy.json";

let INITIAL_VALIDATORS = process.env.BRIDGE_VALIDATORS
let TREASURY = process.env.TREASURY
let FEE_RECEIVER = process.env.FEE_RECEIVER
let VALIDATOR_THRESHOLD = process.env.VALIDATOR_THRESHOLD

async function deploy() {
    // Deploy the contract
    const factory =  await ethers.getContractFactory(
        "Bridge"
    );
    [owner] = await ethers.getSigners();

    const contract = await upgrades.deployProxy(factory, [owner.address], {
        initializer: "initialize",
    });
    await contract.waitForDeployment();
    const bridgeAddr = await contract.getAddress();

    const chainId = await ethers.provider.getNetwork().then(network => network.chainId);
    const initialHeight = contract.deploymentTransaction().blockNumber;
    await saveDeployment(chainId, bridgeAddr, initialHeight);
    await setConfig(contract);

    console.log("deploy bridge contract ", bridgeAddr, " at block ", initialHeight);
}


async function addValidators(contract, validators) {
    let validatorList = validators.split(',')
    for (let i = 0; i < validatorList.length; i++) {
        var tx = await contract.addValidator(validatorList[i])
        await tx.wait();
        console.log("added validator:", validatorList[i]);
    }
}

async function setValidatorRequired(contract, count) {
    var tx = await contract.changeValidRequired(count);
    await tx.wait();
    console.log("set validate threshold to:", count);
}

async function setTreasury(contract, treasury) {
    var tx = await contract.setTreasury(treasury);
    await tx.wait();
    console.log("set treasury to:", treasury);
}

async function setFeeReceiver(contract, receiver) {
    var tx = await contract.setFeeReceiver(receiver);
    await tx.wait();
    console.log("set fee receiver to:", receiver);
}

async function setConfig(bridge) {
    if (!INITIAL_VALIDATORS || !TREASURY) {
        throw new Error("Environment variables BRIDGE_VALIDATORS or TREASURY are not set.");
    }

    await addValidators(bridge, INITIAL_VALIDATORS);
    await setValidatorRequired(bridge, VALIDATOR_THRESHOLD);
    await setFeeReceiver(bridge, FEE_RECEIVER);
    // if TREASURY not empty, set treasury.
    if (TREASURY) {
        await setTreasury(bridge, TREASURY);
    }
}


function saveDeployment(chainId, bridge, initialHeight) {
    let deployData = {};
    if (fs.existsSync(DEPLOY_FILE)) {
        deployData = JSON.parse(fs.readFileSync(DEPLOY_FILE, "utf8"));
    }

    deployData[chainId] = {
        bridge,
        initialHeight,
    };

    fs.writeFileSync(DEPLOY_FILE, JSON.stringify(deployData, null, 2));
    console.log("Deployment at chain ", chainId, " saved to deploy.json");
}

// Define the script
async function main() {
    await deploy();
}

// Run the script
main().catch((error) => {
    console.error("Error:", error);
});