package util

import (
	"crypto/ecdsa"
	"encoding/binary"
	"fmt"
	"strconv"
	"strings"

	"github.com/btcsuite/btcutil/hdkeychain"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
)

func GetDepositAddress(input common.Address, xpub string) (output common.Address, err error) {
	paths := make([]string, 7)
	for i := 0; i < 7; i++ {
		uint32Temp := uint32(0)
		if i == 6 {
			uint32Temp = binary.BigEndian.Uint32(common.LeftPadBytes(input.Bytes()[i*3:i*3+2], 4))
		} else {
			uint32Temp = binary.BigEndian.Uint32(common.LeftPadBytes(input.Bytes()[i*3:i*3+3], 4))
		}
		paths[i] = fmt.Sprintf("%d", uint32Temp)
	}
	pubkey, err := GetPubKeyByPub(xpub, strings.Join(paths, "/"))
	if err != nil {
		return common.Address{}, err
	}
	return crypto.PubkeyToAddress(*pubkey), nil
}

func GetPubKeyByPub(xpub string, path string) (pubkey *ecdsa.PublicKey, err error) {
	pathList := parsePath(path)
	var next *hdkeychain.ExtendedKey
	for _, floor := range pathList {
		idx := floor[0]
		isHardened, _ := strconv.ParseBool(strconv.Itoa(floor[1]))
		next, err = nextFloor(xpub, isHardened, uint32(idx))
		if err != nil {
			return
		}
		xpub = next.String()
	}
	pk, err := next.ECPubKey()
	if err != nil {
		return
	}
	return pk.ToECDSA(), nil
}

// 返回一个二维数组 参数1 对应每一层偏移 参数2  1代表hardened 0普通
func parsePath(path string) [][]int {
	l := strings.Split(path, "/")
	var resList [][]int
	// m开头或者/开头 去掉第一个
	if l[0] == "m" || l[0] == "" {
		l = l[1:]
	}
	// /结尾 去掉最后一个
	if l[len(l)-1] == "" {
		l = l[:len(l)-1]
	}
	for _, s := range l {
		if strings.HasSuffix(s, "'") {
			idx, _ := strconv.Atoi(s[:len(s)-1])
			resList = append(resList, []int{idx, 1})
		} else {
			idx, _ := strconv.Atoi(s)
			resList = append(resList, []int{idx, 0})
		}
	}
	return resList
}

func nextFloor(key string, hardened bool, idx uint32) (*hdkeychain.ExtendedKey, error) {
	key1, err := hdkeychain.NewKeyFromString(key)
	if err != nil {
		return nil, err
	}
	if hardened {
		return key1.Child(hdkeychain.HardenedKeyStart + idx)
	} else {
		return key1.Child(idx)
	}
}
