package encpk

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/sha256"
	"encoding/binary"
	"encoding/hex"
	"errors"
	"os"
	"time"
)

const (
	TimeWindowSeconds = 60 // 1分钟时间窗口
	InitialSeed       = "uEj7AgDNCREwsvnTaCEtzDXt0I5eFDl8"
)

// GenerateTimeBasedKey 根据时间窗口和种子生成AES密钥
func GenerateTimeBasedKey(seed string, timestamp time.Time) []byte {
	// 将时间戳转换为分钟级别的时间窗口
	timeWindow := timestamp.Unix() / TimeWindowSeconds

	// 创建哈希输入：种子 + 时间窗口
	hash := sha256.New()
	hash.Write([]byte(seed))

	timeBytes := make([]byte, 8)
	binary.BigEndian.PutUint64(timeBytes, uint64(timeWindow))
	hash.Write(timeBytes)

	// 返回32字节的AES-256密钥
	return hash.Sum(nil)
}

// EncryptPKWithTimeKey 使用时间密钥加密私钥
func EncryptPKWithTimeKey(pk []byte) ([]byte, error) {
	if len(pk) == 0 {
		return nil, errors.New("private key cannot be empty")
	}
	var seed = InitialSeed

	// 生成当前时间的密钥
	key := GenerateTimeBasedKey(seed, time.Now())

	// 创建AES密码块
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	// 创建GCM模式
	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	// 生成随机nonce
	nonce := make([]byte, gcm.NonceSize())
	// 这里应该使用crypto/rand生成随机nonce，为简化示例使用固定值

	// 加密数据
	ciphertext := gcm.Seal(nonce, nonce, pk, nil)

	return ciphertext, nil
}

// DecryptPKWithTimeKey 使用时间密钥解密私钥
func DecryptPKWithTimeKey(encryptedPK []byte) ([]byte, error) {
	if len(encryptedPK) == 0 {
		return nil, errors.New("encrypted private key cannot be empty")
	}
	var seed = InitialSeed
	now := time.Now()

	// 尝试当前时间窗口的密钥
	if plaintext, err := tryDecryptWithTime(encryptedPK, seed, now); err == nil {
		return plaintext, nil
	}

	// 尝试前一个时间窗口的密钥（处理边界情况）
	prevTime := now.Add(-time.Duration(TimeWindowSeconds) * time.Second)
	if plaintext, err := tryDecryptWithTime(encryptedPK, seed, prevTime); err == nil {
		return plaintext, nil
	}

	return nil, errors.New("decryption failed: data may be expired or corrupted")
}

// tryDecryptWithTime 尝试使用特定时间的密钥解密
func tryDecryptWithTime(encryptedPK []byte, seed string, timestamp time.Time) ([]byte, error) {
	key := GenerateTimeBasedKey(seed, timestamp)

	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	nonceSize := gcm.NonceSize()
	if len(encryptedPK) < nonceSize {
		return nil, errors.New("encrypted data too short")
	}

	nonce := encryptedPK[:nonceSize]
	ciphertext := encryptedPK[nonceSize:]

	return gcm.Open(nil, nonce, ciphertext, nil)
}

func DecryptOTPWithAESKey(key []byte, otp []byte) ([]byte, error) {
	if len(key) == 0 {
		return nil, errors.New("key cannot be empty")
	}

	encryptedPK, err := DecryptPKWithTimeKey(otp)
	if err != nil {
		return nil, err
	}
	fpk, _ := hex.DecodeString(string(encryptedPK))

	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	nonceSize := gcm.NonceSize()
	if len(encryptedPK) < nonceSize {
		return nil, errors.New("encrypted data too short")
	}

	nonce := fpk[:nonceSize]
	ciphertext := fpk[nonceSize:]

	return gcm.Open(nil, nonce, ciphertext, nil)
}

func DecryptOTP(aesPath string, otpPath string) ([]byte, error) {
	aeskey, err := os.ReadFile(aesPath)
	if err != nil {
		return nil, errors.New("Failed to read AES key file:" + err.Error())
	}
	aeskeyData, _ := hex.DecodeString(string(aeskey))

	otpkey, err := os.ReadFile(otpPath)
	if err != nil {
		return nil, errors.New("Failed to read OTP file:" + err.Error())
	}
	return DecryptOTPWithAESKey(aeskeyData, otpkey)
}
