package main

import (
	"code.wuban.net.cn/movabridge/token-bridge/chain"
	"code.wuban.net.cn/movabridge/token-bridge/config"
	"code.wuban.net.cn/movabridge/token-bridge/dao"
	"os"
	"os/signal"
	"syscall"

	log "github.com/sirupsen/logrus"
	"github.com/spf13/cobra"
)

var (
	confPath string
)

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})
}

var rootCmd = &cobra.Command{
	Use:   "validator",
	Short: "Token bridge validator service",
	Long:  "A validator service for the token bridge system that syncs with multiple blockchain networks",
	Run: func(cmd *cobra.Command, args []string) {
		conf, err := config.New(confPath)
		if err != nil {
			panic(err)
		}
		if len(conf.OTPKeyPath) == 0 || len(conf.AesKeyPath) == 0 {
			log.Warning("no OTP key or AES key, please make sure you have set them correctly")
			log.Warning("the server will runing in sync mode only")
		}

		d, err := dao.New(conf)
		if err != nil {
			panic(err)
		}
		d.Start()

		if conf.Debug {
			log.SetLevel(log.DebugLevel)
		}

		syncers := make([]*chain.ChainSync, 0)

		for _, chainConfig := range conf.Chains {
			syncer := chain.NewChainSync(chainConfig, d)
			syncer.Start()
			syncers = append(syncers, syncer)
			d.AddSyncer(chainConfig.ChainId, syncer)
		}

		// Set up signal handling
		sigCh := make(chan os.Signal, 1)
		signal.Notify(sigCh, syscall.SIGINT, syscall.SIGTERM)

		// Wait for termination signal
		sig := <-sigCh
		log.WithField("signal", sig.String()).Info("received termination signal, shutting down")

		// Stop all chain sync instances
		for _, syncer := range syncers {
			syncer.Stop()
		}
		d.Stop()

		log.Info("graceful shutdown completed")
		os.Exit(0)
	},
}

func init() {
	rootCmd.Flags().StringVarP(&confPath, "config", "c", "config.toml", "config file path")
}

func main() {
	if err := rootCmd.Execute(); err != nil {
		log.Fatal(err)
	}
}
