package config

import (
	"flag"
	"github.com/BurntSushi/toml"
)

type Config struct {
	Debug  bool
	Chains map[string]*ChainConfig `toml:"chains"`
	MySQL  MySQLConfig
	Server ServerConfig
}

type ChainConfig struct {
	Name                string `toml:"name"`
	RPC                 string `toml:"rpc"`
	InitialHeight       int64  `toml:"initial_height"`
	BatchBlock          int    `toml:"batch_block"`
	BehindBlock         int    `toml:"behind_block"`
	BridgeContract      string `toml:"bridge_contract"`
	ValidatorPrivateKey string `toml:"validator_private_key"`
	ChainId             int64  `toml:"chain_id"` // Will be populated by code
}

type MySQLConfig struct {
	Host        string
	Port        int
	User        string
	Password    string
	Database    string
	MaxConn     int `toml:"max_conn"`
	MaxIdleConn int `toml:"max_idle_conn"`
}

type ServerConfig struct {
	Listen         string
	InvalidHeaders []string `toml:"invalid_headers"`
}

var confPath = flag.String("c", "config.toml", "config file path")

func New() (*Config, error) {
	var cfg Config
	cfg.Chains = make(map[string]*ChainConfig)

	// Parse the TOML configuration
	_, err := toml.DecodeFile(*confPath, &cfg)
	if err != nil {
		return nil, err
	}

	// Process the chains from array to map using name as key
	var chainArray []ChainConfig
	_, err = toml.DecodeFile(*confPath, &struct {
		Chains *[]ChainConfig `toml:"chains"`
		*Config
	}{
		Chains: &chainArray,
		Config: &cfg,
	})
	if err != nil {
		return nil, err
	}

	// Convert to map for easier access
	for _, chain := range chainArray {
		chainCopy := chain // Create a copy to avoid pointer issues
		cfg.Chains[chain.Name] = &chainCopy
	}

	return &cfg, nil
}
